/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.view;

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Rect;
import android.graphics.RectF;
import android.util.Size;
import android.util.SizeF;
import android.view.Display;
import android.view.TextureView;
import android.view.View;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.experimental.UseExperimental;
import androidx.camera.core.ExperimentalUseCaseGroup;
import androidx.camera.core.Logger;
import androidx.camera.core.SurfaceRequest;
import androidx.camera.view.PreviewView;
import androidx.core.util.Preconditions;

final class PreviewTransformation {
    private static final String TAG = "PreviewTransform";
    private static final PreviewView.ScaleType DEFAULT_SCALE_TYPE = PreviewView.ScaleType.FILL_CENTER;
    private static final int FLOAT_NUMBER_PER_VERTEX = 2;
    private Size mResolution;
    private Rect mSurfaceCropRect;
    private int mPreviewRotationDegrees;
    private int mTargetRotation;
    private boolean mIsFrontCamera;
    private PreviewView.ScaleType mScaleType = DEFAULT_SCALE_TYPE;

    PreviewTransformation() {
    }

    @UseExperimental(markerClass=ExperimentalUseCaseGroup.class)
    void setTransformationInfo(@NonNull SurfaceRequest.TransformationInfo transformationInfo, Size resolution, boolean isFrontCamera) {
        Logger.d((String)TAG, (String)("Transformation info set: " + transformationInfo + " " + resolution + " " + isFrontCamera));
        this.mSurfaceCropRect = transformationInfo.getCropRect();
        this.mPreviewRotationDegrees = transformationInfo.getRotationDegrees();
        this.mTargetRotation = transformationInfo.getTargetRotation();
        this.mResolution = resolution;
        this.mIsFrontCamera = isFrontCamera;
    }

    @VisibleForTesting
    Matrix getTextureViewCorrectionMatrix() {
        Preconditions.checkState((boolean)this.isTransformationInfoReady());
        Matrix matrix = new Matrix();
        float[] surfaceVertices = PreviewTransformation.sizeToVertices(this.mResolution);
        float[] rotatedSurfaceVertices = PreviewTransformation.createRotatedVertices(surfaceVertices, -PreviewTransformation.rotationValueToRotationDegrees(this.mTargetRotation));
        matrix.setPolyToPoly(surfaceVertices, 0, rotatedSurfaceVertices, 0, 4);
        return matrix;
    }

    void transformView(Size previewViewSize, int layoutDirection, @NonNull View preview) {
        if (!this.isTransformationInfoReady()) {
            return;
        }
        if (preview instanceof TextureView) {
            ((TextureView)preview).setTransform(this.getTextureViewCorrectionMatrix());
        } else {
            Display display = preview.getDisplay();
            if (display != null && display.getRotation() != this.mTargetRotation) {
                Logger.e((String)TAG, (String)"Non-display rotation not supported with SurfaceView / PERFORMANCE mode.");
            }
        }
        RectF surfaceRectInPreviewView = this.getTransformedSurfaceRect(previewViewSize, layoutDirection);
        preview.setPivotX(0.0f);
        preview.setPivotY(0.0f);
        preview.setScaleX(surfaceRectInPreviewView.width() / (float)this.mResolution.getWidth());
        preview.setScaleY(surfaceRectInPreviewView.height() / (float)this.mResolution.getHeight());
        preview.setTranslationX(surfaceRectInPreviewView.left - (float)preview.getLeft());
        preview.setTranslationY(surfaceRectInPreviewView.top - (float)preview.getTop());
    }

    void setScaleType(PreviewView.ScaleType scaleType) {
        this.mScaleType = scaleType;
    }

    PreviewView.ScaleType getScaleType() {
        return this.mScaleType;
    }

    private RectF getTransformedSurfaceRect(Size previewViewSize, int layoutDirection) {
        Preconditions.checkState((boolean)this.isTransformationInfoReady());
        Matrix surfaceToPreviewView = this.getSurfaceToPreviewViewMatrix(previewViewSize, layoutDirection);
        float[] surfaceVertices = PreviewTransformation.sizeToVertices(this.mResolution);
        surfaceToPreviewView.mapPoints(surfaceVertices);
        return PreviewTransformation.verticesToRect(surfaceVertices);
    }

    private Matrix getSurfaceToPreviewViewMatrix(Size previewViewSize, int layoutDirection) {
        float[] previewViewCropRectVertices;
        Preconditions.checkState((boolean)this.isTransformationInfoReady());
        Matrix matrix = new Matrix();
        if (this.isCropRectAspectRatioMatchPreviewView(previewViewSize)) {
            previewViewCropRectVertices = PreviewTransformation.sizeToVertices(previewViewSize);
        } else {
            RectF previewViewCropRect = this.getPreviewViewCropRectForMismatchedAspectRatios(previewViewSize, layoutDirection);
            previewViewCropRectVertices = PreviewTransformation.rectToVertices(previewViewCropRect);
        }
        float[] rotatedPreviewViewCropRectVertices = PreviewTransformation.createRotatedVertices(previewViewCropRectVertices, this.mPreviewRotationDegrees);
        float[] surfaceCropRectVertices = PreviewTransformation.rectToVertices(new RectF(this.mSurfaceCropRect));
        matrix.setPolyToPoly(surfaceCropRectVertices, 0, rotatedPreviewViewCropRectVertices, 0, 4);
        if (this.mIsFrontCamera) {
            if (PreviewTransformation.is90or270(this.mPreviewRotationDegrees)) {
                matrix.preScale(1.0f, -1.0f, (float)this.mSurfaceCropRect.centerX(), (float)this.mSurfaceCropRect.centerY());
            } else {
                matrix.preScale(-1.0f, 1.0f, (float)this.mSurfaceCropRect.centerX(), (float)this.mSurfaceCropRect.centerY());
            }
        }
        return matrix;
    }

    RectF getPreviewViewCropRectForMismatchedAspectRatios(Size previewViewSize, int layoutDirection) {
        RectF previewViewRect = new RectF(0.0f, 0.0f, (float)previewViewSize.getWidth(), (float)previewViewSize.getHeight());
        SizeF rotatedCropRectSize = this.getRotatedCropRectSize();
        RectF rotatedSurfaceCropRect = new RectF(0.0f, 0.0f, rotatedCropRectSize.getWidth(), rotatedCropRectSize.getHeight());
        Matrix matrix = new Matrix();
        PreviewTransformation.setMatrixRectToRect(matrix, rotatedSurfaceCropRect, previewViewRect, this.mScaleType);
        matrix.mapRect(rotatedSurfaceCropRect);
        if (layoutDirection == 1) {
            return PreviewTransformation.flipHorizontally(rotatedSurfaceCropRect, (float)previewViewSize.getWidth() / 2.0f);
        }
        return rotatedSurfaceCropRect;
    }

    private static void setMatrixRectToRect(Matrix matrix, RectF source, RectF destination, PreviewView.ScaleType scaleType) {
        boolean isFitTypes;
        Matrix.ScaleToFit matrixScaleType;
        switch (scaleType) {
            case FIT_CENTER: 
            case FILL_CENTER: {
                matrixScaleType = Matrix.ScaleToFit.CENTER;
                break;
            }
            case FIT_END: 
            case FILL_END: {
                matrixScaleType = Matrix.ScaleToFit.END;
                break;
            }
            case FIT_START: 
            case FILL_START: {
                matrixScaleType = Matrix.ScaleToFit.START;
                break;
            }
            default: {
                Logger.e((String)TAG, (String)("Unexpected crop rect: " + (Object)((Object)scaleType)));
                matrixScaleType = Matrix.ScaleToFit.FILL;
            }
        }
        boolean bl = isFitTypes = scaleType == PreviewView.ScaleType.FIT_CENTER || scaleType == PreviewView.ScaleType.FIT_START || scaleType == PreviewView.ScaleType.FIT_END;
        if (isFitTypes) {
            matrix.setRectToRect(source, destination, matrixScaleType);
        } else {
            matrix.setRectToRect(destination, source, matrixScaleType);
            matrix.invert(matrix);
        }
    }

    private static RectF flipHorizontally(RectF original, float flipLineX) {
        return new RectF(flipLineX + flipLineX - original.right, original.top, flipLineX + flipLineX - original.left, original.bottom);
    }

    private SizeF getRotatedCropRectSize() {
        Preconditions.checkNotNull((Object)this.mSurfaceCropRect);
        if (PreviewTransformation.is90or270(this.mPreviewRotationDegrees)) {
            return new SizeF((float)this.mSurfaceCropRect.height(), (float)this.mSurfaceCropRect.width());
        }
        return new SizeF((float)this.mSurfaceCropRect.width(), (float)this.mSurfaceCropRect.height());
    }

    @VisibleForTesting
    boolean isCropRectAspectRatioMatchPreviewView(Size previewViewSize) {
        float previewViewRatio = (float)previewViewSize.getWidth() / (float)previewViewSize.getHeight();
        SizeF rotatedSize = this.getRotatedCropRectSize();
        float upperBound = (rotatedSize.getWidth() + 0.5f) / (rotatedSize.getHeight() - 0.5f);
        float lowerBound = (rotatedSize.getWidth() - 0.5f) / (rotatedSize.getHeight() + 0.5f);
        return previewViewRatio >= lowerBound && previewViewRatio <= upperBound;
    }

    Bitmap createTransformedBitmap(@NonNull Bitmap original, Size previewViewSize, int layoutDirection) {
        if (!this.isTransformationInfoReady()) {
            return original;
        }
        Matrix textureViewCorrection = this.getTextureViewCorrectionMatrix();
        RectF surfaceRectInPreviewView = this.getTransformedSurfaceRect(previewViewSize, layoutDirection);
        Bitmap transformed = Bitmap.createBitmap((int)previewViewSize.getWidth(), (int)previewViewSize.getHeight(), (Bitmap.Config)original.getConfig());
        Canvas canvas = new Canvas(transformed);
        Matrix canvasTransform = new Matrix();
        canvasTransform.postConcat(textureViewCorrection);
        canvasTransform.postScale(surfaceRectInPreviewView.width() / (float)this.mResolution.getWidth(), surfaceRectInPreviewView.height() / (float)this.mResolution.getHeight());
        canvasTransform.postTranslate(surfaceRectInPreviewView.left, surfaceRectInPreviewView.top);
        canvas.drawBitmap(original, canvasTransform, new Paint(7));
        return transformed;
    }

    @Nullable
    Matrix getPreviewViewToNormalizedSurfaceMatrix(Size previewViewSize, int layoutDirection) {
        if (!this.isTransformationInfoReady()) {
            return null;
        }
        Matrix matrix = new Matrix();
        this.getSurfaceToPreviewViewMatrix(previewViewSize, layoutDirection).invert(matrix);
        Matrix normalization = new Matrix();
        normalization.setRectToRect(new RectF(0.0f, 0.0f, (float)this.mResolution.getWidth(), (float)this.mResolution.getHeight()), new RectF(0.0f, 0.0f, 1.0f, 1.0f), Matrix.ScaleToFit.FILL);
        matrix.postConcat(normalization);
        return matrix;
    }

    static int rotationValueToRotationDegrees(int rotationValue) {
        switch (rotationValue) {
            case 0: {
                return 0;
            }
            case 1: {
                return 90;
            }
            case 2: {
                return 180;
            }
            case 3: {
                return 270;
            }
        }
        throw new IllegalStateException("Unexpected rotation value " + rotationValue);
    }

    private static boolean is90or270(int rotationDegrees) {
        if (rotationDegrees == 90 || rotationDegrees == 270) {
            return true;
        }
        if (rotationDegrees == 0 || rotationDegrees == 180) {
            return false;
        }
        throw new IllegalArgumentException("Invalid rotation degrees: " + rotationDegrees);
    }

    @VisibleForTesting
    static float[] sizeToVertices(Size size) {
        return new float[]{0.0f, 0.0f, size.getWidth(), 0.0f, size.getWidth(), size.getHeight(), 0.0f, size.getHeight()};
    }

    private static float[] rectToVertices(RectF rectF) {
        return new float[]{rectF.left, rectF.top, rectF.right, rectF.top, rectF.right, rectF.bottom, rectF.left, rectF.bottom};
    }

    private static RectF verticesToRect(float[] vertices) {
        return new RectF(PreviewTransformation.min(vertices[0], vertices[2], vertices[4], vertices[6]), PreviewTransformation.min(vertices[1], vertices[3], vertices[5], vertices[7]), PreviewTransformation.max(vertices[0], vertices[2], vertices[4], vertices[6]), PreviewTransformation.max(vertices[1], vertices[3], vertices[5], vertices[7]));
    }

    private static float max(float value1, float value2, float value3, float value4) {
        return Math.max(Math.max(value1, value2), Math.max(value3, value4));
    }

    private static float min(float value1, float value2, float value3, float value4) {
        return Math.min(Math.min(value1, value2), Math.min(value3, value4));
    }

    private boolean isTransformationInfoReady() {
        return this.mSurfaceCropRect != null && this.mResolution != null;
    }

    private static float[] createRotatedVertices(float[] original, int rotationDegrees) {
        float[] rotated = new float[original.length];
        int offset = -rotationDegrees / 90 * 2;
        for (int originalIndex = 0; originalIndex < original.length; ++originalIndex) {
            int rotatedIndex = (originalIndex + offset) % original.length;
            rotatedIndex = rotatedIndex < 0 ? rotatedIndex + original.length : rotatedIndex;
            rotated[rotatedIndex] = original[originalIndex];
        }
        return rotated;
    }
}

