/*
 * Copyright 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
@file:JvmName("ViewModelKt")
@file:JvmMultifileClass

package androidx.lifecycle.viewmodel.compose

import androidx.compose.runtime.Composable
import androidx.lifecycle.HasDefaultViewModelProviderFactory
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.ViewModelStoreOwner
import androidx.lifecycle.viewmodel.CreationExtras
import androidx.lifecycle.viewmodel.initializer
import androidx.lifecycle.viewmodel.viewModelFactory
import kotlin.reflect.KClass

/**
 * Returns an existing [ViewModel] or creates a new one in the given owner (usually, a fragment or
 * an activity), defaulting to the owner provided by [LocalViewModelStoreOwner].
 *
 * The created [ViewModel] is associated with the given [viewModelStoreOwner] and will be retained
 * as long as the owner is alive (e.g. if it is an activity, until it is finished or process is
 * killed).
 *
 * If default arguments are provided via the [CreationExtras], they will be available to the
 * appropriate factory when the [ViewModel] is created.
 *
 * @param viewModelStoreOwner The owner of the [ViewModel] that controls the scope and lifetime of
 *   the returned [ViewModel]. Defaults to using [LocalViewModelStoreOwner].
 * @param key The key to use to identify the [ViewModel].
 * @param factory The [ViewModelProvider.Factory] that should be used to create the [ViewModel] or
 *   null if you would like to use the default factory from the [LocalViewModelStoreOwner]
 * @param extras The default extras used to create the [ViewModel].
 * @return A [ViewModel] that is an instance of the given [VM] type.
 * @sample androidx.lifecycle.viewmodel.compose.samples.CreationExtrasViewModel
 */
@Suppress("MissingJvmstatic")
@Composable
public inline fun <reified VM : ViewModel> viewModel(
    viewModelStoreOwner: ViewModelStoreOwner =
        checkNotNull(LocalViewModelStoreOwner.current) {
            "No ViewModelStoreOwner was provided via LocalViewModelStoreOwner"
        },
    key: String? = null,
    factory: ViewModelProvider.Factory? = null,
    extras: CreationExtras =
        if (viewModelStoreOwner is HasDefaultViewModelProviderFactory) {
            viewModelStoreOwner.defaultViewModelCreationExtras
        } else {
            CreationExtras.Empty
        }
): VM = viewModel(VM::class, viewModelStoreOwner, key, factory, extras)

/**
 * Returns an existing [ViewModel] or creates a new one in the scope (usually, a fragment or an
 * activity)
 *
 * The created [ViewModel] is associated with the given [viewModelStoreOwner] and will be retained
 * as long as the scope is alive (e.g. if it is an activity, until it is finished or process is
 * killed).
 *
 * If default arguments are provided via the [CreationExtras], they will be available to the
 * appropriate factory when the [ViewModel] is created.
 *
 * @param modelClass The class of the [ViewModel] to create an instance of it if it is not present.
 * @param viewModelStoreOwner The scope that the created [ViewModel] should be associated with.
 * @param key The key to use to identify the [ViewModel].
 * @param factory The [ViewModelProvider.Factory] that should be used to create the [ViewModel] or
 *   null if you would like to use the default factory from the [LocalViewModelStoreOwner]
 * @param extras The default extras used to create the [ViewModel].
 * @return A [ViewModel] that is an instance of the given [VM] type.
 * @sample androidx.lifecycle.viewmodel.compose.samples.CreationExtrasViewModel
 */
@Suppress("MissingJvmstatic")
@Composable
public fun <VM : ViewModel> viewModel(
    modelClass: KClass<VM>,
    viewModelStoreOwner: ViewModelStoreOwner =
        checkNotNull(LocalViewModelStoreOwner.current) {
            "No ViewModelStoreOwner was provided via LocalViewModelStoreOwner"
        },
    key: String? = null,
    factory: ViewModelProvider.Factory? = null,
    extras: CreationExtras =
        if (viewModelStoreOwner is HasDefaultViewModelProviderFactory) {
            viewModelStoreOwner.defaultViewModelCreationExtras
        } else {
            CreationExtras.Empty
        }
): VM = viewModelStoreOwner.get(modelClass, key, factory, extras)

/**
 * Returns an existing [ViewModel] or creates a new one in the scope (usually, a fragment or an
 * activity)
 *
 * The created [ViewModel] is associated with the given [viewModelStoreOwner] and will be retained
 * as long as the scope is alive (e.g. if it is an activity, until it is finished or process is
 * killed).
 *
 * If the [viewModelStoreOwner] implements [HasDefaultViewModelProviderFactory] its default
 * [CreationExtras] are the ones that will be provided to the receiver scope from the [initializer]
 *
 * @param viewModelStoreOwner The scope that the created [ViewModel] should be associated with.
 * @param key The key to use to identify the [ViewModel].
 * @param initializer lambda used to create an instance of the ViewModel class
 * @return A [ViewModel] that is an instance of the given [VM] type.
 * @sample androidx.lifecycle.viewmodel.compose.samples.CreationExtrasViewModelInitializer
 */
@Composable
public inline fun <reified VM : ViewModel> viewModel(
    viewModelStoreOwner: ViewModelStoreOwner =
        checkNotNull(LocalViewModelStoreOwner.current) {
            "No ViewModelStoreOwner was provided via LocalViewModelStoreOwner"
        },
    key: String? = null,
    noinline initializer: CreationExtras.() -> VM
): VM =
    viewModel(
        VM::class,
        viewModelStoreOwner,
        key,
        viewModelFactory { initializer(initializer) },
        if (viewModelStoreOwner is HasDefaultViewModelProviderFactory) {
            viewModelStoreOwner.defaultViewModelCreationExtras
        } else {
            CreationExtras.Empty
        }
    )

internal fun <VM : ViewModel> ViewModelStoreOwner.get(
    modelClass: KClass<VM>,
    key: String? = null,
    factory: ViewModelProvider.Factory? = null,
    extras: CreationExtras =
        if (this is HasDefaultViewModelProviderFactory) {
            this.defaultViewModelCreationExtras
        } else {
            CreationExtras.Empty
        }
): VM {
    val provider =
        if (factory != null) {
            ViewModelProvider.create(this.viewModelStore, factory, extras)
        } else if (this is HasDefaultViewModelProviderFactory) {
            ViewModelProvider.create(
                this.viewModelStore,
                this.defaultViewModelProviderFactory,
                extras
            )
        } else {
            ViewModelProvider.create(this)
        }
    return if (key != null) {
        provider[key, modelClass]
    } else {
        provider[modelClass]
    }
}
