/*
 * Decompiled with CFR 0.152.
 */
package com.airbnb.lottie.network;

import android.util.Pair;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;
import com.airbnb.lottie.LottieComposition;
import com.airbnb.lottie.LottieCompositionFactory;
import com.airbnb.lottie.LottieResult;
import com.airbnb.lottie.network.FileExtension;
import com.airbnb.lottie.network.LottieFetchResult;
import com.airbnb.lottie.network.LottieNetworkFetcher;
import com.airbnb.lottie.network.NetworkCache;
import com.airbnb.lottie.utils.Logger;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipInputStream;

public class NetworkFetcher {
    @NonNull
    private final NetworkCache networkCache;
    @NonNull
    private final LottieNetworkFetcher fetcher;

    public NetworkFetcher(@NonNull NetworkCache networkCache, @NonNull LottieNetworkFetcher fetcher) {
        this.networkCache = networkCache;
        this.fetcher = fetcher;
    }

    @NonNull
    @WorkerThread
    public LottieResult<LottieComposition> fetchSync(@NonNull String url, @Nullable String cacheKey) {
        LottieComposition result = this.fetchFromCache(url, cacheKey);
        if (result != null) {
            return new LottieResult<LottieComposition>(result);
        }
        Logger.debug("Animation for " + url + " not found in cache. Fetching from network.");
        return this.fetchFromNetwork(url, cacheKey);
    }

    @Nullable
    @WorkerThread
    private LottieComposition fetchFromCache(@NonNull String url, @Nullable String cacheKey) {
        if (cacheKey == null) {
            return null;
        }
        Pair<FileExtension, InputStream> cacheResult = this.networkCache.fetch(url);
        if (cacheResult == null) {
            return null;
        }
        FileExtension extension = (FileExtension)((Object)cacheResult.first);
        InputStream inputStream = (InputStream)cacheResult.second;
        LottieResult<LottieComposition> result = extension == FileExtension.ZIP ? LottieCompositionFactory.fromZipStreamSync(new ZipInputStream(inputStream), url) : LottieCompositionFactory.fromJsonInputStreamSync(inputStream, url);
        if (result.getValue() != null) {
            return result.getValue();
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @NonNull
    @WorkerThread
    private LottieResult<LottieComposition> fetchFromNetwork(@NonNull String url, @Nullable String cacheKey) {
        Logger.debug("Fetching " + url);
        LottieFetchResult fetchResult = null;
        try {
            fetchResult = this.fetcher.fetchSync(url);
            if (fetchResult.isSuccessful()) {
                InputStream inputStream = fetchResult.bodyByteStream();
                String contentType = fetchResult.contentType();
                LottieResult<LottieComposition> result = this.fromInputStream(url, inputStream, contentType, cacheKey);
                Logger.debug("Completed fetch from network. Success: " + (result.getValue() != null));
                LottieResult<LottieComposition> lottieResult = result;
                return lottieResult;
            }
            LottieResult<LottieComposition> inputStream = new LottieResult<LottieComposition>(new IllegalArgumentException(fetchResult.error()));
            return inputStream;
        }
        catch (Exception e) {
            LottieResult<LottieComposition> lottieResult = new LottieResult<LottieComposition>(e);
            return lottieResult;
        }
        finally {
            if (fetchResult != null) {
                try {
                    fetchResult.close();
                }
                catch (IOException e) {
                    Logger.warning("LottieFetchResult close failed ", e);
                }
            }
        }
    }

    @NonNull
    private LottieResult<LottieComposition> fromInputStream(@NonNull String url, @NonNull InputStream inputStream, @Nullable String contentType, @Nullable String cacheKey) throws IOException {
        LottieResult<LottieComposition> result;
        FileExtension extension;
        if (contentType == null) {
            contentType = "application/json";
        }
        if (contentType.contains("application/zip") || url.split("\\?")[0].endsWith(".lottie")) {
            Logger.debug("Handling zip response.");
            extension = FileExtension.ZIP;
            result = this.fromZipStream(url, inputStream, cacheKey);
        } else {
            Logger.debug("Received json response.");
            extension = FileExtension.JSON;
            result = this.fromJsonStream(url, inputStream, cacheKey);
        }
        if (cacheKey != null && result.getValue() != null) {
            this.networkCache.renameTempFile(url, extension);
        }
        return result;
    }

    @NonNull
    private LottieResult<LottieComposition> fromZipStream(@NonNull String url, @NonNull InputStream inputStream, @Nullable String cacheKey) throws IOException {
        if (cacheKey == null) {
            return LottieCompositionFactory.fromZipStreamSync(new ZipInputStream(inputStream), null);
        }
        File file = this.networkCache.writeTempCacheFile(url, inputStream, FileExtension.ZIP);
        return LottieCompositionFactory.fromZipStreamSync(new ZipInputStream(new FileInputStream(file)), url);
    }

    @NonNull
    private LottieResult<LottieComposition> fromJsonStream(@NonNull String url, @NonNull InputStream inputStream, @Nullable String cacheKey) throws IOException {
        if (cacheKey == null) {
            return LottieCompositionFactory.fromJsonInputStreamSync(inputStream, null);
        }
        File file = this.networkCache.writeTempCacheFile(url, inputStream, FileExtension.JSON);
        return LottieCompositionFactory.fromJsonInputStreamSync(new FileInputStream(file.getAbsolutePath()), url);
    }
}

