/*
 * Decompiled with CFR 0.152.
 */
package shadow.bundletool.com.android.utils;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.google.common.io.FileWriteMode;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import kotlin.io.FilesKt;
import shadow.bundletool.com.android.utils.PathUtils;

public final class FileUtils {
    private static final Joiner PATH_JOINER = Joiner.on((char)File.separatorChar);
    private static final Joiner COMMA_SEPARATED_JOINER = Joiner.on((String)", ");
    private static final Joiner UNIX_NEW_LINE_JOINER = Joiner.on((char)'\n');

    private FileUtils() {
    }

    public static void deletePath(File path) throws IOException {
        FileUtils.deleteRecursivelyIfExists(path);
    }

    public static void deleteDirectoryContents(File directory) throws IOException {
        Preconditions.checkArgument((boolean)directory.isDirectory(), (Object)"!directory.isDirectory");
        File[] files = directory.listFiles();
        Preconditions.checkNotNull((Object)files);
        for (File file : files) {
            FileUtils.deletePath(file);
        }
    }

    public static void cleanOutputDir(File path) throws IOException {
        if (!path.isDirectory()) {
            if (path.exists()) {
                FileUtils.deletePath(path);
            }
            if (!path.mkdirs()) {
                throw new IOException(String.format("Could not create empty folder %s", path));
            }
            return;
        }
        FileUtils.deleteDirectoryContents(path);
    }

    public static void copyFile(File from, File to2) throws IOException {
        FileUtils.copyFile(from.toPath(), to2.toPath());
    }

    public static void copyFile(Path from, Path to2) throws IOException {
        if (System.getProperty("os.name").toLowerCase(Locale.US).contains("windows")) {
            FileUtils.copyFile(from, to2, StandardCopyOption.REPLACE_EXISTING);
        } else {
            FileUtils.copyFile(from, to2, StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    public static void copyFile(Path from, Path to2, CopyOption ... options) throws IOException {
        Files.copy(from, to2, options);
        FileUtils.setWritable(to2);
    }

    private static void setWritable(Path path) {
        File fileOrNull;
        try {
            fileOrNull = path.toFile();
        }
        catch (UnsupportedOperationException e11) {
            fileOrNull = null;
        }
        if (fileOrNull != null && !fileOrNull.canWrite()) {
            fileOrNull.setWritable(true);
        }
    }

    public static void copyDirectory(File from, File to2) throws IOException {
        Preconditions.checkArgument((boolean)from.isDirectory(), (Object)"Source path is not a directory.");
        Preconditions.checkArgument((!to2.exists() || to2.isDirectory() ? 1 : 0) != 0, (Object)"Destination path exists and is not a directory.");
        FileUtils.mkdirs(to2);
        File[] children = from.listFiles();
        if (children != null) {
            for (File child : children) {
                if (child.isFile()) {
                    FileUtils.copyFileToDirectory(child, to2);
                    continue;
                }
                if (child.isDirectory()) {
                    FileUtils.copyDirectoryToDirectory(child, to2);
                    continue;
                }
                throw new IllegalArgumentException("Don't know how to copy file " + child.getAbsolutePath());
            }
        }
    }

    public static void copyFileToDirectory(File from, File to2) throws IOException {
        FileUtils.copyFile(from, new File(to2, from.getName()));
    }

    public static void copyDirectoryToDirectory(File from, File to2) throws IOException {
        FileUtils.copyDirectory(from, new File(to2, from.getName()));
    }

    public static void copyDirectoryContentToDirectory(File from, File to2) throws IOException {
        Preconditions.checkArgument((boolean)from.isDirectory(), (Object)"Source path is not a directory.");
        File[] children = from.listFiles();
        if (children != null) {
            for (File f11 : children) {
                File destination;
                if (f11.isDirectory()) {
                    destination = new File(to2, FilesKt.toRelativeString((File)f11, (File)from));
                    com.google.common.io.Files.createParentDirs((File)destination);
                    FileUtils.mkdirs(destination);
                    FileUtils.copyDirectoryContentToDirectory(f11, destination);
                    continue;
                }
                if (!f11.isFile()) continue;
                destination = new File(to2, FilesKt.toRelativeString((File)f11.getParentFile(), (File)from));
                com.google.common.io.Files.createParentDirs((File)destination);
                FileUtils.mkdirs(destination);
                FileUtils.copyFileToDirectory(f11, destination);
            }
        }
    }

    public static File mkdirs(File folder) {
        if (!folder.mkdirs() && !folder.isDirectory()) {
            throw new RuntimeException("Cannot create directory " + folder);
        }
        return folder;
    }

    public static void delete(File file) throws IOException {
        Files.delete(file.toPath());
    }

    public static void deleteIfExists(File file) throws IOException {
        Files.deleteIfExists(file.toPath());
    }

    public static void deleteRecursivelyIfExists(File file) throws IOException {
        PathUtils.deleteRecursivelyIfExists(file.toPath());
    }

    public static void renameTo(File file, File to2) throws IOException {
        boolean result = file.renameTo(to2);
        if (!result) {
            throw new IOException("Failed to rename " + file.getAbsolutePath() + " to " + to2);
        }
    }

    public static File join(File dir, String ... paths) {
        if (paths.length == 0) {
            return dir;
        }
        return new File(dir, PATH_JOINER.join((Object[])paths));
    }

    public static File join(File dir, Iterable<String> paths) {
        return new File(dir, PATH_JOINER.join(FileUtils.removeEmpty(paths)));
    }

    public static String join(String ... paths) {
        return PATH_JOINER.join(FileUtils.removeEmpty(Lists.newArrayList((Object[])paths)));
    }

    public static String join(Iterable<String> paths) {
        return PATH_JOINER.join(paths);
    }

    private static Iterable<String> removeEmpty(Iterable<String> input) {
        return Lists.newArrayList(input).stream().filter(it -> !it.isEmpty()).collect(Collectors.toList());
    }

    public static String loadFileWithUnixLineSeparators(File file) throws IOException {
        return UNIX_NEW_LINE_JOINER.join((Iterable)com.google.common.io.Files.readLines((File)file, (Charset)Charsets.UTF_8));
    }

    public static String toSystemDependentPath(String path) {
        if (File.separatorChar != '/') {
            path = path.replace('/', File.separatorChar);
        }
        return path;
    }

    public static String escapeSystemDependentCharsIfNecessary(String path) {
        if (File.separatorChar == '\\') {
            return path.replace("\\", "\\\\");
        }
        return path;
    }

    public static String toSystemIndependentPath(String path) {
        if (File.separatorChar != '/') {
            path = path.replace(File.separatorChar, '/');
        }
        return path;
    }

    public static String toExportableSystemDependentPath(File file) {
        if (File.separatorChar != '/' && !file.getAbsolutePath().startsWith("\\\\?\\")) {
            return "\\\\?\\" + file.getAbsolutePath();
        }
        return file.getAbsolutePath();
    }

    public static String sha1(File file) throws IOException {
        return Hashing.sha1().hashBytes(com.google.common.io.Files.toByteArray((File)file)).toString();
    }

    public static FluentIterable<File> getAllFiles(File dir) {
        return FluentIterable.from((Iterable)com.google.common.io.Files.fileTraverser().depthFirstPreOrder((Object)dir)).filter(com.google.common.io.Files.isFile());
    }

    public static String getNamesAsCommaSeparatedList(Iterable<File> files) {
        return COMMA_SEPARATED_JOINER.join(Iterables.transform(files, File::getName));
    }

    public static String sanitizeFileName(String input) {
        return input.replaceAll("[:\\\\/*\"?|<>']", "_");
    }

    public static String getDirectoryNameForJar(File inputFile) {
        HashFunction hashFunction2 = Hashing.sha1();
        HashCode hashCode = hashFunction2.hashString((CharSequence)inputFile.getAbsolutePath(), Charsets.UTF_16LE);
        String name = com.google.common.io.Files.getNameWithoutExtension((String)inputFile.getName());
        if (name.equals("classes") && inputFile.getAbsolutePath().contains("exploded-aar")) {
            File versionDir = inputFile.getParentFile().getParentFile();
            File artifactDir = versionDir.getParentFile();
            File groupDir = artifactDir.getParentFile();
            name = Joiner.on((char)'-').join((Object)groupDir.getName(), (Object)artifactDir.getName(), new Object[]{versionDir.getName()});
        }
        name = name + "_" + hashCode.toString();
        return name;
    }

    public static void createFile(File file, String content) throws IOException {
        Preconditions.checkArgument((!file.exists() ? 1 : 0) != 0, (String)"%s exists already.", (Object)file);
        FileUtils.writeToFile(file, content);
    }

    public static void writeToFile(File file, String content) throws IOException {
        com.google.common.io.Files.createParentDirs((File)file);
        com.google.common.io.Files.asCharSink((File)file, (Charset)StandardCharsets.UTF_8, (FileWriteMode[])new FileWriteMode[0]).write((CharSequence)content);
    }

    public static List<File> find(File base, Pattern pattern) {
        Preconditions.checkArgument((boolean)base.isDirectory(), (String)"'%s' must be a directory.", (Object)base.getAbsolutePath());
        return FluentIterable.from((Iterable)com.google.common.io.Files.fileTraverser().depthFirstPreOrder((Object)base)).filter(file -> pattern.matcher(FileUtils.toSystemIndependentPath(file.getPath())).find()).toList();
    }

    public static Optional<File> find(File base, String name) {
        Preconditions.checkArgument((boolean)base.isDirectory(), (String)"'%s' must be a directory.", (Object)base.getAbsolutePath());
        return FluentIterable.from((Iterable)com.google.common.io.Files.fileTraverser().depthFirstPreOrder((Object)base)).filter(file -> name.equals(file.getName())).last();
    }

    public static String joinFilePaths(Iterable<File> files) {
        return Joiner.on((char)File.pathSeparatorChar).join(Iterables.transform(files, File::getAbsolutePath));
    }

    public static boolean parentDirExists(File file) {
        File canonicalFile;
        try {
            canonicalFile = file.getCanonicalFile();
        }
        catch (IOException e11) {
            throw new UncheckedIOException(e11);
        }
        return canonicalFile.getParentFile() != null && canonicalFile.getParentFile().exists();
    }

    public static boolean isFileInDirectory(File fileOrDir, File baseDir) {
        String relativePath;
        try {
            relativePath = baseDir.toPath().relativize(fileOrDir.toPath()).toString();
        }
        catch (IllegalArgumentException e11) {
            return false;
        }
        return !relativePath.isEmpty() && !relativePath.startsWith("..");
    }

    public static boolean isSameFile(File file1, File file2) {
        try {
            if (file1.exists() && file2.exists()) {
                return Files.isSameFile(file1.toPath(), file2.toPath());
            }
            return file1.getCanonicalFile().equals(file2.getCanonicalFile());
        }
        catch (IOException e11) {
            throw new UncheckedIOException(e11);
        }
    }

    public static FileSystem createZipFilesystem(Path archive) throws IOException {
        URI uri = URI.create("jar:" + archive.toUri().toString());
        return FileSystems.newFileSystem(uri, Collections.emptyMap());
    }
}

