/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.api.variant

import org.gradle.api.Named
import org.gradle.api.Task
import org.gradle.api.file.Directory
import org.gradle.api.file.RegularFile
import org.gradle.api.file.RegularFileProperty
import org.gradle.api.provider.Provider
import org.gradle.api.tasks.TaskProvider

/**
 * Defines a set of manifests for particular variant.
 * Adding manifest is always added static and generated manifests carry the highest
 * possible priority across all manifests.
 */
interface ManifestFiles: Named {

    /**
     * Get all registered manifests as a [List] of [RegularFile].
     *
     * The outer [List] represents the priority of manifests respective to each other, meaning that
     * elements first in the list overrides elements last in the list.
     *
     * The returned [Provider] can be used directly in a [org.gradle.api.tasks.InputFiles] annotated
     * property of a [Task]
     */
    val all: Provider<out List<RegularFile>>

    /**
     * Add existing manifest file to the variant.
     *
     * The file will be added last in the list of manifest files for the variant. As long as there is
     * a manifest merging process, added manifest will overlay others and become the one with
     * the highest priority.
     *
     * Do not use [addStaticManifestFile] to add sources that are generated by a task,
     * instead use [addGeneratedManifestFile]
     *
     * @param relativeFilePath the manifest file path, that will be resolved using the [Directory.file] API
     * relative to the Gradle project directory.
     */
    fun addStaticManifestFile(relativeFilePath: String)

    /**
     * Add the output of a custom task to the list of manifests.
     *
     * The [RegularFileProperty] is the output of a Task [TASK] that has been registered using the
     * Gradle's Task manager.
     *
     * The manifest is added last to the variant's list and has the highest priority.
     *
     * @param taskProvider the [TaskProvider] returned by Gradle's Task manager when registering the
     * Task of type [TASK].
     * @param wiredWith the method reference returning the [TASK] task's output to use as a source
     * directory. The generated manifest location is automatically determined by the
     * Android Gradle Plugin
     */
    fun <TASK: Task> addGeneratedManifestFile(
            taskProvider: TaskProvider<TASK>,
            wiredWith: (TASK) -> RegularFileProperty
    )
}
