/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.ssh.util;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.io.ByteStreams;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.interfaces.DSAParams;
import java.security.interfaces.DSAPublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.DSAPublicKeySpec;
import java.security.spec.ECFieldFp;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.EllipticCurve;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.annotation.Nonnull;
import javax.annotation.concurrent.NotThreadSafe;
import net.i2p.crypto.eddsa.EdDSAPublicKey;
import net.i2p.crypto.eddsa.spec.EdDSANamedCurveTable;
import net.i2p.crypto.eddsa.spec.EdDSAParameterSpec;
import net.i2p.crypto.eddsa.spec.EdDSAPublicKeySpec;
import org.apache.commons.lang3.StringUtils;
import org.apache.sshd.common.util.buffer.ByteArrayBuffer;
import org.apache.sshd.common.util.security.SecurityUtils;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.ECPointUtil;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.math.ec.custom.sec.SecP256R1Curve;
import org.bouncycastle.math.ec.custom.sec.SecP384R1Curve;
import org.bouncycastle.math.ec.custom.sec.SecP521R1Curve;
import org.bouncycastle.util.encoders.Base64;
import org.bouncycastle.util.encoders.DecoderException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KeyUtils {
    private static final Logger log = LoggerFactory.getLogger(KeyUtils.class);
    private static final String KEY_PREFIX = "AAAA";
    private static final String RSA_ALGORITHM = "RSA";
    private static final String RSA_TYPE = "ssh-rsa";
    private static final String DSA_ALGORITHM = "DSA";
    private static final String DSA_TYPE = "ssh-dss";
    private static final String ECDSA_ALGORITHM = "EC";
    private static final String P256_CURVE = "nistp256";
    private static final String P384_CURVE = "nistp384";
    private static final String P521_CURVE = "nistp521";
    private static final String ECDSA_P256_TYPE = "ecdsa-sha2-nistp256";
    private static final String ECDSA_P384_TYPE = "ecdsa-sha2-nistp384";
    private static final String ECDSA_P521_TYPE = "ecdsa-sha2-nistp521";
    private static final org.bouncycastle.jce.spec.ECParameterSpec ECDSA_P256_SPEC = ECNamedCurveTable.getParameterSpec((String)"secp256r1");
    private static final org.bouncycastle.jce.spec.ECParameterSpec ECDSA_P384_SPEC = ECNamedCurveTable.getParameterSpec((String)"secp384r1");
    private static final org.bouncycastle.jce.spec.ECParameterSpec ECDSA_P521_SPEC = ECNamedCurveTable.getParameterSpec((String)"secp521r1");
    private static final String ED25519_ALGORITHM = "EDDSA";
    private static final String ED25519_TYPE = "ssh-ed25519";
    private static final EdDSAParameterSpec ED25519_SPEC = EdDSANamedCurveTable.getByName((String)"ed25519-sha-512");
    private static final Map<String, KeySpecDecoder> DECODERS = ImmutableMap.builder().put((Object)"ssh-rsa", (Object)new RSAKeySpecDecoder()).put((Object)"ssh-dss", (Object)new DSAKeySpecDecoder()).put((Object)"ecdsa-sha2-nistp256", (Object)new ECDSAKeySpecDecoder()).put((Object)"ecdsa-sha2-nistp384", (Object)new ECDSAKeySpecDecoder()).put((Object)"ecdsa-sha2-nistp521", (Object)new ECDSAKeySpecDecoder()).put((Object)"ssh-ed25519", (Object)new Ed25519KeySpecDecoder()).build();

    public static String computeMD5(byte[] b) {
        try {
            return KeyUtils.asHex(MessageDigest.getInstance("MD5").digest(b));
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException(e);
        }
    }

    public static String getKeyComment(String keyText) {
        if (keyText != null) {
            try {
                return KeyUtils.decodeKeyAndLabel(keyText).getLabel();
            }
            catch (IllegalArgumentException | IllegalStateException e) {
                log.debug("Invalid key: [{}]", (Object)keyText, (Object)e);
            }
        }
        return "";
    }

    public static PublicKey getPublicKey(String keyText) {
        return KeyUtils.decodeKeyAndLabel(keyText).getPublicKey();
    }

    public static String getKeyText(PublicKey publicKey) {
        KeyTypeAndBytes keyTypeAndBytes = KeyUtils.encodePublicKey(publicKey);
        return keyTypeAndBytes.getType() + " " + Base64.toBase64String((byte[])keyTypeAndBytes.getBytes());
    }

    public static String calculateFingerprint(@Nonnull PublicKey key) {
        KeyTypeAndBytes keyTypeAndBytes = KeyUtils.encodePublicKey(key);
        return KeyUtils.computeMD5(keyTypeAndBytes.getBytes());
    }

    private static String asHex(byte[] bytes) {
        StringBuilder str = new StringBuilder(bytes.length * 2);
        for (byte aByte : bytes) {
            int b = 0xFF & aByte;
            if (b < 16) {
                str.append('0');
            }
            str.append(Integer.toHexString(b));
        }
        return str.toString();
    }

    private static PublicKeyAndLabel decodeKeyAndLabel(String keyText) {
        try {
            return KeyUtils.decodeKeyAndLabelInternal(keyText);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException(e);
        }
    }

    private static PublicKeyAndLabel decodeKeyAndLabelInternal(String keyText) throws NoSuchAlgorithmException {
        IllegalArgumentException lastException = null;
        for (KeyBytesAndLabel keyAndLabel : KeyUtils.getKeyPossibilities(keyText)) {
            try {
                return new PublicKeyAndLabel(KeyUtils.decodePublicKey(new SimpleDataReader(keyAndLabel.getDecodedBytes())), keyAndLabel.getLabel());
            }
            catch (InvalidKeySpecException e) {
                lastException = new IllegalArgumentException(e);
            }
            catch (EOFException e) {
                lastException = new IllegalArgumentException("The base64 byte array was not long enough to contain a full key.", e);
            }
        }
        throw lastException;
    }

    private static PublicKey decodePublicKey(SimpleDataReader reader) throws NoSuchAlgorithmException, InvalidKeySpecException, EOFException {
        String type = reader.readString();
        KeySpecDecoder decoder = DECODERS.get(type);
        if (decoder == null) {
            throw new IllegalArgumentException("Unknown public key type " + type);
        }
        try {
            return SecurityUtils.getKeyFactory((String)decoder.getType()).generatePublic(decoder.getSpec(reader));
        }
        catch (GeneralSecurityException e) {
            throw new IllegalArgumentException("BouncyCastle provider was not provided properly", e);
        }
    }

    private static KeyTypeAndBytes encodePublicKey(@Nonnull PublicKey key) {
        String type;
        ByteArrayBuffer buffer = new ByteArrayBuffer();
        switch (key.getAlgorithm().toUpperCase()) {
            case "DSA": {
                DSAPublicKey dsaKey = (DSAPublicKey)key;
                DSAParams params = dsaKey.getParams();
                type = DSA_TYPE;
                buffer.putString(DSA_TYPE);
                buffer.putMPInt(params.getP());
                buffer.putMPInt(params.getQ());
                buffer.putMPInt(params.getG());
                buffer.putMPInt(dsaKey.getY());
                break;
            }
            case "RSA": {
                RSAPublicKey rsaKey = (RSAPublicKey)key;
                type = RSA_TYPE;
                buffer.putString(RSA_TYPE);
                buffer.putMPInt(rsaKey.getPublicExponent());
                buffer.putMPInt(rsaKey.getModulus());
                break;
            }
            case "EC": {
                String identifier;
                String algorithm;
                ECPublicKey ecKey = (ECPublicKey)key;
                BCECPublicKey bcEcKey = new BCECPublicKey(ecKey, BouncyCastleProvider.CONFIGURATION);
                org.bouncycastle.jce.spec.ECParameterSpec ecParameterSpec = bcEcKey.getParameters();
                if (ecParameterSpec.equals((Object)ECDSA_P256_SPEC)) {
                    algorithm = ECDSA_P256_TYPE;
                    identifier = P256_CURVE;
                } else if (ecParameterSpec.equals((Object)ECDSA_P384_SPEC)) {
                    algorithm = ECDSA_P384_TYPE;
                    identifier = P384_CURVE;
                } else if (ecParameterSpec.equals((Object)ECDSA_P521_SPEC)) {
                    algorithm = ECDSA_P521_TYPE;
                    identifier = P521_CURVE;
                } else {
                    throw new IllegalArgumentException("Unable to encode public key: Unsupported elliptic curve");
                }
                type = algorithm;
                buffer.putString(algorithm);
                buffer.putBytes(identifier.getBytes());
                buffer.putBytes(bcEcKey.getQ().getEncoded(false));
                break;
            }
            case "EDDSA": {
                EdDSAPublicKey ed25519Key = (EdDSAPublicKey)key;
                type = ED25519_TYPE;
                buffer.putString(ED25519_TYPE);
                buffer.putBytes(ed25519Key.getAbyte());
                break;
            }
            default: {
                throw new IllegalArgumentException(String.format("Unable to encode public key: Unsupported algorithm of type %s", key.getAlgorithm()));
            }
        }
        return new KeyTypeAndBytes(type, buffer.getCompactData());
    }

    private static List<KeyBytesAndLabel> getKeyPossibilities(String keyText) {
        Object[] keyParts = keyText.split("\\s+");
        ArrayList keyAndLabels = Lists.newArrayList();
        for (int i = 0; i < keyParts.length; ++i) {
            if (!keyParts[i].startsWith(KEY_PREFIX)) continue;
            for (int j = i + 1; j <= keyParts.length; ++j) {
                String encodedText = StringUtils.join((Object[])keyParts, (String)"", (int)i, (int)j);
                String labelText = StringUtils.join((Object[])keyParts, (String)" ", (int)j, (int)keyParts.length);
                try {
                    keyAndLabels.add(new KeyBytesAndLabel(Base64.decode((String)encodedText), labelText));
                    continue;
                }
                catch (DecoderException e) {
                    log.trace("Could not decode possible key text: {}.", (Object)encodedText);
                }
            }
            break;
        }
        if (keyAndLabels.size() > 0) {
            return keyAndLabels;
        }
        throw new IllegalArgumentException("Invalid key: " + keyText);
    }

    @NotThreadSafe
    private static class SimpleDataReader {
        private final DataInputStream data;

        public SimpleDataReader(byte[] data) {
            this.data = new DataInputStream(new ByteArrayInputStream(data));
        }

        private byte[] readBytes() throws EOFException {
            try {
                int i = this.data.readInt();
                byte[] bytes = new byte[i];
                ByteStreams.readFully((InputStream)this.data, (byte[])bytes);
                return bytes;
            }
            catch (EOFException e) {
                throw e;
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        private String readString() throws EOFException {
            return new String(this.readBytes(), StandardCharsets.US_ASCII);
        }

        private BigInteger readBigInteger() throws EOFException {
            return new BigInteger(this.readBytes());
        }
    }

    private static class KeyTypeAndBytes {
        private final String type;
        private final byte[] bytes;

        private KeyTypeAndBytes(String type, byte[] bytes) {
            this.type = type;
            this.bytes = bytes;
        }

        public String getType() {
            return this.type;
        }

        public byte[] getBytes() {
            return this.bytes;
        }
    }

    private static class PublicKeyAndLabel {
        private final PublicKey publicKey;
        private final String label;

        public PublicKeyAndLabel(PublicKey publicKey, String label) {
            this.publicKey = publicKey;
            this.label = label;
        }

        public PublicKey getPublicKey() {
            return this.publicKey;
        }

        public String getLabel() {
            return this.label;
        }
    }

    private static class KeyBytesAndLabel {
        private final byte[] decodedBytes;
        private final String label;

        public KeyBytesAndLabel(byte[] decodedBytes, String label) {
            this.decodedBytes = decodedBytes;
            this.label = label;
        }

        public byte[] getDecodedBytes() {
            return this.decodedBytes;
        }

        public String getLabel() {
            return this.label;
        }
    }

    private static class Ed25519KeySpecDecoder
    implements KeySpecDecoder {
        private Ed25519KeySpecDecoder() {
        }

        @Override
        public String getType() {
            return KeyUtils.ED25519_ALGORITHM;
        }

        @Override
        public KeySpec getSpec(SimpleDataReader reader) throws EOFException {
            byte[] a = reader.readBytes();
            return new EdDSAPublicKeySpec(a, ED25519_SPEC);
        }
    }

    private static class ECDSAKeySpecDecoder
    implements KeySpecDecoder {
        private ECDSAKeySpecDecoder() {
        }

        @Override
        public String getType() {
            return KeyUtils.ECDSA_ALGORITHM;
        }

        @Override
        public ECPublicKeySpec getSpec(SimpleDataReader reader) throws EOFException {
            SecP256R1Curve ecdsaCurve;
            org.bouncycastle.jce.spec.ECParameterSpec ecdsaSpec;
            String identifier = reader.readString();
            byte[] q = reader.readBytes();
            if (identifier.equalsIgnoreCase(KeyUtils.P256_CURVE)) {
                ecdsaSpec = ECDSA_P256_SPEC;
                ecdsaCurve = new SecP256R1Curve();
            } else if (identifier.equalsIgnoreCase(KeyUtils.P384_CURVE)) {
                ecdsaSpec = ECDSA_P384_SPEC;
                ecdsaCurve = new SecP384R1Curve();
            } else if (identifier.equalsIgnoreCase(KeyUtils.P521_CURVE)) {
                ecdsaSpec = ECDSA_P521_SPEC;
                ecdsaCurve = new SecP521R1Curve();
            } else {
                throw new IllegalArgumentException("Unsupported elliptic curve");
            }
            ECFieldFp field = new ECFieldFp(ecdsaCurve.getField().getCharacteristic());
            BigInteger a = ecdsaCurve.getA().toBigInteger();
            BigInteger b = ecdsaCurve.getB().toBigInteger();
            EllipticCurve ec = new EllipticCurve(field, a, b);
            ECPoint g = new ECPoint(ecdsaSpec.getG().getAffineXCoord().toBigInteger(), ecdsaSpec.getG().getAffineYCoord().toBigInteger());
            ECParameterSpec ecSpec = new ECParameterSpec(ec, g, ecdsaSpec.getN(), ecdsaSpec.getH().intValue());
            ECPoint point = ECPointUtil.decodePoint((EllipticCurve)ec, (byte[])q);
            return new ECPublicKeySpec(point, ecSpec);
        }
    }

    private static class DSAKeySpecDecoder
    implements KeySpecDecoder {
        private DSAKeySpecDecoder() {
        }

        @Override
        public String getType() {
            return KeyUtils.DSA_ALGORITHM;
        }

        @Override
        public DSAPublicKeySpec getSpec(SimpleDataReader reader) throws EOFException {
            BigInteger prime = reader.readBigInteger();
            BigInteger subPrime = reader.readBigInteger();
            BigInteger base = reader.readBigInteger();
            BigInteger publicKey = reader.readBigInteger();
            return new DSAPublicKeySpec(publicKey, prime, subPrime, base);
        }
    }

    private static class RSAKeySpecDecoder
    implements KeySpecDecoder {
        private RSAKeySpecDecoder() {
        }

        @Override
        public String getType() {
            return KeyUtils.RSA_ALGORITHM;
        }

        @Override
        public RSAPublicKeySpec getSpec(SimpleDataReader reader) throws EOFException {
            BigInteger publicExponent = reader.readBigInteger();
            BigInteger modulus = reader.readBigInteger();
            return new RSAPublicKeySpec(modulus, publicExponent);
        }
    }

    private static interface KeySpecDecoder {
        public String getType();

        public KeySpec getSpec(SimpleDataReader var1) throws EOFException;
    }
}

