package com.atlassian.cache;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.annotations.PublicApi;

/**
 * An immutable representation of settings for a Cache. The property getters will return
 * <tt>null</tt> to indicate that no setting is specified. The caller is
 * responsible for handling that <tt>null</tt> may be returned.
 * @since 2.0
 */
@PublicApi
public interface CacheSettings
{
    /**
     * @return how long to retain entries (in milliseconds) from when they were last accessed, or <tt>null</tt>.
     */
    @Nullable
    Long getExpireAfterAccess();

    /**
     * Convenience method that returns {@link #getExpireAfterAccess()} if it is not <tt>null</tt>,
     * otherwise returns the supplied <tt>defaultValue</tt>.
     */
    long getExpireAfterAccess(long defaultValue);

    /**
     * @return how long to retain entries (in milliseconds) from when they were created, or <tt>null</tt>.
     */
    @Nullable
    Long getExpireAfterWrite();

    /**
     * Convenience method that returns {@link #getExpireAfterWrite()} if it is not <tt>null</tt>,
     * otherwise returns the supplied <tt>defaultValue</tt>.
     */
    long getExpireAfterWrite(long defaultValue);

    /**
     * @return whether this cache can be flushed by the cache manager when desired, or <tt>null</tt>.
     */
    @Nullable
    Boolean getFlushable();

    /**
     * Convenience method that returns {@link #getFlushable()} if it is not <tt>null</tt>,
     * otherwise returns the supplied <tt>defaultValue</tt>.
     */
    boolean getFlushable(boolean defaultValue);

    /**
     * @return whether this cache should be local to the node (jvm) where the cache is created, or <tt>null</tt>.
     */
    @Nullable
    Boolean getLocal();

    /**
     * Convenience method that returns {@link #getLocal()} if it is not <tt>null</tt>,
     * otherwise returns the supplied <tt>defaultValue</tt>.
     */
    boolean getLocal(boolean defaultValue);

    /**
     * @return the maximum number of entries in the cache, or <tt>null</tt>.
     */
    @Nullable
    Integer getMaxEntries();

    /**
     * Convenience method that returns {@link #getMaxEntries()} if it is not <tt>null</tt>,
     * otherwise returns the supplied <tt>defaultValue</tt>.
     */
    int getMaxEntries(int defaultValue);

    /**
     * @return whether this cache should replicate asynchronously, or <tt>null</tt>.
     */
    @Nullable
    Boolean getReplicateAsynchronously();

    /**
     * Convenience method that returns {@link #getReplicateAsynchronously()} if it is not <tt>null</tt>,
     * otherwise returns the supplied <tt>defaultValue</tt>.
     */
    boolean getReplicateAsynchronously(boolean defaultValue);

    /**
     * @return whether this cache should replicate put and update operations by copying the relevant
     * key and value across the wire, or <tt>null</tt>.
     */
    @Nullable
    Boolean getReplicateViaCopy();

    /**
     * Convenience method that returns {@link #getReplicateViaCopy()} if it is not <tt>null</tt>,
     * otherwise returns the supplied <tt>defaultValue</tt>.
     */
    boolean getReplicateViaCopy(boolean defaultValue);

    /**
     * Returns a new {@link CacheSettings} instance where the current settings
     * are overridden with settings specified in <tt>overrides</tt>. Only properties
     * in <tt>overrides</tt> that are not <tt>null</tt> will be applied.
     * @param overrides contains the settings to override
     * @return a new {@link CacheSettings} instance with the <tt>overrides</tt> settings applied.
     */
    @Nonnull
    CacheSettings override(@Nonnull CacheSettings overrides);
}
