package com.atlassian.crowd.model.event;

import java.util.Objects;

import javax.annotation.Nullable;

import com.google.common.base.MoreObjects;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * @since 2.6.2
 */
// note that this is currently used both as an in-application event sent through the com.atlassian.event.api.EventPublisher,
// as well as an OperationEvent that's returned from com.atlassian.crowd.event.EventStore
public class AliasEvent implements OperationEvent {
    private final Operation operation;
    private final long applicationId;
    private final String username;
    @Nullable
    private final String newAlias;

    private AliasEvent(Operation operation, long applicationId, String username, @Nullable String newAlias) {
        this.operation = checkNotNull(operation);
        this.applicationId = applicationId;
        this.username = checkNotNull(username);
        this.newAlias = newAlias;
    }

    @Override
    public Operation getOperation() {
        return operation;
    }

    public long getApplicationId() {
        return applicationId;
    }

    public String getUsername() {
        return username;
    }

    /**
     * @return the new alias for CREATED and UPDATED events
     */
    @Nullable
    public String getNewAlias() {
        return newAlias;
    }

    /**
     * @return always return null, because aliases are not associated to any directory
     */
    @Override
    public Long getDirectoryId() {
        return null;
    }

    public static AliasEvent created(long applicationId, String user, String newAlias) {
        return new AliasEvent(Operation.CREATED, applicationId, user, newAlias);
    }

    public static AliasEvent updated(long applicationId, String user, String newAlias) {
        return new AliasEvent(Operation.UPDATED, applicationId, user, newAlias);
    }

    public static AliasEvent deleted(String user, long applicationId) {
        return new AliasEvent(Operation.DELETED, applicationId, user, null);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        AliasEvent that = (AliasEvent) o;
        return applicationId == that.applicationId &&
                operation == that.operation &&
                Objects.equals(username, that.username) &&
                Objects.equals(newAlias, that.newAlias);
    }

    @Override
    public int hashCode() {
        return Objects.hash(operation, applicationId, username, newAlias);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("operation", operation)
                .add("applicationId", applicationId)
                .add("username", username)
                .add("newAlias", newAlias)
                .toString();
    }
}
