package com.atlassian.crowd.model.user;

import com.atlassian.crowd.embedded.api.UserComparator;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;

public final class ImmutableUser implements User {
    private final long directoryId;
    private final String name;
    private final boolean active;
    private final String emailAddress;
    private final String displayName;
    private final String firstName;
    private final String lastName;
    private final String externalId;

    public ImmutableUser(final User user) {
        this(user.getDirectoryId(),
                user.getName(),
                user.getDisplayName(),
                user.getEmailAddress(),
                user.isActive(),
                user.getFirstName(),
                user.getLastName(),
                user.getExternalId());
    }

    public ImmutableUser(final long directoryId, final String name, final String displayName, final String emailAddress, final boolean active, final String firstName, final String lastName, final String externalId) {
        this.directoryId = directoryId;
        this.name = Preconditions.checkNotNull(name);
        this.displayName = displayName;
        this.emailAddress = emailAddress;
        this.active = active;
        this.firstName = firstName;
        this.lastName = lastName;
        this.externalId = externalId;
    }

    private ImmutableUser(Builder builder) {
        this.directoryId = builder.directoryId;
        this.name = Preconditions.checkNotNull(builder.name);
        this.displayName = builder.displayName;
        this.emailAddress = builder.emailAddress;
        this.active = builder.active;
        this.firstName = builder.firstName;
        this.lastName = builder.lastName;
        this.externalId = builder.externalId;
    }

    /**
     * @return an {@link ImmutableUser} with the same properties as the given user.
     * Will avoid creating a copy if possible.
     */
    public static ImmutableUser from(final User user) {
        if (user instanceof ImmutableUser) {
            return (ImmutableUser) user;
        }

        return ImmutableUser.builder(user).build();
    }

    public static Builder builder(long directoryId, String name) {
        return new Builder(directoryId, name);
    }

    public static Builder builder(String name) {
        return new Builder(name);
    }

    public static Builder builder(User user) {
        return new Builder(user);
    }


    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public long getDirectoryId() {
        return directoryId;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getEmailAddress() {
        return emailAddress;
    }

    @Override
    public String getDisplayName() {
        return displayName;
    }

    @Override
    public String getFirstName() {
        return firstName;
    }

    @Override
    public String getLastName() {
        return lastName;
    }

    @Override
    public String getExternalId() {
        return externalId;
    }

    @Override
    public boolean equals(final Object o) {
        return UserComparator.equalsObject(this, o);
    }

    @Override
    public int hashCode() {
        return UserComparator.hashCode(this);
    }

    @Override
    public int compareTo(final com.atlassian.crowd.embedded.api.User user) {
        return UserComparator.compareTo(this, user);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("directoryId", directoryId)
                .add("name", name)
                .add("active", active)
                .add("emailAddress", emailAddress)
                .add("displayName", displayName)
                .add("firstName", firstName)
                .add("lastName", lastName)
                .add("externalId", externalId)
                .toString();
    }

    public static class Builder {
        private long directoryId;
        private String name;
        private boolean active;
        private String emailAddress;
        private String displayName;
        private String firstName;
        private String lastName;
        private String externalId;

        public Builder(User user) {
            this.directoryId = user.getDirectoryId();
            this.name = user.getName();
            this.active = user.isActive();
            this.emailAddress = user.getEmailAddress();
            this.displayName = user.getDisplayName();
            this.firstName = user.getFirstName();
            this.lastName = user.getLastName();
            this.externalId = user.getExternalId();
        }

        public Builder(String name) {
            this.name = name;
        }

        public Builder(long directoryId, String name) {
            this.directoryId = directoryId;
            this.name = name;
        }

        public Builder directoryId(long directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder active(boolean active) {
            this.active = active;
            return this;
        }

        public Builder emailAddress(String emailAddress) {
            this.emailAddress = emailAddress;
            return this;
        }

        public Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public Builder firstName(String firstName) {
            this.firstName = firstName;
            return this;
        }

        public Builder lastName(String lastName) {
            this.lastName = lastName;
            return this;
        }

        public Builder externalId(String externalId) {
            this.externalId = externalId;
            return this;
        }

        public ImmutableUser build() {
            return new ImmutableUser(this);
        }
    }
}
