package com.atlassian.crowd.integration.springsecurity;

import com.atlassian.crowd.integration.http.util.CrowdHttpTokenHelper;
import com.atlassian.crowd.model.authentication.CookieConfiguration;
import com.atlassian.crowd.model.authentication.ValidationFactor;
import com.atlassian.crowd.service.client.ClientProperties;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.authentication.LoginUrlAuthenticationEntryPoint;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;
import org.springframework.security.web.savedrequest.DefaultSavedRequest;
import org.springframework.security.web.savedrequest.HttpSessionRequestCache;

import javax.annotation.Nullable;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

public abstract class AbstractCrowdSSOAuthenticationProcessingFilter extends UsernamePasswordAuthenticationFilter {
    protected static final Consumer<AuthenticationException> SILENT_AUTHENTICATION_EXCEPTION_SWALLOWER = (ignore) -> {
    };
    private static final Logger logger = LoggerFactory.getLogger(AbstractCrowdSSOAuthenticationProcessingFilter.class);
    protected final ClientProperties clientProperties;
    protected final CrowdHttpTokenHelper tokenHelper;

    @Nullable
    private RequestToApplicationMapper requestToApplicationMapper;
    @Nullable
    private LoginUrlAuthenticationEntryPoint authenticationProcessingFilterEntryPoint;

    protected AbstractCrowdSSOAuthenticationProcessingFilter(ClientProperties clientProperties, CrowdHttpTokenHelper tokenHelper) {
        this.clientProperties = clientProperties;
        this.tokenHelper = tokenHelper;
    }

    private static String requestUriWithoutContext(HttpServletRequest request) {
        return request.getRequestURI().substring(request.getContextPath().length());
    }

    /**
     * This filter will process all requests, however, if the filterProcessesUrl
     * is part of the request URI, the filter will assume the request is a
     * username/password authentication (login) request and will not check
     * for Crowd SSO authentication. Authentication will proceed as defined in
     * the AuthenticationProcessingFilter.
     * <p>
     * Otherwise, an authentication request to Crowd will be made to verify
     * any existing Crowd SSO token (via the ProviderManager).
     *
     * @param request  servlet request containing either username/password paramaters
     *                 or the Crowd token as a cookie.
     * @param response servlet response to write out cookie.
     * @return <code>true</code> only if the filterProcessesUrl is in the request URI.
     */
    @Override
    protected boolean requiresAuthentication(final HttpServletRequest request, final HttpServletResponse response) {
        boolean usernamePasswordAuthentication = super.requiresAuthentication(request, response);

        if (!usernamePasswordAuthentication) {
            // if this request isn't a direct request for username/password authentication (login)
            // we MUST authenticate the request with Crowd (keep Spring Security's context in sync with
            // Crowd's SSO user)

            Authentication authenticatedToken = getAuthenticatedToken(request, response);

            // update the authentication token to a new token corresponding to the Crowd SSO user
            if (authenticatedToken == null) {
                // no authenticated SSO user, make sure Spring Security also knows there is no user authenticated
                SecurityContextHolder.clearContext();
            } else {
                SecurityContextHolder.getContext().setAuthentication(authenticatedToken);

                // write the cookie out to a token
                storeTokenIfCrowd(request, response, authenticatedToken);
            }
        }

        return usernamePasswordAuthentication;
    }

    protected Authentication getAuthenticatedToken(final HttpServletRequest request, final HttpServletResponse response) {
        ImmutableList.Builder<Pair<Supplier<AbstractAuthenticationToken>, Consumer<AuthenticationException>>> authenticationHandlersBuilder
                = new ImmutableList.Builder<>();
        appendSuppliers(request, response, authenticationHandlersBuilder);

        CrowdSSOAuthenticationDetails details = getAuthenticationDetails(request);
        for (Pair<Supplier<AbstractAuthenticationToken>, Consumer<AuthenticationException>> authenticationHandler : authenticationHandlersBuilder.build()) {
            try {
                AbstractAuthenticationToken authRequest = authenticationHandler.getLeft().get();
                if (authRequest != null) {
                    authRequest.setDetails(details);
                    return getAuthenticationManager().authenticate(authRequest);
                }
            } catch (AuthenticationException e) {
                authenticationHandler.getRight().accept(e);
            }
        }
        return null;
    }

    protected void appendSuppliers(final HttpServletRequest request, final HttpServletResponse response,
                                   ImmutableList.Builder<Pair<Supplier<AbstractAuthenticationToken>, Consumer<AuthenticationException>>> builder) {
        builder.add(Pair.of(() ->
                        Optional.ofNullable(tokenHelper.getCrowdToken(request, clientProperties.getCookieTokenKey()))
                                .map(CrowdSSOAuthenticationToken::new)
                                .orElse(null),
                SILENT_AUTHENTICATION_EXCEPTION_SWALLOWER)); // don't clear SSO token if it fails to authenticate
    }

    /**
     * Provided so that subclasses may configure what is put into the authentication request's details
     * property.
     * <p>
     * Sets the validation factors from the HttpServletRequest on the authentication request. Also sets
     * the application name to the name of application responsible for authorising a particular request.
     * For single-crowd-application-per-spring-security-context web apps, this will just return the application
     * name specified in the ClientProperties. For multi-crowd-applications-per-spring-security-context web apps,
     * the requestToApplicationMapper will be used to determine the application name.
     *
     * @param request     that an authentication request is being created for
     * @param authRequest the authentication request object that should have its details set
     */
    @Override
    protected void setDetails(HttpServletRequest request, UsernamePasswordAuthenticationToken authRequest) {
        doSetDetails(request, authRequest);
    }

    /**
     * <p>If the request has been redirected from a page it was not authorised to see, we want to
     * authenticate the login page using the application of the source page. The only pages that
     * should receive that special treatment are the login page itself and 'j_spring_security_check', the
     * submission target of the login page.</p>
     * <p>This method contains that definition, and will only return <code>true</code> for those pages.</p>
     *
     * @return is it safe to authenticate this resource as if it were the resource saved in the session?
     */
    protected boolean canUseSavedRequestToAuthenticate(HttpServletRequest request) {
        // Checks for j_spring_security_check
        if (super.requiresAuthentication(request, null)) {
            return true;
        }

        if (authenticationProcessingFilterEntryPoint != null) {
            String loginFormUrl = authenticationProcessingFilterEntryPoint.getLoginFormUrl();

            return requestUriWithoutContext(request).equals(loginFormUrl);
        } else {
            return false;
        }
    }

    protected void doSetDetails(HttpServletRequest request, AbstractAuthenticationToken authRequest) {
        authRequest.setDetails(getAuthenticationDetails(request));
    }

    protected CrowdSSOAuthenticationDetails getAuthenticationDetails(HttpServletRequest request) {
        List<ValidationFactor> validationFactors = tokenHelper.getValidationFactorExtractor().getValidationFactors(request);
        final String application;
        if (requestToApplicationMapper != null) {

            // determine the target path
            final String savedPath = canUseSavedRequestToAuthenticate(request) ? getSavedPath(request) : null;
            final String path = savedPath != null ? savedPath : requestUriWithoutContext(request);

            application = requestToApplicationMapper.getApplication(path);
        } else {
            // default to the Crowd application
            application = clientProperties.getApplicationName();
        }

        return new CrowdSSOAuthenticationDetails(application, validationFactors);
    }

    protected String getSavedPath(final HttpServletRequest request) {
        final DefaultSavedRequest savedRequest = (DefaultSavedRequest) new HttpSessionRequestCache().getRequest(request, null);
        if (savedRequest != null) {
            return savedRequest.getRequestURI().substring(savedRequest.getContextPath().length());
        } else{
            return null;
        }
    }

    /**
     * Attempts to write out the successful SSO token to a cookie,
     * if an SSO token was generated and stored via the AuthenticationProvider.
     * <p>
     * This effectively establishes SSO when using the CrowdAuthenticationProvider
     * in conjunction with this filter.
     *
     * @param request    servlet request.
     * @param response   servlet response.
     * @param authResult result of a successful authentication. If it is a CrowdSSOAuthenticationToken
     *                   then the SSO token will be set to the "credentials" property.
     * @throws java.io.IOException not thrown.
     */
    @Override
    protected void successfulAuthentication(HttpServletRequest request, HttpServletResponse response,
                                            FilterChain filterChain, Authentication authResult)
            throws IOException, ServletException {
        storeTokenIfCrowd(request, response, authResult);
        super.successfulAuthentication(request, response, filterChain, authResult);
    }

    protected void storeTokenIfCrowd(HttpServletRequest request, HttpServletResponse response, Authentication authResult) {
        // write successful SSO token if there is one present
        if (authResult instanceof CrowdSSOAuthenticationToken) {
            if (authResult.getCredentials() != null) {
                try {
                    tokenHelper.setCrowdToken(request, response,
                            (String) authResult.getCredentials(),
                            clientProperties, getCookieConfiguration());
                } catch (Exception e) {
                    // occurs if application's auth token expires while trying to look up the domain property from the Crowd server
                    logger.error("Unable to set Crowd SSO token", e);
                }
            }
        }
    }

    @Override
    protected void unsuccessfulAuthentication(HttpServletRequest request, HttpServletResponse response, AuthenticationException failed) throws IOException, ServletException {
        onUnsuccessfulAuthentication(request, response);
        super.unsuccessfulAuthentication(request, response, failed);
    }

    /**
     * Remove any SSO tokens associated with the request, effectively logging the user out of Crowd.
     *
     * @param request  servlet request.
     * @param response servlet response.
     */
    // since we don't use the default spring security remember-me service that would be invalidated by the superclass,
    // implementations should override this and invalidate the token (both in-request, and in-storage)
    protected abstract void onUnsuccessfulAuthentication(HttpServletRequest request, HttpServletResponse response);

    protected abstract CookieConfiguration getCookieConfiguration() throws Exception;

    /**
     * Optional dependency.
     *
     * @param requestToApplicationMapper only required if multiple Crowd "applications" need to
     *                                   be accessed via the same Spring Security context, eg. when one web-application corresponds to
     *                                   multiple Crowd "applications".
     */
    public void setRequestToApplicationMapper(RequestToApplicationMapper requestToApplicationMapper) {
        this.requestToApplicationMapper = requestToApplicationMapper;
    }

    /**
     * Optional dependency, only required if multiple Crowd applications are coexisting in the same
     * web-application. Used to discover the login page, through and treat it specially.
     */
    public void setLoginUrlAuthenticationEntryPoint(LoginUrlAuthenticationEntryPoint filterEntryPoint) {
        this.authenticationProcessingFilterEntryPoint = filterEntryPoint;
    }
}