package com.atlassian.crowd.integration.springsecurity.user;

import com.atlassian.crowd.embedded.api.Attributes;
import com.atlassian.crowd.model.user.User;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;

import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.function.Predicate;

import static com.google.common.base.MoreObjects.firstNonNull;

/**
 * Implements a basic UserWithAttributes wrapper for Crowd principals.
 *
 * All attributes on this object are obtained directly from
 * the Crowd server.
 *
 * @author Shihab Hamid
 */
public class CrowdUserDetails implements UserDetails {
    private static final Attributes EMPTY_ATTRIBUTES = new EmptyAttributes();

    private final User principal;
    private final Attributes attributes;
    private final Collection<GrantedAuthority> authorities;

    public CrowdUserDetails(User principal, Collection<GrantedAuthority> authorities)
    {
        this(principal, principal instanceof Attributes ? (Attributes) principal : EMPTY_ATTRIBUTES, authorities);
    }

    private CrowdUserDetails(User principal, Attributes attributes, Collection<GrantedAuthority> authorities) {
        this.principal = principal;
        this.attributes = attributes;
        this.authorities = firstNonNull(authorities, Collections.<GrantedAuthority>emptySet());
    }

    /**
     * Returns the authorities granted to the user. Cannot return <code>null</code>.
     *
     * @return the authorities (never <code>null</code>)
     */
    @Override
    public Collection<GrantedAuthority> getAuthorities() {
        return authorities;
    }

    /**
     * Returns the remote principal that has authenticated.
     *
     * Before Crowd 2.7, this method was returning SOAPPrincipal. In Crowd 2.7, the return type was changed to
     * the API-neutral UserWithAttributes. Applications using this method will have to be updated.
     *
     * @return remote principal
     */
    public User getRemotePrincipal() {
        return principal;
    }

    /**
     * Returns the password used to authenticate the user. Cannot return <code>null</code>.
     *
     * @return the password (never <code>null</code>). Always throws UnsupportedOperationException as we don't want to risk exposing the password of a user.
     */
    @Override
    public String getPassword() {
        throw new UnsupportedOperationException("Not giving you the password");
    }

    /**
     * Returns the username used to authenticate the user. Cannot return <code>null</code>.
     *
     * @return the username (never <code>null</code>)
     */
    @Override
    public String getUsername() {
        return principal.getName();
    }

    /**
     * Indicates whether the user's account has expired. An expired account cannot be authenticated.
     *
     * @return <code>true</code> always.
     */
    @Override
    public boolean isAccountNonExpired() {
        return true;
    }

    /**
     * Indicates whether the user is locked or unlocked. A locked user cannot be authenticated.
     *
     * @return <code>true</code> always.
     */
    @Override
    public boolean isAccountNonLocked() {
        return true;
    }

    /**
     * Indicates whether the user's credentials (password) has expired. Expired credentials prevent
     * authentication.
     *
     * @return <code>true</code> always.
     */
    @Override
    public boolean isCredentialsNonExpired() {
        return true;
    }

    /**
     * Indicates whether the user is enabled or disabled. A disabled user cannot be authenticated.
     *
     * @return <code>true</code> if the user is active, <code>false</code> otherwise.
     */
    @Override
    public boolean isEnabled() {
        return principal.isActive();
    }

    public String getFirstName() {
        return principal.getFirstName();
    }

    public String getLastName() {
        return principal.getLastName();
    }

    public String getEmail() {
        return principal.getEmailAddress();
    }

    public String getAttribute(String attributeName) {
        return attributes.getValue(attributeName);
    }

    public String getFullName() {
        return principal.getDisplayName();
    }

    public boolean hasAuthority(Predicate<GrantedAuthority> authorityPredicate) {
        return authorities.stream().anyMatch(authorityPredicate);
    }

    private static class EmptyAttributes implements Attributes, Serializable {
        @Override
        public Set<String> getKeys() {
            return Collections.emptySet();
        }

        @Override
        @Nullable
        public String getValue(String arg0) {
            return null;
        }

        @Override
        public Set<String> getValues(String arg0) {
            return Collections.emptySet();
        }

        @Override
        public boolean isEmpty() {
            return true;
        }
    }
}
