package com.atlassian.crowd.embedded.impl;

import com.google.common.collect.ForwardingSet;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;

import java.util.Collection;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static com.atlassian.crowd.embedded.impl.IdentifierUtils.toLowerCase;

/**
 * This class behaves like a HashSet with lower-case String values. All element arguments are
 * lower-cased before further processing.
 */
public class IdentifierSet extends ForwardingSet<String> {
    public static final IdentifierSet EMPTY = new IdentifierSet(ImmutableSet::of);

    private final Set<String> delegate;

    private IdentifierSet(Supplier<Set<String>> supplier) {
        this.delegate = supplier.get();
    }

    public IdentifierSet() {
        this.delegate = Sets.newHashSet();
    }

    public IdentifierSet(int expectedSize) {
        this.delegate = Sets.newHashSetWithExpectedSize(expectedSize);
    }

    public IdentifierSet(Collection<String> names) {
        this(names.size());
        addAll(names);
    }

    @Override
    protected Set<String> delegate() {
        return delegate;
    }

    @Override
    public boolean removeAll(Collection<?> collection) {
        return delegate().removeAll(lowercase(collection));
    }

    @Override
    public boolean contains(Object object) {
        return delegate().contains(lowercase(object));
    }

    @Override
    public boolean add(String element) {
        return delegate().add(toLowerCase(element));
    }

    @Override
    public boolean remove(Object object) {
        return delegate().remove(lowercase(object));
    }

    @Override
    public boolean containsAll(Collection<?> collection) {
        return delegate().containsAll(lowercase(collection));
    }

    @Override
    public boolean addAll(Collection<? extends String> strings) {
        return delegate().addAll(lowercaseStrings(strings));
    }

    @Override
    public boolean retainAll(Collection<?> collection) {
        return delegate().retainAll(lowercase(collection));
    }

    public static IdentifierSet difference(Collection<String> set1, Collection<String> set2) {
        IdentifierSet result = new IdentifierSet(set1);
        result.removeAll(set2);
        return result;
    }

    public static Set<String> differenceWithOriginalCasing(Collection<String> set1, Collection<String> set2) {
        return set1.stream().filter(IdentifierUtils.containsIdentifierPredicate(set2).negate()).
                collect(Collectors.toSet());
    }

    public static IdentifierSet intersection(Collection<String> set1, Collection<String> set2) {
        boolean firstSmaller = set1.size() < set2.size();
        IdentifierSet smaller = new IdentifierSet(firstSmaller ? set1 : set2);
        smaller.retainAll(firstSmaller ? set2 : set1);
        return smaller;
    }

    private static Object lowercase(Object element) {
        return element instanceof String ? IdentifierUtils.toLowerCase((String) element) : element;
    }

    private static Collection<?> lowercase(Collection<?> collection) {
        if (collection instanceof IdentifierSet) {
            return ((IdentifierSet) collection).delegate();
        }
        return collection.stream().map(IdentifierSet::lowercase).collect(Collectors.toSet());
    }

    private static Collection<? extends String> lowercaseStrings(Collection<? extends String> collection) {
        return (collection instanceof IdentifierSet) ? ((IdentifierSet) collection).delegate() : IdentifierUtils.toLowerCase(collection);
    }
}
