package com.atlassian.plugin.webresource;

import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.webresource.url.UrlParameters;
import com.google.common.base.Objects;
import com.google.common.collect.ImmutableSet;

import java.util.Map;

/**
 * Represents a batch of plugin resources. <p/>
 * <p/>
 * It provides methods to parse and generate urls to locate a batch of plugin resources. <p/>
 * <p/>
 *
 * @since 2.2
 */
public class BatchPluginResource extends AbstractPluginResource
{
    private final String completeKey;
    private final String type;
    private final String name;

    // not thread-safe but a safe race on setting as hash function is referentially transparent
    private int hash = 0;

    /**
     * This constructor should only ever be used internally within this class. It does not ensure that the
     * resourceName's file extension is the same as the given type. It is up to the calling code to ensure this.
     *
     * @param completeKey - the resource key
     * @param type - the resource type
     * @param urlParams - contributed url parameters
     * @param params - the parameters of the resource (ieonly, media, etc)
     */
    BatchPluginResource(Globals globals, String completeKey, String type, final UrlParameters urlParams,
            final Map<String, String> params, BatchedWebResourceDescriptor batchedWebResourceDescriptor)
    {
        super(globals, urlParams, params, ImmutableSet.of(batchedWebResourceDescriptor));
        this.completeKey = completeKey;
        this.type = type;
        this.name = completeKey + "." + type;
    }

    @Override
    public String getResourceName()
    {
        return name;
    }

    @Override
    public String getVersion()
    {
        final Plugin plugin = globals.getConfig().getIntegration().getPluginAccessor().getEnabledPluginModule
                (getCompleteKey()).getPlugin();
        return plugin.getPluginInformation().getVersion();
    }

    public String getCompleteKey()
    {
        return completeKey;
    }

    @Override
    public boolean isCacheSupported()
    {
        return !"false".equals(params.get("cache"));
    }

    @Override
    public String getType()
    {
        return type;
    }

    @Override
    public boolean equals(final Object o)
    {
        if (this == o)
        {
            return true;
        }

        if ((o == null) || (getClass() != o.getClass()))
        {
            return false;
        }

        final BatchPluginResource that = (BatchPluginResource) o;

        return Objects.equal(params, that.params) && Objects.equal(name, that.name);
    }

    @Override
    public int hashCode()
    {
        if (hash != 0)
        {
            hash = Objects.hashCode(name, params);
        }

        return hash;
    }

    @Override
    public String getUrl(boolean isAbsolute)
    {
        return globals.getRouter().cloneWithNewUrlMode(isAbsolute).urlFor(this);
    }

    @Override
    public Map<String, String> getParamsWithBatchableParams()
    {
        return ResourceUtils.mergeParamList(params, urlParams.toQueryString());
    }

    @Override
    public String toString()
    {
        return "[BatchPluginResource " + name + "]";
    }
}
