package com.atlassian.plugin.webresource;

import com.atlassian.plugin.webresource.condition.ConditionsCache;
import com.google.common.base.Function;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;

import com.google.common.collect.Sets;
import com.atlassian.plugin.webresource.condition.ConditionState;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.transform;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newLinkedHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.collect.Sets.newLinkedHashSet;

/**
 * Performs a calculation on many referenced contexts, and produces an set of intermingled batched-contexts and residual
 * (skipped) resources. Some of the input contexts may have been merged into cross-context batches.
 * The batches are constructed in such a way that no batch is dependent on another.
 * The output batches and resources may be intermingled so as to preserve the input order as much as possible.
 *
 * @since 2.9.0
 */
public class ContextBatchBuilder
{
    private static final Logger log = LoggerFactory.getLogger(ContextBatchBuilder.class);

    private final PluginResourceLocator pluginResourceLocator;
    private final ResourceDependencyResolver dependencyResolver;
    private final ResourceBatchingConfiguration batchingConfiguration;

    private final List<String> allIncludedResources = newArrayList();
    private final Set<String> skippedResources = newHashSet();

    public ContextBatchBuilder(final PluginResourceLocator pluginResourceLocator, final ResourceDependencyResolver dependencyResolver, ResourceBatchingConfiguration batchingConfiguration)
    {
        this.pluginResourceLocator = pluginResourceLocator;
        this.dependencyResolver = dependencyResolver;
        this.batchingConfiguration = batchingConfiguration;
    }
        
    public Iterable<PluginResource> build(final List<String> includedContexts, final Set<String> excludedContexts,
                                          ConditionState conditionsRun, ConditionsCache conditionsCache)
    {
        return build(includedContexts, excludedContexts, DefaultWebResourceFilter.INSTANCE, conditionsRun, conditionsCache);
    }
    
    public Iterable<PluginResource> build(final List<String> includedContexts,
                                          final Set<String> excludedContexts,
                                          final WebResourceFilter filter,
                                          ConditionState conditionsRun,
                                          ConditionsCache conditionsCache)
    {
        if (batchingConfiguration.isContextBatchingEnabled())
        {
            return buildBatched(includedContexts, excludedContexts, filter, conditionsRun, conditionsCache);
        }
        else
        {
            return getUnbatchedResources(includedContexts, excludedContexts, filter, conditionsRun, conditionsCache);
        }
    }
    
    /**
     * @param includedContexts the ordering of these contexts is important since their placement within the resultant URL determines the order that resources
     * will be included in the batch.
     * @param excludedContexts order of these contexts is not important, they do not affect the position of resources. Instead they cause resources not to
     * be present.
     * @param filter
     * @param conditionsRun
     * @return
     * @since 2.12
     */
    private Iterable<PluginResource> buildBatched(final List<String> includedContexts,
                                                  final Set<String> excludedContexts,
                                                  final WebResourceFilter filter,
                                                  final ConditionState conditionsRun,
                                                  ConditionsCache conditionsCache)
    {
        Set<String> incrementalExcludedContexts = null == excludedContexts ? Sets.<String>newHashSet() : newHashSet(excludedContexts);
        Set<String> conditionalIncludedResources = new HashSet<String>();
        WebResourceKeysToContextBatches includedBatches = WebResourceKeysToContextBatches.create(includedContexts, dependencyResolver, pluginResourceLocator, filter, conditionsRun, conditionsCache);
        WebResourceKeysToContextBatches excludedBatches = WebResourceKeysToContextBatches.create(excludedContexts, dependencyResolver, pluginResourceLocator, filter, conditionsRun, conditionsCache);
        skippedResources.addAll(includedBatches.getSkippedResources());
        
        // There are three levels to consider here. In order:
        // 1. Type (CSS/JS)
        // 2. Parameters (ieOnly, media, etc)
        // 3. Context
        final List<ContextBatch> batches = newArrayList();

        // This working list will be reduced as each context is handled.
        final List<String> batchesToProcess = newArrayList(includedBatches.getContextBatchKeys());
        // This is the list of everything that's been included
        final Set<String> superAlreadyProcessedBatches = newHashSet();

        final ContextBatchOperations contextBatchOperations = new ContextBatchOperations(pluginResourceLocator, filter);

        while (!batchesToProcess.isEmpty())
        {
            String context = batchesToProcess.remove(0);
            Set<String> innerAlreadyProcessedBatches = newHashSet(superAlreadyProcessedBatches);
            innerAlreadyProcessedBatches.add(context);
            ContextBatch mergedContextBatch = includedBatches.getContextBatch(context);

            if (!context.equals(DefaultResourceDependencyResolver.IMPLICIT_CONTEXT_NAME))
            {
                Iterator<WebResourceModuleDescriptor> resourceIterator = mergedContextBatch.getResources().iterator();
                while (resourceIterator.hasNext())
                {
                    WebResourceModuleDescriptor contextResource = resourceIterator.next();
                    String resourceKey = contextResource.getCompleteKey();
                    // check for an overlap with the other batches (take into account only the batches not yet processed).
                    Collection<ContextBatch> additionalContexts = includedBatches.getAdditionalContextsForResourceKey(resourceKey, innerAlreadyProcessedBatches);

                    if (!additionalContexts.isEmpty())
                    {
                        if (log.isDebugEnabled())
                        {
                            for (ContextBatch additionalContext : additionalContexts)
                            {
                                log.debug("Context: {} shares a resource with {}: {}", new String[]{ mergedContextBatch.getKey(), additionalContext.getKey(), contextResource.getCompleteKey() });
                            }
                        }

                        List<ContextBatch> contextsToMerge = new ArrayList<ContextBatch>(1 + additionalContexts.size());
                        contextsToMerge.add(mergedContextBatch);
                        contextsToMerge.addAll(additionalContexts);
                        mergedContextBatch = contextBatchOperations.merge(contextsToMerge);

                        // remove the merged batches from those to be processed
                        Collection<String> additionalContextKeys = Collections2.transform(additionalContexts, new Function<ContextBatch, String>() {
                            @Override
                            public String apply(ContextBatch context)
                            {
                                return context.getKey();
                            }
                        });
                        batchesToProcess.removeAll(additionalContextKeys);
                        innerAlreadyProcessedBatches.addAll(additionalContextKeys);

                        // As a new overlapping context is merged, restart the resource iterator so we can check for new resources
                        // that may have been added via the merge.
                        resourceIterator = mergedContextBatch.getResources().iterator();
                    }
                }
            }

            superAlreadyProcessedBatches.addAll(innerAlreadyProcessedBatches);

            // We separate the search for excluded batches since we want to perform the subtraction
            // after all the merging has been done. If you do a subtraction then a merge you cannot
            // ensure (with ContextBatch as it is currently implemented) that the merge will not
            // bring back in previously excluded resources.
            if (!excludedBatches.isEmpty())
            {
                Iterator<WebResourceModuleDescriptor> resourceIterator = mergedContextBatch.getResources().iterator();
                while (resourceIterator.hasNext())
                {
                    WebResourceModuleDescriptor contextResource = resourceIterator.next();
                    String resourceKey = contextResource.getCompleteKey();

                    Collection<ContextBatch> excludeContexts = excludedBatches.getContextsForResourceKey(resourceKey);
                    if (!excludeContexts.isEmpty())
                    {
                        mergedContextBatch = contextBatchOperations.subtract(mergedContextBatch, excludeContexts, conditionsRun, conditionsCache);
                    }
                }

                skippedResources.removeAll(excludedBatches.getSkippedResources());
            }

            // check that we still have resources in this batch - if not, the batch is not required.
            if (excludedBatches.isEmpty() || Iterables.size(mergedContextBatch.getResources()) != 0)
            {
                Iterables.addAll(allIncludedResources, mergedContextBatch.getResourceKeys());
                batches.add(mergedContextBatch);
            }
            else if (log.isDebugEnabled())
            {
                log.debug("The context batch {} contains no resources so will be dropped.", mergedContextBatch.getKey());
            }

            incrementalExcludedContexts.addAll(mergedContextBatch.getContexts());
            excludedBatches = WebResourceKeysToContextBatches.create(incrementalExcludedContexts, dependencyResolver, pluginResourceLocator, filter, conditionsRun, conditionsCache);
        }

        final boolean resplitMergedBatches = batchingConfiguration.resplitMergedContextBatchesForThisRequest();
        // Build the batch resources
        return concat(transform(batches, new Function<ContextBatch, Iterable<PluginResource>>()
        {
            public Iterable<PluginResource> apply(final ContextBatch batch)
            {
                return batch.buildPluginResources(resplitMergedBatches);
            }
        }));
    }

    // If context batching is not enabled, then just add all the resources that would have been added in the context anyway.
    private Iterable<PluginResource> getUnbatchedResources(final Iterable<String> includedContexts,
                                                           final Iterable<String> excludedContexts,
                                                           final WebResourceFilter filter,
                                                           final ConditionState conditionsRun,
                                                           ConditionsCache conditionsCache)
    {        
        Set<String> excludedResourceKeys = new HashSet<String>();
        Set<String> excludedSkippedResources = new HashSet<String>(); // the resources that an excluded batch will not contain
        
        if (excludedContexts != null && Iterables.size(excludedContexts) > 0)
        {
            for (final String context : excludedContexts)
            {
                Iterable<WebResourceModuleDescriptor> contextResources = dependencyResolver.getDependenciesInContext(context, false, excludedSkippedResources, conditionsRun, conditionsCache);
                for (final WebResourceModuleDescriptor contextResource : contextResources)
                {
                    excludedResourceKeys.add(contextResource.getCompleteKey());
                }
            }            
        }
        
        LinkedHashSet<PluginResource> includedResources = new LinkedHashSet<PluginResource>();
        Set<String> includedSkippedResources = new HashSet<String>(); // the resources that an included batch will not contain
        
        for (final String context : includedContexts)
        {
            Iterable<WebResourceModuleDescriptor> contextResources = dependencyResolver.getDependenciesInContext(context, false, includedSkippedResources, conditionsRun, conditionsCache);
            for (final WebResourceModuleDescriptor contextResource : contextResources)
            {
                String completeKey = contextResource.getCompleteKey();
                if (!excludedResourceKeys.contains(completeKey) && !allIncludedResources.contains(completeKey))
                {
                    final List<PluginResource> moduleResources = pluginResourceLocator.getPluginResources(contextResource.getCompleteKey(), conditionsRun, conditionsCache);
                    for (final PluginResource moduleResource : moduleResources)
                    {
                        if (filter.matches(moduleResource.getResourceName()))
                        {
                            includedResources.add(moduleResource);
                        }
                    }

                    allIncludedResources.add(contextResource.getCompleteKey());
                }                
            }
        }
        
        includedSkippedResources.removeAll(excludedSkippedResources);
        skippedResources.addAll(includedSkippedResources);

        return includedResources;
    }

    public Iterable<String> getAllIncludedResources()
    {
        return allIncludedResources;
    }

    public Iterable<String> getSkippedResources()
    {
        return skippedResources;
    }
    
    
    private static class WebResourceKeysToContextBatches
    {
        /**
         * Create a Map of {@link WebResourceModuleDescriptor} key to the contexts that includes them. If a
         * {@link WebResourceModuleDescriptor} exists in multiple contexts then each context will be
         * referenced. The ContextBatches created at this point are pure - they do not take into account
         * overlaps or exclusions, they simply contain all the resources for their context name.
         *
         * @param contexts the contexts to create a mapping for
         * @param dependencyResolver used to construct the identified contexts
         * @param pluginResourceLocator used to find the individual resources for a WebResourceModuleDescriptor before filtering them.
         * @param filter the filter selecting the resources to be included in the batch
         * @param conditionsRun
         * @return a WebResourceToContextsMap containing the required mapping.
         */
        static WebResourceKeysToContextBatches create(final Iterable<String> contexts, final ResourceDependencyResolver dependencyResolver,
            PluginResourceLocator pluginResourceLocator,
            final WebResourceFilter filter, ConditionState conditionsRun, ConditionsCache conditionsCache)
        {
            final ListMultimap<String, String> resourceKeyToContext = ArrayListMultimap.create();
            final Map<String, ContextBatch> batches = newLinkedHashMap();
            final Set<String> skippedResources = newHashSet();

            if (null != contexts)
            {
                for (String context : contexts)
                {
                    Iterable<WebResourceModuleDescriptor> dependencies = dependencyResolver.getDependenciesInContext(context, false, skippedResources, conditionsRun, conditionsCache);

                    ContextBatch batch = new ContextBatch(pluginResourceLocator.temporaryWayToGetGlobalsDoNotUseIt(), context, dependencies);
                    for (WebResourceModuleDescriptor moduleDescriptor : dependencies)
                    {
                        String key = moduleDescriptor.getCompleteKey();
                        boolean matchedPluginResource = false;

                        for (final PluginResource pluginResource : pluginResourceLocator.getPluginResources(moduleDescriptor.getCompleteKey(), conditionsRun, conditionsCache))
                        {
                            if (filter.matches(pluginResource.getResourceName()))
                            {
                                batch.addResourceType(pluginResource);
                                matchedPluginResource = true;
                            }
                        }

                        // If this is the default filter, we should return resources that contain data.
                        // This is kinda dodgy - perhaps it would be better to add a returnData() method to WebResourceFilter
                        if (filter == DefaultWebResourceFilter.INSTANCE && !moduleDescriptor.getDataProviders().isEmpty())
                        {
                            matchedPluginResource = true;
                        }

                        if (matchedPluginResource)
                        {
                            resourceKeyToContext.put(key, context);
                            batches.put(context, batch);
                        }
                    }
                }
            }

            return new WebResourceKeysToContextBatches(resourceKeyToContext, batches, skippedResources);
        }
        
        private final ListMultimap<String, String> resourceToContextBatches;
        private final Map<String, ContextBatch> knownBatches;
        private final Set<String> skippedResources;        
        
        private WebResourceKeysToContextBatches(ListMultimap<String, String> resourceKeyToContext, Map<String, ContextBatch> allBatches, Set<String> skippedResources)
        {
            this.resourceToContextBatches = resourceKeyToContext;
            this.knownBatches = allBatches;
            this.skippedResources = skippedResources;
        }
        
        /**
         * @param key the resource key to find contexts for
         * @return all contexts the specified resource is included in. An empty List is returned if none.
         */
        Collection<ContextBatch> getContextsForResourceKey(String key)
        {
            return getAdditionalContextsForResourceKey(key, null);
        }        
        
        /**
         * @param key the resource key to be mapped to contexts
         * @param knownContexts the contexts we already know about (may be null)
         * @return a List of any additional contexts that the identified resource key can be found in. If there
         * are no additional contexts then an empty List is returned.
         */
        Collection<ContextBatch> getAdditionalContextsForResourceKey(String key, Collection<String> knownContexts)
        {
            List<String> allContexts = resourceToContextBatches.get(key);
            if (CollectionUtils.isEmpty(allContexts))
            {
                return Collections.emptyList();
            }
            
            LinkedHashSet<String> contexts = newLinkedHashSet(allContexts);
            if (CollectionUtils.isNotEmpty(knownContexts))
            {
                contexts.removeAll(knownContexts);
            }

            return Collections2.transform(contexts, new Function<String, ContextBatch>()
            {
                @Override
                public ContextBatch apply(String context)
                {
                    return knownBatches.get(context);
                }
            });
        }
        
        /**
         * @return all the ContextBatches referenced in this class.
         */
        Iterable<String> getContextBatchKeys()
        {
            return knownBatches.keySet();
        }

        private ContextBatch getContextBatch(String context)
        {
            return knownBatches.get(context);
        }

        /**
         * @return the conditional resources that could not be mapped
         */
        public Set<String> getSkippedResources()
        {
            return skippedResources;
        }

        boolean isEmpty() { return knownBatches.isEmpty(); }
    }
}
