package com.atlassian.plugin.webresource;

import com.atlassian.annotations.Internal;
import com.atlassian.plugin.cache.filecache.Cache;
import com.atlassian.plugin.cache.filecache.impl.FileCacheImpl;
import com.atlassian.plugin.cache.filecache.impl.PassThroughCache;
import com.atlassian.plugin.webresource.http.Router;
import com.atlassian.plugin.webresource.support.ResettableLazyReferenceWithVersionCheck;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * WARNING Do not use it, it will be removed in the next version!
 *
 * Set of global final stateless objects like configuration, cache etc.
 *
 * @since 3.3
 */
@Internal
public class Globals
{
    private final Config config;
    private final Logger logger = LoggerFactory.getLogger("webresource");
    private final ResettableLazyReferenceWithVersionCheck<Map<String, Bundle>> cachedBundles;
    private final List<StateChangeCallback> stateChangeCallbacks;

    // Not all fields are final because some of them are initialized in a complex way and can't be set in constructor,
    // but, there are guards that will not allow to change those fields once they are initialized.
    private Router router;
    private Cache cache;

    public Globals(final Config config)
    {
        this.config = config;

        stateChangeCallbacks = new ArrayList<StateChangeCallback>();

        // Caching web resources. This cache get flushed on two events, the first is the plugin change event,
        // the second if config version has been changed (checked for every access).
        cachedBundles = new ResettableLazyReferenceWithVersionCheck<Map<String, Bundle>>()
        {
            @Override
            protected Map<String, Bundle> create()
            {
                return config.getWebResourcesWithoutCache();
            }

            @Override
            protected int getVersion()
            {
                return config.partialHashCode();
            }
        };
        onStateChange(new StateChangeCallback()
        {
            @Override
            public void apply()
            {
                cachedBundles.reset();
            }
        });
    }

    public Config getConfig()
    {
        return config;
    }

    public Router getRouter()
    {
        return router;
    }

    public void setRouter(Router router)
    {
        if (this.router != null)
        {
            throw new RuntimeException("router already set!");
        }
        this.router = router;
    }

    public Cache getCache()
    {
        return cache;
    }

    public Logger getLogger()
    {
        return logger;
    }

    interface StateChangeCallback
    {
        public void apply();
    }

    /**
     * Provided callback would be called if the state of the system changed (any plugin change event). Used mostly to
     * setup cache clear listeners.
     */
    public void onStateChange(StateChangeCallback callback)
    {
        stateChangeCallbacks.add(callback);
    }

    /**
     * Trigger the state change event, it will notify all state change listeners. Used mostly to clear cache.
     */
    public void triggerStateChange()
    {
        for (StateChangeCallback callback : stateChangeCallbacks)
        {
            callback.apply();
        }
    }

    /**
     * Build and setup cache according to configuration setting.
     */
    public void buildAndSetCache()
    {
        if (cache != null)
        {
            throw new RuntimeException("cache already set!");
        }
        if (config.isCacheEnabled())
        {
            try
            {
                cache = new FileCacheImpl(config.getCacheDirectory(), config.getCacheSize());
                onStateChange(new StateChangeCallback()
                {
                    @Override
                    public void apply()
                    {
                        cache.clear();
                    }
                });
            }
            catch (Exception e)
            {
                logger.error("Could not create file cache object, will startup with filecaching disabled, "
                        + "please investigate the cause and correct it.", e);
                cache = new PassThroughCache();
            }
        }
        else
        {
            cache = new PassThroughCache();
        }
    }

    /**
     * Analyzes current state of the plugin system and return web resources. Uses cache, the cache would be cleared on
     * any plugin change event.
     */
    public Map<String, Bundle> getBundles()
    {
        return cachedBundles.get();
    }
}