package com.atlassian.plugin.webresource;

import com.atlassian.annotations.Internal;
import com.atlassian.plugin.servlet.DownloadException;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.webresource.condition.DecoratingCondition;
import com.atlassian.plugin.webresource.support.LineCountingProxyOutputStream;
import com.atlassian.plugin.webresource.support.SourceMapJoinerStub;
import com.atlassian.plugin.webresource.transformer.TransformerParameters;
import com.atlassian.sourcemap.SourceMap;
import com.atlassian.sourcemap.SourceMapJoiner;
import com.atlassian.sourcemap.Util;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.base.Supplier;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import static com.atlassian.plugin.webresource.Config.couldBeVirtualContext;
import static com.atlassian.plugin.webresource.Config.virtualContextKeyToWebResourceKey;
import static com.google.common.base.Preconditions.checkArgument;

/**
 * WARNING Do not use it, it will be removed in the next version!
 *
 * Stateless helper functions.
 *
 * @since 3.3
 */
@Internal
public class Helpers
{
    private static final Logger log = LoggerFactory.getLogger(Helpers.class);

    /**
     * If condition for given Web Resource is satisfied.
     */
    public static boolean isConditionSatisfied(Bundle bundle, Map<String, String> httpParams)
    {
        if (bundle instanceof WebResource)
        {
            DecoratingCondition condition = ((WebResource) bundle).getCondition();
            return condition == null || condition.shouldDisplay(QueryParams.of(httpParams));
        }
        else
        {
            return true;
        }
    }

    /**
     * Condition check in a form of predicate.
     */
    public static Predicate<Bundle> isConditionSatisfiedPredicate(final Map<String, String> httpParams)
    {
        return new Predicate<Bundle>()
        {
            @Override
            public boolean apply(Bundle bundle)
            {
                return isConditionSatisfied(bundle, httpParams);
            }
        };
    }

    /**
     * Get resource for Web Resource or Plugin, also resolves relative paths.
     *
     * @param completeKey key of web resource or plugin.
     * @param resourceName name of Resource.
     */
    public static Resource getResource(Globals globals, RequestCache cache, String completeKey, String resourceName)
    {
        Resource resource = getWebResourceResource(globals, cache, completeKey, resourceName);
        if (resource == null)
        {
            resource = getModuleResource(globals, completeKey, resourceName);
        }
        if (resource == null)
        {
            resource = getResourceRelativeToWebResource(globals, cache, completeKey, resourceName);
        }
        if (resource == null)
        {
            resource = getPluginResource(globals, completeKey, resourceName);
        }
        if (resource == null)
        {
            resource = getResourceRelativeToPlugin(globals, completeKey, resourceName);
        }
        return resource;
    }

    /**
     * Get Resources for list of Web Resources.
     *
     * @param keys keys of Web Resources
     */
    public static Collection<Resource> getResources(Globals globals, RequestCache cache, Collection<String> keys)
    {
        List<Resource> resources = new LinkedList<Resource>();
        for (String key : keys)
        {
            Bundle bundle = globals.getBundles().get(key);
            if (bundle == null)
            {
                continue;
            }
            for (Resource resource : bundle.getResources(cache).values())
            {
                resources.add(resource);
            }
        }
        return resources;
    }

    /**
     * Get Resource for non WebResource Module.
     *
     * @param completeKeyOrPluginKey complete key or plugin key.
     * @param resourceName name of Resource.
     */
    public static Resource getModuleResource(Globals globals, String completeKeyOrPluginKey, String resourceName)
    {
        return globals.getConfig().getModuleResource(completeKeyOrPluginKey, resourceName);
    }

    /**
     * Get Resource for Plugin.
     *
     * @param completeKeyOrPluginKey complete key or plugin key.
     * @param resourceName name of Resource.
     */
    public static Resource getPluginResource(Globals globals, String completeKeyOrPluginKey, String resourceName)
    {
        return globals.getConfig().getPluginResource(getPluginKey(completeKeyOrPluginKey), resourceName);
    }

    /**
     * Get Resource for Web Resource.
     *
     * @param completeKey complete key of Web Resource.
     * @param resourceName name of Resource.
     */
    public static Resource getWebResourceResource(Globals globals, RequestCache cache, String completeKey,
            String resourceName)
    {
        Bundle bundle = globals.getBundles().get(completeKey);
        if (bundle == null)
        {
            return null;
        }
        return bundle.getResources(cache).get(resourceName);
    }

    /**
     * Get Resource for one of Web Resources.
     *
     * @param bundles list of keys of Web Resources.
     * @param resourceName name of Resource.
     */
    public static Resource getResource(Globals globals, RequestCache cache, Collection<String> bundles,
            String resourceName)
    {
        Resource resource = null;
        for (String key : bundles)
        {
            if ((resource = getResource(globals, cache, key, resourceName)) != null)
            {
                break;
            }
        }
        return resource;
    }

    /**
     * Resolve dependencies with filtering predicate and return list of Web Resources.
     *
     * @param included keys of included Web Resources.
     * @param excluded keys of excluded Web Resources.
     */
    public static Collection<String> resolveBundles(Globals globals, Collection<String> included,
            Collection<String> excluded, boolean resolveDeepDependencies, boolean includeLegacy,
            Map<String, String> httpParams)
    {
        LinkedHashSet<String> includedResolved = resolveDependencies(globals, included, Sets.newHashSet(excluded),
                resolveDeepDependencies, includeLegacy, isConditionSatisfiedPredicate(httpParams));
        // Excluding for the second time, to exclude shared deep dependencies.
        includedResolved.removeAll(resolveDependencies(globals, excluded, true, includeLegacy));
        return includedResolved;
    }

    /**
     * Apply conditions and transformations to list of Resources and generate resulting Batch.
     *
     * @param resources list of Resources
     * @param params http params.
     * @return resulting content.
     */
    public static Content transform(final Globals globals, final String type
        , final Supplier<Collection<Resource>> resources, final Map<String, String> params)
    {
        return new ContentImpl(null, true)
        {
            @Override
            public SourceMap writeTo(OutputStream out, boolean isSourceMapEnabled)
            {
                ResourceContentAnnotator[] annotators = globals.getConfig().getContentAnnotators(type);
                try
                {
                    SourceMapJoiner sourceMapJoiner = isSourceMapEnabled ? new SourceMapJoiner() : new
                            SourceMapJoinerStub();
                    boolean isFirst = true;
                    for (Resource resource : resources.get())
                    {
                        // Resources should be delimited by new line, it's needed for source map to work properly.
                        if (isFirst)
                        {
                            isFirst = false;
                        }
                        else
                        {
                            out.write('\n');
                        }

                        int offset = 0;
                        // Byte counting proxy needed to calculate the length of the resource,
                        // needed for the source map generation.
                        LineCountingProxyOutputStream lineCountingStream = new LineCountingProxyOutputStream(out);
                        if (isSourceMapEnabled)
                        {
                            out = lineCountingStream;
                        }
                        for (ResourceContentAnnotator annotator : annotators)
                        {
                            offset += annotator.beforeOffset();
                            annotator.before(resource, out);
                        }

                        Content content = transform(globals, resource, params);
                        SourceMap sourceMap = null;
                        try
                        {
                            sourceMap = content.writeTo(out, isSourceMapEnabled);
                        }
                        catch(RuntimeException e)
                        {
                            // PLUGWEB-261 there's an unknown bug that seems to be caused by reading the non existing resource.
                            // It's unclear what may cause it - the stale cache in WRM or other reasons.
                            // This workaround doesn't fix it but at least it silence the exception and allows other resources
                            // to be served.
                            boolean hasWebResourceInGlobals = globals.getBundles().get(resource.getParent().getKey()) != null;
                            boolean hasWebResourceInPluginSystem = globals.getConfig().getIntegration().getPluginAccessor().getEnabledPluginModule(resource.getParent().getKey()) != null;
                            log.error(
                                "can't get content for \"" + resource.getParent().getKey() + "/" + resource.getLocation() + "\" (" +
                                "it " + (hasWebResourceInGlobals ? "exists in globals" : "doesn't exists in globals") +
                                ", it " + (hasWebResourceInPluginSystem? "exists in plugin system" : "doesn't exists in plugin system") +
                                ")", e
                            );
                        }

                        int resourceLength = lineCountingStream.getLinesCount() - offset;

                        // Apply the after annotators in reverse order
                        for (int i = annotators.length - 1; i >= 0; i--)
                        {
                            annotators[i].after(resource, out);
                        }

                        // If there's no source map generated by transformers,
                        // the 1to1 source map should be generated because
                        // the source map needed to generate the batch source map.
                        // And, instead of the source url the url of transformed single resource should be used (because
                        // transformers without source map support may be already applied).
                        if (isSourceMapEnabled && sourceMap == null)
                        {
                            String singleResourceUrl = globals.getRouter().resourceUrlWithoutHash(resource, params);
                            sourceMap = Util.create1to1SourceMap(resourceLength, singleResourceUrl);
                        }

                        // Adding the resource source map to the batch source map.
                        sourceMapJoiner.add(sourceMap, lineCountingStream.getLinesCount(), offset);
                    }
                    return sourceMapJoiner.join();
                }
                catch (IOException e)
                {
                    throw new RuntimeException(e);
                }
            }
        };
    }

    /**
     * Transform given Resource by applying Transformers and Static Transformers.
     *
     * @param resource the resource.
     * @param params http params.
     * @return resulting content.
     */
    public static Content transform(final Globals globals, final Resource resource, final Map<String, String> params)
    {
        checkArgument(!resource.isRedirect(), "can't transform redirect resource!");

        Content content = resource.getContent();
        if (!resource.isTransformable())
        {
            return content;
        }

        String sourceUrl = globals.getRouter().sourceUrl(resource);
        content = applyTransformers(globals, resource, content, params, sourceUrl);
        content = applyStaticTransformers(globals, resource, content, params, sourceUrl);
        return content;
    }

    /**
     * Apply Transformers to Resource.
     *
     * @param resource resource.
     * @param content content of resource.
     * @param params http params.
     * @param sourceUrl url of source code for resource.
     * @return resulting content.
     */
    protected static Content applyTransformers(final Globals globals, final Resource resource, final Content content,
            final Map<String, String> params, final String sourceUrl)
    {
        if (resource.getParent() instanceof WebResource)
        {
            WebResource webResource = (WebResource) resource.getParent();
            final List<WebResourceTransformation> transformations = webResource.getTransformations();
            if (transformations == null || transformations.isEmpty())
            {
                return content;
            }

            Content lastContent = content;
            for (WebResourceTransformation transformation : transformations)
            {
                // There's another version of `matches` that matches against extension,
                // but it can't be used because sometimes
                // transformers are matched against things like "some-name.public.js".matches("public.js") so if it
                // would be
                // changed to matching the extension it wouldn't work.
                if (transformation.matches(resource.getResourceLocation()))
                {
                    lastContent = transformation.transform(globals.getConfig().getTransformerCache(), lastContent,
                            resource.getResourceLocation(), webResource.getPluginKey(), resource.getFilePath(),
                            webResource.getWebResourceKey(), QueryParams.of(params), sourceUrl);
                }
            }
            return lastContent;
        }
        else
        {
            return content;
        }
    }

    /**
     * Apply Static Transformers to Resource.
     *
     * @param resource resource.
     * @param content content of resource.
     * @param params http params.
     * @param sourceUrl url of source code for resource.
     * @return resulting content.
     */
    protected static Content applyStaticTransformers(final Globals globals, Resource resource, Content content,
            Map<String, String> params, String sourceUrl)
    {
        if (resource.getParent() instanceof WebResource)
        {
            WebResource webResource = (WebResource) resource.getParent();
            return globals.getConfig().getStaticTransformers().transform(content,
                    new TransformerParameters(webResource.getPluginKey(), webResource.getWebResourceKey()),
                    resource.getResourceLocation(), QueryParams.of(params), sourceUrl);
        }
        else
        {
            return content;
        }
    }

    /**
     * Select Resources that should be included in batch.
     *
     * @param resources list of Resources
     * @param type type of batch.
     * @param params http params of batch.
     * @return list of selected Resources.
     */
    public static Collection<Resource> selectForBatch(Collection<Resource> resources, String type, Map<String,
            String> params)
    {
        List<Resource> selected = new LinkedList<Resource>();
        for (Resource resource : resources)
        {
            if (type.equals(resource.getType()) && resource.isBatchable(params))
            {
                selected.add(resource);
            }
        }
        return selected;
    }

    /**
     * Resolve dependencies with conditions taken into account for Web Resources.
     *
     * @param included keys of Web Resources.
     * @return keys of resolved Web Resources.
     */
    protected static LinkedHashSet<String> resolveDependencies(Globals globals, Collection<String> included,
            boolean resolveDeepDependencies, boolean includeLegacy)
    {
        return resolveDependencies(globals, included, new HashSet<String>(), resolveDeepDependencies
            , includeLegacy, Predicates.<Bundle>alwaysTrue());
    }

    /**
     * Resolve dependencies with filtering predicate for lists of included and excluded Web Resources.
     *
     * @param included keys of included Web Resources.
     * @param excluded keys of excluded Web Resources.
     * @return keys of resolved Web Resources.
     */
    protected static LinkedHashSet<String> resolveDependencies(Globals globals, Collection<String> included,
            Set<String> excluded, boolean resolveDeepDependencies, boolean includeLegacy, Predicate<Bundle> isIncluded)
    {
        LinkedHashSet<String> dependencies = new LinkedHashSet<String>();
        for (String key : included)
        {
            if (!excluded.contains(key))
            {
                resolveBundleDependencies(globals, key, excluded, new LinkedHashSet<String>(), resolveDeepDependencies,
                        includeLegacy, isIncluded, dependencies);
            }
        }
        return dependencies;
    }

    /**
     * Resolve dependencies for Web Resource.
     */
    protected static void resolveBundleDependencies(Globals globals, String key, Set<String> excluded,
            Set<String> stack, boolean resolveDeepDependencies, boolean includeLegacy,
            Predicate<Bundle> isIncluded, LinkedHashSet<String> dependencies)
    {
        if (dependencies.contains(key))
        {
            return;
        }
        if (excluded.contains(key))
        {
            return;
        }
        if (!stack.add(key))
        {
            globals.getLogger().warn("cyclic plugin resource dependency has been detected with: {}, stack trace: {}",
                    key, stack);
            return;
        }

        Bundle bundle = globals.getBundles().get(key);
        if (bundle != null)
        {
            if (isIncluded.apply(bundle) && (includeLegacy || !bundle.hasLegacyConditions()))
            {
                if (resolveDeepDependencies)
                {
                    for (String dependencyKey : bundle.getDependencies())
                    {
                        resolveBundleDependencies(globals, dependencyKey, excluded, stack, true, includeLegacy
                            , isIncluded, dependencies);
                    }
                }
                dependencies.add(key);
            }
        }
        else
        {
            // If bundle not found it could be the virtual context, trying to resolve it.
            if (couldBeVirtualContext(key))
            {
                resolveBundleDependencies(globals, virtualContextKeyToWebResourceKey(key), excluded, stack, true, includeLegacy
                    , isIncluded, dependencies);
            }
        }

        stack.remove(key);
    }

    /**
     * Helper to check for equality without bothering to handle nulls.
     */
    public static boolean equals(Object a, Object b)
    {
        return a == null ? b == null : a.equals(b);
    }

    /**
     * Get Resource relative to Web Resource.
     */
    @Deprecated
    public static Resource getResourceRelativeToWebResource(Globals globals, RequestCache cache, String completeKey,
            String resourceName)
    {
        Bundle bundle = globals.getBundles().get(completeKey);
        if (bundle == null)
        {
            return null;
        }
        String filePath = "";
        Resource resource = null;
        while (resource == null)
        {
            String[] parts = splitLastPathPart(resourceName);
            if (parts == null)
            {
                return null;
            }
            resourceName = parts[0];
            filePath = parts[1] + filePath;
            resource = bundle.getResources(cache).get(resourceName);
        }

        final String finalFilePath = filePath;
        return new Resource(resource.getParent(), resource.getResourceLocation(), resource.getType(),
                resource.isBatchable(), resource.isRedirect(), resource.isCacheable())
        {
            @Override
            public String getFilePath()
            {
                return finalFilePath;
            }
        };
    }

    /**
     * Get Resource relative to Plugin.
     */
    @Deprecated
    public static Resource getResourceRelativeToPlugin(Globals globals, String completeKeyOrPluginKey,
            String resourceName)
    {
        String pluginKey = getPluginKey(completeKeyOrPluginKey);
        String filePath = "";
        Resource resource = null;
        while (resource == null)
        {
            String[] parts = splitLastPathPart(resourceName);
            if (parts == null)
            {
                return null;
            }
            resourceName = parts[0];
            filePath = parts[1] + filePath;
            resource = globals.getConfig().getPluginResource(pluginKey, resourceName);
        }

        final String finalFilePath = filePath;
        return new Resource(resource.getParent(), resource.getResourceLocation(), resource.getType(),
                resource.isBatchable(), resource.isRedirect(), resource.isCacheable())
        {
            @Override
            public String getFilePath()
            {
                return finalFilePath;
            }
        };
    }

    /**
     * Split path into parent folder and name.
     */
    @Deprecated
    protected static String[] splitLastPathPart(String resourcePath)
    {
        int indexOfSlash = resourcePath.lastIndexOf('/');
        // skip over the trailing slash
        if (resourcePath.endsWith("/"))
        {
            indexOfSlash = resourcePath.lastIndexOf('/', indexOfSlash - 1);
        }
        if (indexOfSlash < 0)
        {
            return null;
        }
        return new String[] { resourcePath.substring(0, indexOfSlash + 1), resourcePath.substring(indexOfSlash + 1) };
    }

    /**
     * In case of Plugin Key - returns the same key, in case of Web Resource key - extracts Plugin key from it.
     */
    public static String getPluginKey(String completeKeyOrPluginKey)
    {
        return completeKeyOrPluginKey.split(":")[0];
    }

    /**
     * Adapter, turns Content into DownloadableResource.
     */
    public static DownloadableResource asDownloadableResource(final Content content)
    {
        return new DownloadableResource()
        {
            @Override
            public boolean isResourceModified(HttpServletRequest request, HttpServletResponse response)
            {
                throw new RuntimeException("not supported for content wrapper!");
            }

            @Override
            public void serveResource(HttpServletRequest request, HttpServletResponse response)
            {
                throw new RuntimeException("not supported for content wrapper!");
            }

            @Override
            public void streamResource(OutputStream out)
            {
                content.writeTo(out, false);
            }

            @Override
            public String getContentType()
            {
                return content.getContentType();
            }
        };
    }

    /**
     * Adapter, turns DownloadableResource into Content.
     */
    public static Content asContent(final DownloadableResource downloadableResource, final SourceMap sourceMap,
            boolean isTransformed)
    {
        return new ContentImpl(downloadableResource.getContentType(), isTransformed)
        {
            @Override
            public SourceMap writeTo(OutputStream out, boolean isSourceMapEnabled)
            {
                try
                {
                    downloadableResource.streamResource(out);
                }
                catch (DownloadException e)
                {
                    throw new RuntimeException(e);
                }
                return sourceMap;
            }
        };
    }

    public static Content buildEmptyContent(Content content)
    {
        return new ContentImpl(content.getContentType(), false)
        {
            @Override
            public SourceMap writeTo(OutputStream out, boolean isSourceMapEnabled)
            {
                return null;
            }
        };
    }
}