package com.atlassian.plugin.webresource;

import com.atlassian.fugue.Option;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.elements.ResourceDescriptor;
import com.atlassian.plugin.event.PluginEventListener;
import com.atlassian.plugin.event.PluginEventManager;
import com.atlassian.plugin.event.events.PluginDisabledEvent;
import com.atlassian.plugin.event.events.PluginEnabledEvent;
import com.atlassian.plugin.event.events.PluginModuleDisabledEvent;
import com.atlassian.plugin.event.events.PluginModuleEnabledEvent;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.servlet.ServletContextFactory;
import com.atlassian.plugin.webresource.condition.ConditionState;
import com.atlassian.plugin.webresource.condition.ConditionsCache;
import com.atlassian.plugin.webresource.data.DefaultPluginDataResource;
import com.atlassian.plugin.webresource.http.Controller;
import com.atlassian.plugin.webresource.http.Router;
import com.atlassian.plugin.webresource.support.http.Request;
import com.atlassian.plugin.webresource.support.http.Response;
import com.atlassian.plugin.webresource.transformer.DefaultStaticTransformers;
import com.atlassian.plugin.webresource.transformer.DefaultStaticTransformersSupplier;
import com.atlassian.plugin.webresource.transformer.StaticTransformers;
import com.atlassian.plugin.webresource.transformer.TransformerCache;
import com.atlassian.plugin.webresource.url.DefaultUrlBuilderMap;
import com.atlassian.plugin.webresource.url.UrlParameters;
import com.atlassian.sourcemap.SourceMap;
import com.atlassian.webresource.api.data.PluginDataResource;
import com.atlassian.webresource.api.data.WebResourceDataProvider;
import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import static com.atlassian.plugin.webresource.Helpers.asDownloadableResource;

/**
 * Default implementation of {@link PluginResourceLocator}.
 *
 * @since 2.2
 */
public class PluginResourceLocatorImpl implements PluginResourceLocator
{
    private static final Logger log = LoggerFactory.getLogger(PluginResourceLocatorImpl.class);

    final private PluginAccessor pluginAccessor;
    final private TransformerCache transformerCache;
    final private StaticTransformers staticTransformers;
    final private ResourceBatchingConfiguration batchingConfiguration;
    static final String RESOURCE_SOURCE_PARAM = "source";

    private final Globals globals;

    public PluginResourceLocatorImpl(final WebResourceIntegration webResourceIntegration,
            final ServletContextFactory servletContextFactory, final WebResourceUrlProvider webResourceUrlProvider,
            final PluginEventManager pluginEventManager)
    {
        this(webResourceIntegration, servletContextFactory, webResourceUrlProvider,
                new DefaultResourceBatchingConfiguration(), pluginEventManager);
    }

    public PluginResourceLocatorImpl(final WebResourceIntegration webResourceIntegration,
            final ServletContextFactory servletContextFactory, final WebResourceUrlProvider webResourceUrlProvider,
            final ResourceBatchingConfiguration batchingConfiguration, final PluginEventManager pluginEventManager)
    {
        this(webResourceIntegration, servletContextFactory, webResourceUrlProvider,
            new DefaultResourceDependencyResolver(webResourceIntegration, batchingConfiguration),
            batchingConfiguration, pluginEventManager,
            new DefaultStaticTransformers(new DefaultStaticTransformersSupplier(webResourceIntegration, webResourceUrlProvider)));
    }

    public PluginResourceLocatorImpl(final WebResourceIntegration webResourceIntegration,
            final ServletContextFactory servletContextFactory, final WebResourceUrlProvider webResourceUrlProvider,
            final ResourceBatchingConfiguration batchingConfiguration, final PluginEventManager pluginEventManager,
            final StaticTransformers staticTransformers)
    {
        this(webResourceIntegration, servletContextFactory, webResourceUrlProvider,
            new DefaultResourceDependencyResolver(webResourceIntegration, batchingConfiguration),
            batchingConfiguration, pluginEventManager, staticTransformers);
    }

    private PluginResourceLocatorImpl(final WebResourceIntegration webResourceIntegration,
            final ServletContextFactory servletContextFactory, final WebResourceUrlProvider webResourceUrlProvider,
            final ResourceDependencyResolver dependencyResolver, final ResourceBatchingConfiguration batchingConfiguration,
            final PluginEventManager pluginEventManager, final StaticTransformers staticTransformers)
    {
        this.pluginAccessor = webResourceIntegration.getPluginAccessor();
        this.staticTransformers = staticTransformers;
        this.batchingConfiguration = batchingConfiguration;

        this.transformerCache = new TransformerCache(pluginEventManager, pluginAccessor);

        // Creating globals.
        Config config = new Config(batchingConfiguration, webResourceIntegration, webResourceUrlProvider, servletContextFactory
            , staticTransformers, transformerCache);
        globals = new Globals(config);
        globals.buildAndSetCache();
        globals.setRouter(new Router(globals));

        // TODO find better way to register, move it outside of the constructor.
        pluginEventManager.register(this);
    }

    @PluginEventListener
    public void onPluginDisabled(final PluginDisabledEvent event)
    {
        globals.triggerStateChange();
    }

    @PluginEventListener
    public void onPluginEnabled(final PluginEnabledEvent event)
    {
        globals.triggerStateChange();
    }

    @PluginEventListener
    public void onPluginModuleEnabled(final PluginModuleEnabledEvent event)
    {
        if (event.getModule() instanceof WebResourceModuleDescriptor)
        {
            globals.triggerStateChange();
        }
    }

    @PluginEventListener
    public void onPluginModuleDisabled(final PluginModuleDisabledEvent event)
    {
        if (event.getModule() instanceof WebResourceModuleDescriptor)
        {
            globals.triggerStateChange();
        }
    }

    @Override
    public boolean matches(final String url)
    {
        return globals.getRouter().canDispatch(url);
    }

    @Override
    public DownloadableResource getDownloadableResource(final String url, Map<String, String> queryParams)
    {
        // For unknown reason Confluence could pass null as query params.
        if (queryParams == null)
        {
            queryParams = new HashMap<String, String>();
        }

        // This code should be deleted when Confluence would be updated and stop using
        // PluginResourceLocator.getDownloadableResource
        // See https://ecosystem.atlassian.net/browse/PLUGWEB-193
        final DownloadableResource[] downloadableResource = new DownloadableResource[] { null };
        Router router = new Router(globals)
        {
            @Override
            protected Controller createController(final Globals globals, final Request request, final Response response)
            {
                return new Controller(globals, request, response)
                {
                    @Override
                    protected void sendCached(final String cacheKey, final Content content, final boolean
                            isCacheEnabled)
                    {
                        final CachedContent cachedContent = buildCachedContent(cacheKey, content, isCacheEnabled);
                        downloadableResource[0] = asDownloadableResource(
                            new ContentImpl(content.getContentType(), content.isTransformed())
                            {
                                @Override
                                public SourceMap writeTo(final OutputStream out, final boolean isSourceMapEnabled)
                                {
                                    cachedContent.writeTo(out, null, null, false, null);
                                    return null;
                                }
                            }
                        );
                    }

                    @Override
                    protected boolean checkIfCachedAndNotModified(final Date updatedAt)
                    {
                        return false;
                    }
                };
            }
        };
        router.dispatch(new Request(globals, url, queryParams), null);
        return downloadableResource[0];
    }

    @Override
    public Iterable<PluginDataResource> getPluginDataResources(final String moduleCompleteKey, ConditionsCache conditionsCache)
    {
        Option<WebResourceModuleDescriptor> option = getDescriptor(moduleCompleteKey);
        if (option.isEmpty())
        {
            return Collections.emptyList();
        }

        return getPluginDataResources(option.get(), conditionsCache);
    }

    @Override
    public Iterable<PluginDataResource> getPluginDataResources(final WebResourceModuleDescriptor moduleDescriptor, ConditionsCache conditionsCache)
    {
        if (moduleDescriptor.getDataProviders().isEmpty() || (moduleDescriptor.canEncodeStateIntoUrl() && !moduleDescriptor.shouldDisplayImmediate(conditionsCache)))
        {
            return new ArrayList<PluginDataResource>();
        }
        else
        {
            return Iterables.transform(moduleDescriptor.getDataProviders().entrySet(), new Function<Map.Entry<String, WebResourceDataProvider>, PluginDataResource>()
            {
                @Override
                public PluginDataResource apply(Map.Entry<String, WebResourceDataProvider> input)
                {
                    return new DefaultPluginDataResource(moduleDescriptor.getCompleteKey() + "." + input.getKey(), input.getValue().get());
                }
            });
        }
    }

    @Override
    public List<PluginResource> getPluginResources(final String moduleCompleteKey, ConditionState conditionsRun, ConditionsCache conditionsCache)
    {
        Option<WebResourceModuleDescriptor> option = getDescriptor(moduleCompleteKey);
        if (option.isEmpty())
        {
            return Collections.emptyList();
        }

        WebResourceModuleDescriptor wrmd = option.get();
        return getPluginResources(wrmd, conditionsRun, conditionsCache);
    }

    @Override
    public List<PluginResource> getPluginResources(WebResourceModuleDescriptor wrmd,
                                                   ConditionState conditionsRun,
                                                   ConditionsCache conditionsCache)
    {
        final boolean singleMode = !batchingConfiguration.isPluginWebResourceBatchingEnabled();
        final Set<PluginResource> resources = new LinkedHashSet<PluginResource>();

        DefaultUrlBuilderMap urlBuilderMap = new DefaultUrlBuilderMap(wrmd, transformerCache, staticTransformers);

        for (final ResourceDescriptor resourceDescriptor : wrmd.getResourceDescriptors())
        {
            String sourceType = ResourceUtils.getType(resourceDescriptor.getLocation());
            String destinationType = ResourceUtils.getType(resourceDescriptor.getName());
            UrlParameters urlParams = urlBuilderMap.getOrCreateForType(sourceType, conditionsRun, conditionsCache, globals);

            if (singleMode || skipBatch(resourceDescriptor))
            {
                final boolean cache = !"false".equalsIgnoreCase(resourceDescriptor.getParameter("cache"));
                resources.add(new SinglePluginResource(globals, resourceDescriptor.getName(), wrmd.getCompleteKey(),
                    cache, urlParams, resourceDescriptor.getParameters()));
            }
            else
            {
                final BatchPluginResource batchResource = createBatchResource(wrmd.getCompleteKey(), resourceDescriptor,
                        destinationType, urlParams,
                        new BatchedWebResourceDescriptor(wrmd.getPlugin().getPluginInformation().getVersion(), wrmd.getCompleteKey()));
                resources.add(batchResource);
            }
        }
        return ImmutableList.copyOf(resources);
    }

    @Override
    public Option<WebResourceModuleDescriptor> getDescriptor(String moduleCompleteKey)
    {
        final ModuleDescriptor<?> moduleDescriptor = pluginAccessor.getEnabledPluginModule(moduleCompleteKey);
        if ((moduleDescriptor == null) || !(moduleDescriptor instanceof WebResourceModuleDescriptor))
        {
            log.error("Error loading resource \"{}\". Resource is not a Web Resource Module", moduleCompleteKey);
            return Option.none();
        }
        return Option.some((WebResourceModuleDescriptor) moduleDescriptor);
    }

    private BatchPluginResource createBatchResource(final String completeKey,
                                                    final ResourceDescriptor resourceDescriptor,
                                                    final String type,
                                                    final UrlParameters urlParams,
                                                    final BatchedWebResourceDescriptor batchedWebResourceDescriptor)
    {
        final Map<String, String> params = new TreeMap<String, String>();
        for (final String param : BATCH_PARAMS)
        {
            final String value = resourceDescriptor.getParameter(param);
            if (StringUtils.isNotEmpty(value))
            {
                params.put(param, value);
            }
        }

        return new BatchPluginResource(globals, completeKey, type, urlParams, params, batchedWebResourceDescriptor);
    }

    public TransformerCache getTransformerCache() {
        return transformerCache;
    }

    public Globals temporaryWayToGetGlobalsDoNotUseIt()
    {
        return globals;
    }

    private boolean skipBatch(final ResourceDescriptor resourceDescriptor)
    {
        // you can't batch forwarded requests
        final boolean doNotBatch = "false".equalsIgnoreCase(resourceDescriptor.getParameter(Config.RESOURCE_BATCH_PARAM));
        return doNotBatch || "webContext".equalsIgnoreCase(resourceDescriptor.getParameter(Config.RESOURCE_SOURCE_PARAM));
    }
}
