package com.atlassian.plugins.avatar;

import com.google.common.base.Function;

/**
 * Interface for services that provide avatar images for a users (or other domain objects). The typical use case for
 * avatars is users, although this interface doesn't care if they are users. T is the type of the domain object the
 * avatar is for and I is the type of the id of the Avatar objects (if any).
 */
@SuppressWarnings("UnusedDeclaration")
public interface AvatarProvider<T, I> {

    /**
     * Gets the Avatar for the given user at the given size. If the size is not possible, a best effort is made, to
     * return one with a size that is usable in a user-facing client. This should be the next larger size if one is
     * available, otherwise the next smaller size. If the size cannot be interpreted, the default size should be
     * returned.
     * <p>
     * Implementations that cannot provide an avatar for the given user can return a URL for default/unknown users.
     * </p>
     *
     * This method should only be called when implementations are expected to be able to fulfil the request in a
     * performant way - if the implementation may need to lookup the underlying domain object, prefer
     * {@link #getAvatar(AvatarOwner, int)} or otherwise ensure the fetch is cheap.
     *
     * @param identifier the String version of the application-specific unique primary key for the user.
     * @param size the size of the avatar in pixels.
     * @return the avatar at the requested size or another size if that size is not possible.
     */
    Avatar getAvatar(String identifier, int size);

    /**
     * Fetches the Avatar belonging to the given avatarOwner domain object.
     *
     * @param avatarOwner the user or other application-specific domain object to which the Avatar belongs.
     * @param size the size of the avatar in pixels.
     * @return the Avatar.
     */
    Avatar getAvatar(AvatarOwner<T> avatarOwner, int size);

    /**
     * Optional - Returns the Avatar by its own ID.
     *
     * @throws UnsupportedOperationException for implementations that do not have AvatarIds
     */
    Avatar getAvatarById(I avatarId, int size);


    /**
     * Fetches the Avatar belonging to the given avatarOwner domain object or just applies provided fallback function.
     *
     * @param avatarOwner the user or other application-specific domain object to which the Avatar belongs.
     * @param fallbackFunction implementation of the function that fetches avatar for the user provided by the application.
     * @param size the size of the avatar in pixels.
     * @return the Avatar.
     */
    Avatar getAvatar(AvatarOwner<T> avatarOwner, Function<AvatarOwner<T>, Avatar> fallbackFunction, int size);

    // TODO need way for front-end to ask if/how they can edit avatar. JIRA's AvatarService method is hasCustomUserAvatar(ApplicationUser remoteUser, ApplicationUser user)
}
