package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;

import java.time.Duration;
import java.util.Optional;

import static java.util.Objects.requireNonNull;

/**
 * Builder for creating {@link ExternalCacheSettings} instances.
 *
 * @since 1.0
 */
@PublicApi
public class ExternalCacheSettingsBuilder {
    private Optional<Duration> defaultTtl = Optional.empty();
    private Optional<Integer> entryCountHint = Optional.empty();
    private Optional<ChangeRate> dataChangeRateHint = Optional.empty();
    private Optional<ChangeRate> entryGrowthRateHint = Optional.empty();

    /**
     * Creates an instance with no settings initialised.
     */
    public ExternalCacheSettingsBuilder() {
    }

    /**
     * Constructs an instance initialised with the supplied settings.
     *
     * @param settings the settings to configure the instance with.
     */
    public ExternalCacheSettingsBuilder(ExternalCacheSettings settings) {
        this.defaultTtl = requireNonNull(settings.getDefaultTtl());
        this.entryCountHint = requireNonNull(settings.getEntryCountHint());
        this.dataChangeRateHint = requireNonNull(settings.getDataChangeRateHint());
        this.entryGrowthRateHint = requireNonNull(settings.getEntryGrowthRateHint());
    }

    /**
     * Returns a new {@link ExternalCacheSettings} instance configured using the supplied settings.
     *
     * @return a new {@link ExternalCacheSettings} instance configured using the supplied settings.
     */
    public ExternalCacheSettings build() {
        return new ExternalCacheSettings(
                defaultTtl, entryCountHint, dataChangeRateHint, entryGrowthRateHint);
    }

    /**
     * Specifies the default time-to-live for entries to be held in the cache and must be a positive number.
     * The time-to-live is calculated on the time the entry was added or updated.
     *
     * @param ttl the default time-to-live for entries to be held in the cache
     * @return the builder
     * @throws IllegalArgumentException thrown if <tt>ttl</tt> is not positive
     */
    public ExternalCacheSettingsBuilder defaultTtl(Duration ttl) {
        if (ttl.isNegative() || ttl.isZero()) {
            throw new IllegalArgumentException("ttl must be greater than zero, passed: " + ttl);
        }

        defaultTtl = Optional.of(ttl);
        return this;
    }

    /**
     * Provides a hint on the expected number of entries to be held in the cache. Must not be a negative number.
     * The implementation is free to decide how to interpret <tt>0</tt>. For example, it may decide to still cache
     * values externally, or it may decide to not store entries in the cache.
     *
     * @param hint the expected number of entries to be held in the cache
     * @return the builder
     * @throws IllegalArgumentException thrown if <tt>hint</tt> is negative
     */
    public ExternalCacheSettingsBuilder entryCountHint(int hint) {
        if (hint < 0) {
            throw new IllegalArgumentException("maxEntries cannot be negative, passed: " + hint);
        }

        entryCountHint = Optional.of(hint);
        return this;
    }

    /**
     * Provides a hint on the expected change rate for data updates.
     *
     * @param hint the expected change rate for data updates.
     * @return the builder
     */
    public ExternalCacheSettingsBuilder dataChangeRateHint(ChangeRate hint) {
        dataChangeRateHint = Optional.of(hint);
        return this;
    }

    /**
     * Provides a hint on the expected change rate for entry additions.
     *
     * @param hint the expected change rate for entry additions.
     * @return the builder
     */
    public ExternalCacheSettingsBuilder entryGrowthRateHint(ChangeRate hint) {
        entryGrowthRateHint = Optional.of(hint);
        return this;
    }
}
