/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http.policy;

import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.ExponentialBackoff;
import com.azure.core.http.policy.FixedDelay;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.http.policy.RetryOptions;
import com.azure.core.http.policy.RetryStrategy;
import com.azure.core.implementation.ImplUtils;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.logging.ClientLogger;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Objects;
import java.util.function.Supplier;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class RetryPolicy
implements HttpPipelinePolicy {
    private static final ClientLogger LOGGER = new ClientLogger(RetryPolicy.class);
    private final RetryStrategy retryStrategy;
    private final String retryAfterHeader;
    private final ChronoUnit retryAfterTimeUnit;

    public RetryPolicy() {
        this(new ExponentialBackoff(), null, null);
    }

    public RetryPolicy(String retryAfterHeader, ChronoUnit retryAfterTimeUnit) {
        this(new ExponentialBackoff(), retryAfterHeader, retryAfterTimeUnit);
    }

    public RetryPolicy(RetryStrategy retryStrategy, String retryAfterHeader, ChronoUnit retryAfterTimeUnit) {
        this.retryStrategy = Objects.requireNonNull(retryStrategy, "'retryStrategy' cannot be null.");
        this.retryAfterHeader = retryAfterHeader;
        this.retryAfterTimeUnit = retryAfterTimeUnit;
        if (!CoreUtils.isNullOrEmpty(retryAfterHeader)) {
            Objects.requireNonNull(retryAfterTimeUnit, "'retryAfterTimeUnit' cannot be null.");
        }
    }

    public RetryPolicy(RetryStrategy retryStrategy) {
        this(retryStrategy, null, null);
    }

    public RetryPolicy(RetryOptions retryOptions) {
        this(RetryPolicy.getRetryStrategyFromOptions(Objects.requireNonNull(retryOptions, "'retryOptions' cannot be null.")), null, null);
    }

    private static RetryStrategy getRetryStrategyFromOptions(RetryOptions retryOptions) {
        if (retryOptions.getExponentialBackoffOptions() != null) {
            return new ExponentialBackoff(retryOptions.getExponentialBackoffOptions());
        }
        if (retryOptions.getFixedDelayOptions() != null) {
            return new FixedDelay(retryOptions.getFixedDelayOptions());
        }
        throw new IllegalArgumentException("'retryOptions' didn't define any retry strategy options");
    }

    @Override
    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        return this.attemptAsync(context, next, context.getHttpRequest(), 0);
    }

    private Mono<HttpResponse> attemptAsync(HttpPipelineCallContext context, HttpPipelineNextPolicy next, HttpRequest originalHttpRequest, int tryCount) {
        context.setHttpRequest(originalHttpRequest.copy());
        context.setData("requestRetryCount", tryCount + 1);
        return next.clone().process().flatMap(httpResponse -> {
            if (this.shouldRetry((HttpResponse)httpResponse, tryCount)) {
                Duration delayDuration = RetryPolicy.determineDelayDuration(httpResponse, tryCount, this.retryStrategy, this.retryAfterHeader, this.retryAfterTimeUnit);
                LOGGER.verbose("[Retrying] Try count: {}, Delay duration in seconds: {}", tryCount, delayDuration.getSeconds());
                Flux<ByteBuffer> responseBody = httpResponse.getBody();
                if (responseBody == null) {
                    return this.attemptAsync(context, next, originalHttpRequest, tryCount + 1).delaySubscription(delayDuration);
                }
                return httpResponse.getBody().ignoreElements().then(this.attemptAsync(context, next, originalHttpRequest, tryCount + 1).delaySubscription(delayDuration));
            }
            if (tryCount >= this.retryStrategy.getMaxRetries()) {
                LOGGER.info("Retry attempts have been exhausted after {} attempts.", tryCount);
            }
            return Mono.just((Object)httpResponse);
        }).onErrorResume(err -> {
            if (this.shouldRetryException((Throwable)err, tryCount)) {
                LOGGER.verbose("[Error Resume] Try count: {}, Error: {}", tryCount, err);
                return this.attemptAsync(context, next, originalHttpRequest, tryCount + 1).delaySubscription(this.retryStrategy.calculateRetryDelay(tryCount));
            }
            LOGGER.info("Retry attempts have been exhausted after {} attempts.", tryCount, err);
            return Mono.error((Throwable)err);
        });
    }

    private boolean shouldRetry(HttpResponse response, int tryCount) {
        return tryCount < this.retryStrategy.getMaxRetries() && this.retryStrategy.shouldRetry(response);
    }

    private boolean shouldRetryException(Throwable throwable, int tryCount) {
        return tryCount < this.retryStrategy.getMaxRetries() && this.retryStrategy.shouldRetryException(throwable);
    }

    static Duration determineDelayDuration(HttpResponse response, int tryCount, RetryStrategy retryStrategy, String retryAfterHeader, ChronoUnit retryAfterTimeUnit) {
        if (CoreUtils.isNullOrEmpty(retryAfterHeader)) {
            return RetryPolicy.getWellKnownRetryDelay(response.getHeaders(), tryCount, retryStrategy, OffsetDateTime::now);
        }
        String retryHeaderValue = response.getHeaderValue(retryAfterHeader);
        if (CoreUtils.isNullOrEmpty(retryHeaderValue)) {
            return retryStrategy.calculateRetryDelay(tryCount);
        }
        return Duration.of(Integer.parseInt(retryHeaderValue), retryAfterTimeUnit);
    }

    static Duration getWellKnownRetryDelay(HttpHeaders responseHeaders, int tryCount, RetryStrategy retryStrategy, Supplier<OffsetDateTime> nowSupplier) {
        Duration retryDelay = ImplUtils.getRetryAfterFromHeaders(responseHeaders, nowSupplier);
        if (retryDelay != null) {
            return retryDelay;
        }
        return retryStrategy.calculateRetryDelay(tryCount);
    }
}

