// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.security.keyvault.keys.cryptography;

import com.azure.core.annotation.ServiceClientBuilder;
import com.azure.core.credential.TokenCredential;
import com.azure.core.http.HttpClient;
import com.azure.core.http.HttpPipeline;
import com.azure.core.http.HttpPipelineBuilder;
import com.azure.core.http.policy.HttpLogDetailLevel;
import com.azure.core.http.policy.HttpLogOptions;
import com.azure.core.http.policy.HttpLoggingPolicy;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.http.policy.HttpPolicyProviders;
import com.azure.core.http.policy.RetryPolicy;
import com.azure.core.http.policy.UserAgentPolicy;
import com.azure.core.util.Configuration;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.logging.ClientLogger;
import com.azure.security.keyvault.keys.implementation.KeyVaultCredentialPolicy;
import com.azure.security.keyvault.keys.models.JsonWebKey;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * This class provides a fluent builder API to help aid the configuration and instantiation of the
 * {@link CryptographyAsyncClient} and {@link CryptographyClient}, by calling
 * {@link CryptographyClientBuilder#buildAsyncClient()} and {@link CryptographyClientBuilder#buildClient()} respectively
 * It constructs an instance of the desired client.
 *
 * <p>The minimal configuration options required by {@link CryptographyClientBuilder cryptographyClientBuilder} to build
 * {@link CryptographyAsyncClient} are {@link JsonWebKey jsonWebKey} or {@link String Azure Key Vault key identifier}
 * and {@link TokenCredential credential}.</p>
 *
 * <pre>
 * CryptographyAsyncClient cryptographyAsyncClient = new CryptographyClientBuilder&#40;&#41;
 *     .keyIdentifier&#40;&quot;&lt;your-key-id&gt;&quot;&#41;
 *     .credential&#40;new DefaultAzureCredentialBuilder&#40;&#41;.build&#40;&#41;&#41;
 *     .buildAsyncClient&#40;&#41;;
 * </pre>
 *
 * <p>The {@link HttpLogDetailLevel log detail level}, multiple custom {@link HttpLoggingPolicy policies} and a custom
 * {@link HttpClient http client} can be optionally configured in the {@link CryptographyClientBuilder}.</p>
 *
 * <pre>
 * CryptographyAsyncClient cryptographyAsyncClient = new CryptographyClientBuilder&#40;&#41;
 *     .keyIdentifier&#40;&quot;&lt;your-key-id&gt;&quot;&#41;
 *     .httpLogOptions&#40;new HttpLogOptions&#40;&#41;.setLogLevel&#40;HttpLogDetailLevel.BODY_AND_HEADERS&#41;&#41;
 *     .addPolicy&#40;new KeyVaultCredentialPolicy&#40;new DefaultAzureCredentialBuilder&#40;&#41;.build&#40;&#41;&#41;&#41;
 *     .httpClient&#40;HttpClient.createDefault&#40;&#41;&#41;
 *     .buildAsyncClient&#40;&#41;;
 * </pre>
 *
 * <p>Alternatively, a custom {@link HttpPipeline http pipeline} with custom {@link HttpPipelinePolicy} policies
 * can be specified. It provides finer control over the construction of {@link CryptographyAsyncClient} and
 * {@link CryptographyClient}</p>
 *
 * <pre>
 * HttpPipeline pipeline = new HttpPipelineBuilder&#40;&#41;
 *     .policies&#40;new KeyVaultCredentialPolicy&#40;new DefaultAzureCredentialBuilder&#40;&#41;.build&#40;&#41;&#41;, new RetryPolicy&#40;&#41;&#41;
 *     .build&#40;&#41;;
 * CryptographyAsyncClient cryptographyAsyncClient = new CryptographyClientBuilder&#40;&#41;
 *     .pipeline&#40;pipeline&#41;
 *     .keyIdentifier&#40;&quot;&lt;your-key-id&gt;&quot;&#41;
 *     .buildAsyncClient&#40;&#41;;
 * </pre>
 *
 * <p>The minimal configuration options required by {@link CryptographyClientBuilder cryptographyClientBuilder} to
 * build {@link CryptographyClient} are {@link JsonWebKey jsonWebKey} or
 * {@link String Azure Key Vault key identifier} and {@link TokenCredential credential}.</p>
 *
 * <pre>
 * CryptographyClient cryptographyClient = new CryptographyClientBuilder&#40;&#41;
 *     .keyIdentifier&#40;&quot;&lt;your-key-id&gt;&quot;&#41;
 *     .credential&#40;new DefaultAzureCredentialBuilder&#40;&#41;.build&#40;&#41;&#41;
 *     .buildClient&#40;&#41;;
 * </pre>
 *
 * @see CryptographyAsyncClient
 * @see CryptographyClient
 */
@ServiceClientBuilder(serviceClients = CryptographyClient.class)
public final class CryptographyClientBuilder {
    private final ClientLogger logger = new ClientLogger(CryptographyClientBuilder.class);

    // This is properties file's name.
    private static final String AZURE_KEY_VAULT_KEYS = "azure-key-vault-keys.properties";
    private static final String SDK_NAME = "name";
    private static final String SDK_VERSION = "version";

    private final List<HttpPipelinePolicy> policies;
    private final Map<String, String> properties;

    private Configuration configuration;
    private CryptographyServiceVersion version;
    private HttpClient httpClient;
    private HttpLogOptions httpLogOptions;
    private HttpPipeline pipeline;
    private JsonWebKey jsonWebKey;
    private RetryPolicy retryPolicy;
    private String keyId;
    private TokenCredential credential;

    /**
     * The constructor with defaults.
     */
    public CryptographyClientBuilder() {
        httpLogOptions = new HttpLogOptions();
        policies = new ArrayList<>();
        properties = CoreUtils.getProperties(AZURE_KEY_VAULT_KEYS);
        retryPolicy = new RetryPolicy();
    }

    /**
     * Creates a {@link CryptographyClient} based on options set in the builder. Every time {@code buildClient()} is
     * called, a new instance of {@link CryptographyClient} is created.
     *
     * <p>If {@link CryptographyClientBuilder#pipeline(HttpPipeline) pipeline} is set, then the {@code pipeline} and
     * {@link CryptographyClientBuilder#keyIdentifier(String) jsonWebKey identifier} are used to create the
     * {@link CryptographyClient client}. All other builder settings are ignored. If {@code pipeline} is not set,
     * then an {@link CryptographyClientBuilder#credential(TokenCredential) Azure Key Vault credential} and
     * {@link CryptographyClientBuilder#keyIdentifier(String) JSON Web Key identifier} are required to build the
     * {@link CryptographyClient client}.</p>
     *
     * @return A {@link CryptographyClient} with the options set from the builder.
     *
     * @throws IllegalStateException If {@link CryptographyClientBuilder#credential(TokenCredential)} is {@code null} or
     * {@link CryptographyClientBuilder#keyIdentifier(String)} is empty or {@code null}.
     */
    public CryptographyClient buildClient() {
        return new CryptographyClient(buildAsyncClient());
    }

    /**
     * Creates a {@link CryptographyAsyncClient} based on options set in the builder. Every time
     * {@link #buildAsyncClient()} is called, a new instance of {@link CryptographyAsyncClient} is created.
     *
     * <p>If {@link CryptographyClientBuilder#pipeline(HttpPipeline) pipeline} is set, then the {@code pipeline} and
     * {@link CryptographyClientBuilder#keyIdentifier(String) jsonWebKey identifier}) are used to create the
     * {@link CryptographyAsyncClient async client}. All other builder settings are ignored. If {@code pipeline} is
     * not set, then an {@link CryptographyClientBuilder#credential(TokenCredential) Azure Key Vault credential} and
     * {@link CryptographyClientBuilder#keyIdentifier(String) JSON Web Key identifier} are required to build the
     * {@link CryptographyAsyncClient async client}.</p>
     *
     * @return A {@link CryptographyAsyncClient} with the options set from the builder.
     *
     * @throws IllegalStateException If {@link CryptographyClientBuilder#credential(TokenCredential)} is {@code null} or
     * {@link CryptographyClientBuilder#keyIdentifier(String)} is empty or {@code null}.
     */
    public CryptographyAsyncClient buildAsyncClient() {
        if (jsonWebKey == null && Strings.isNullOrEmpty(keyId)) {
            throw logger.logExceptionAsError(new IllegalStateException(
                "Json Web Key or jsonWebKey identifier are required to create cryptography client"));
        }
        CryptographyServiceVersion serviceVersion = version != null ? version : CryptographyServiceVersion.getLatest();

        if (pipeline != null) {
            if (jsonWebKey != null) {
                return new CryptographyAsyncClient(jsonWebKey, pipeline, serviceVersion);
            } else {
                return new CryptographyAsyncClient(keyId, pipeline, serviceVersion);
            }
        }

        if (credential == null) {
            throw logger.logExceptionAsError(new IllegalStateException(
                "Azure Key Vault credentials are required to build the cryptography client if a JSON Web Key is not"
                    + " provided."));
        }

        HttpPipeline pipeline = setupPipeline();

        if (jsonWebKey != null) {
            return new CryptographyAsyncClient(jsonWebKey, pipeline, serviceVersion);
        } else {
            return new CryptographyAsyncClient(keyId, pipeline, serviceVersion);
        }
    }

    HttpPipeline setupPipeline() {
        Configuration buildConfiguration =
            (configuration == null) ? Configuration.getGlobalConfiguration().clone() : configuration;

        // Closest to API goes first, closest to wire goes last.
        final List<HttpPipelinePolicy> policies = new ArrayList<>();

        String clientName = properties.getOrDefault(SDK_NAME, "UnknownName");
        String clientVersion = properties.getOrDefault(SDK_VERSION, "UnknownVersion");
        policies.add(new UserAgentPolicy(httpLogOptions.getApplicationId(), clientName, clientVersion,
            buildConfiguration));
        HttpPolicyProviders.addBeforeRetryPolicies(policies);
        policies.add(retryPolicy);
        policies.add(new KeyVaultCredentialPolicy(credential));
        policies.addAll(this.policies);
        HttpPolicyProviders.addAfterRetryPolicies(policies);
        policies.add(new HttpLoggingPolicy(httpLogOptions));

        return new HttpPipelineBuilder()
            .policies(policies.toArray(new HttpPipelinePolicy[0]))
            .httpClient(httpClient)
            .build();
    }

    TokenCredential getCredential() {
        return credential;
    }

    HttpPipeline getPipeline() {
        return pipeline;
    }

    CryptographyServiceVersion getServiceVersion() {
        return version;
    }

    /**
     * Sets the Azure Key Vault key identifier of the JSON Web Key to be used for cryptography operations.
     *
     * @param keyId The Azure Key Vault key identifier of the JSON Web Key stored in the key vault.
     *
     * @return The updated {@link CryptographyClientBuilder} object.
     */
    public CryptographyClientBuilder keyIdentifier(String keyId) {
        this.keyId = keyId;

        return this;
    }

    /**
     * Sets the credential to use when authenticating HTTP requests.
     *
     * @param credential The credential to use for authenticating HTTP requests.
     *
     * @return The updated {@link CryptographyClientBuilder} object.
     *
     * @throws NullPointerException If {@code credential} is {@code null}.
     */
    public CryptographyClientBuilder credential(TokenCredential credential) {
        if (credential == null) {
            throw logger.logExceptionAsError(new NullPointerException("'credential' cannot be null."));
        }

        this.credential = credential;

        return this;
    }

    /**
     * Sets the logging configuration for HTTP requests and responses.
     *
     * <p>If {@code logLevel} is not provided, default value of {@link HttpLogDetailLevel#NONE} is set.</p>
     *
     * @param logOptions The logging configuration to use when sending and receiving HTTP requests/responses.
     *
     * @return The updated {@link CryptographyClientBuilder} object.
     */
    public CryptographyClientBuilder httpLogOptions(HttpLogOptions logOptions) {
        httpLogOptions = logOptions;

        return this;
    }

    /**
     * Adds a policy to the set of existing policies that are executed after the client required policies.
     *
     * @param policy The {@link HttpPipelinePolicy policy} to be added.
     *
     * @return The updated {@link CryptographyClientBuilder} object.
     *
     * @throws NullPointerException If {@code policy} is {@code null}.
     */
    public CryptographyClientBuilder addPolicy(HttpPipelinePolicy policy) {
        if (policy == null) {
            throw logger.logExceptionAsError(new NullPointerException("'policy' cannot be null."));
        }

        return this;
    }

    /**
     * Sets the HTTP client to use for sending and receiving requests to and from the service.
     *
     * @param client The HTTP client to use for requests.
     *
     * @return The updated {@link CryptographyClientBuilder} object.
     *
     * @throws NullPointerException If {@code client} is {@code null}.
     */
    public CryptographyClientBuilder httpClient(HttpClient client) {
        if (client == null) {
            throw logger.logExceptionAsError(new NullPointerException("'client' cannot be null."));
        }

        this.httpClient = client;

        return this;
    }

    /**
     * Sets the HTTP pipeline to use for the service client.
     *
     * If {@code pipeline} is set, all other settings are ignored, aside from
     * {@link CryptographyClientBuilder#keyIdentifier(String) JSON Web Key identifier}.
     *
     * @param pipeline The HTTP pipeline to use for sending service requests and receiving responses.
     *
     * @return The updated {@link CryptographyClientBuilder} object.
     *
     * @throws NullPointerException If {@code pipeline} is {@code null}.
     */
    public CryptographyClientBuilder pipeline(HttpPipeline pipeline) {
        if (pipeline == null) {
            throw logger.logExceptionAsError(new NullPointerException("'pipeline' cannot be null."));
        }

        this.pipeline = pipeline;

        return this;
    }

    /**
     * Sets the configuration store that is used during construction of the service client.
     *
     * The default configuration store is a clone of the
     * {@link Configuration#getGlobalConfiguration() global configuration store}, use {@link Configuration#NONE} to
     * bypass using configuration settings during construction.
     *
     * @param configuration The configuration store used to get configuration details.
     *
     * @return The updated {@link CryptographyClientBuilder} object.
     */
    public CryptographyClientBuilder configuration(Configuration configuration) {
        this.configuration = configuration;

        return this;
    }

    /**
     * Sets the {@link CryptographyServiceVersion} that is used when making API requests.
     * <p>
     * If a service version is not provided, the service version that will be used will be the latest known service
     * version based on the version of the client library being used. If no service version is specified, updating to a
     * newer version the client library will have the result of potentially moving to a newer service version.
     *
     * @param version {@link CryptographyServiceVersion} of the service to be used when making requests.
     *
     * @return The updated {@link CryptographyClientBuilder} object.
     */
    public CryptographyClientBuilder serviceVersion(CryptographyServiceVersion version) {
        this.version = version;

        return this;
    }

    /**
     * Sets the {@link RetryPolicy} that is used when each request is sent. The default retry policy will be used in
     * the pipeline, if not provided.
     *
     * @param retryPolicy User's {@link RetryPolicy} applied to each request.
     *
     * @return The updated {@link CryptographyClientBuilder} object.
     *
     * @throws NullPointerException If {@code retryPolicy} is {@code null}.
     */
    public CryptographyClientBuilder retryPolicy(RetryPolicy retryPolicy) {
        if (retryPolicy == null) {
            throw logger.logExceptionAsError(new NullPointerException("'retryPolicy' cannot be null."));
        }

        this.retryPolicy = retryPolicy;

        return this;
    }
}
