package com.atlassian.instrumentation.driver;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.CopyOnWriteArrayList;

import com.atlassian.instrumentation.instruments.Context;

/**
 * This class allows multiple listeners to register to keep notified about JDBC related operations.
 */
public class Instrumentation
{
    /** Represents a timer split. */
    public interface Split
    {
        /** Stops the timer split and records the result. Should only be called once. */
        void stop();
    }

    /**
     * Represents a factory for creating {@link Split}.
     */
    public interface SplitFactory
    {
        /**
         * Creates a split associated with a timer of a specified name.
         * @param context the context of the timer to associate the split time with.
         * @return a started split.
         */
        Split startSplit(Context context);
    }

    private static final CopyOnWriteArrayList<SplitFactory> splitFactories = new CopyOnWriteArrayList<>();

    /**
     * Register a factory for use.
     * @param factory the factory for use.
     */
    public static void registerFactory(SplitFactory factory)
    {
        splitFactories.addIfAbsent(factory);
    }

    /**
     * Unregister a factory.
     * @param factory the factory to unregister. If <tt>null</tt>, then no factory is unregistered.
     */
    public static void unregisterFactory(SplitFactory factory)
    {
        splitFactories.remove(factory);
    }

    /**
     * Create a started split, using the registered factory. If no factory is registered, then a <i>NOP</i>
     * instance is returned instead.
     * @param context the context of the timer to associate the split time with.
     * @return an instance for use. Will never be <tt>null</tt>.
     */
    public static Split startSplit(Context context)
    {
        final int splitFactoryCount = splitFactories.size();
        if (splitFactoryCount == 0)
        {
            return () -> {};
        }
        final Collection<Split> splits = new ArrayList<>(splitFactoryCount);
        for (SplitFactory factory : splitFactories)
        {
            splits.add(factory.startSplit(context));
        }
        return new CombinedSplit(splits);
    }

    private static class CombinedSplit implements Split
    {
        private final Iterable<Split> splits;

        private CombinedSplit(Iterable<Split> splits)
        {
            this.splits = splits;
        }

        @Override
        public void stop()
        {
            splits.forEach(Split::stop);
        }
    }
}
