package com.github.gquintana.metrics.sql;

/*
 * #%L
 * Metrics SQL
 * %%
 * Copyright (C) 2014 Open-Source
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import com.atlassian.instrumentation.instruments.EventType;
import com.atlassian.instrumentation.instruments.Timer;

import javax.sql.PooledConnection;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Optional;

/**
 * Defaut implementation of {@link MetricNamingStrategy}
 */
public class DefaultMetricNamingStrategy implements MetricNamingStrategy {

    /**
     * Generate SQL Id from SQL query.
     * This method can be used to normalize SQL queries, remove special characters,
     * truncate long SQL queries...
     *
     * @return [sql]
     */
    protected String getSqlId(String sql) {
        return "[" + sql.toLowerCase() + "]";
    }

    /*
     * Start Timer for given Class and names
     */
    protected Timer startTimer(Class<?> clazz, String databaseName) {
        Timer timer = new Timer(clazz, databaseName, Optional.empty(), Optional.empty(), Optional.empty());
        timer.start();
        return timer;
    }

    protected Timer startTimer(Class<?> clazz, String databaseName, String sql) {
        Timer timer = new Timer(clazz, databaseName, Optional.of(sql), Optional.empty(), Optional.empty());
        timer.start();
        return timer;
    }

    protected Timer startTimer(Class<?> clazz, String databaseName, String sql, String sqlId) {
        Timer timer = new Timer(clazz, databaseName, Optional.of(sql), Optional.of(sqlId), Optional.empty());
        timer.start();
        return timer;
    }

    protected Timer startTimer(Class<?> clazz, String databaseName, String sql, String sqlId, EventType eventType) {
        Timer timer = new Timer(clazz, databaseName, Optional.of(sql), Optional.of(sqlId), Optional.of(eventType));
        timer.start();
        return timer;
    }

    public Timer startPooledConnectionTimer(String databaseName) {
        return startTimer(PooledConnection.class, databaseName);
    }

    public Timer startConnectionTimer(String databaseName) {
        return startTimer(Connection.class, databaseName);
    }

    public Timer startStatementTimer(String databaseName) {
        return startTimer(Statement.class, databaseName);
    }

    protected StatementTimerContext startStatementTimer(Class<? extends Statement> clazz, String databaseName, String sql, String sqlId) {
        final String lSqlId = sqlId == null ? getSqlId(sql) : sqlId;
        final Timer timerContext = startTimer(clazz, databaseName, sql, lSqlId);
        return new StatementTimerContext(timerContext, sql, lSqlId);
    }

    protected StatementTimerContext startStatementExecuteTimer(Class<? extends Statement> clazz, String databaseName, String sql, String sqlId) {
        final String lSqlId = sqlId == null ? getSqlId(sql) : sqlId;
        final Timer timerContext = startTimer(clazz, databaseName, sql, lSqlId, EventType.EXECUTION);
        return new StatementTimerContext(timerContext, sql, lSqlId);
    }

    public StatementTimerContext startStatementExecuteTimer(String databaseName, String sql) {
        return startStatementExecuteTimer(Statement.class, databaseName, sql, null);
    }

    public StatementTimerContext startPreparedStatementTimer(String databaseName, String sql, String sqlId) {
        return startStatementTimer(PreparedStatement.class, databaseName, sql, sqlId);
    }

    public StatementTimerContext startPreparedStatementExecuteTimer(String databaseName, String sql, String sqlId) {
        return startStatementExecuteTimer(PreparedStatement.class, databaseName, sql, sqlId);
    }

    public StatementTimerContext startCallableStatementTimer(String databaseName, String sql, String sqlId) {
        return startStatementTimer(CallableStatement.class, databaseName, sql, sqlId);
    }

    public StatementTimerContext startCallableStatementExecuteTimer(String databaseName, String sql, String sqlId) {
        return startStatementExecuteTimer(CallableStatement.class, databaseName, sql, sqlId);
    }

    public Timer startResultSetTimer(String databaseName, String sql, String sqlId) {
        return startTimer(ResultSet.class, databaseName, sql);
    }
}
