/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.block.factory.primitive;

import com.gs.collections.api.block.predicate.primitive.BytePredicate;

/**
 * Provides a set of common predicates for byte values.
 * This file was automatically generated from template file primitivePredicates.stg.
 */
public final class BytePredicates
{
    private static final ByteIsEvenPredicate IS_EVEN = new ByteIsEvenPredicate();
    private static final ByteIsOddPredicate IS_ODD = new ByteIsOddPredicate();
    private static final BytePredicate ALWAYS_TRUE = new AlwaysTrueBytePredicate();
    private static final BytePredicate ALWAYS_FALSE = new AlwaysFalseBytePredicate();

    private BytePredicates()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    @SuppressWarnings("MisspelledEquals")
    public static BytePredicate equal(byte expected)
    {
        return new EqualsBytePredicate(expected);
    }

    public static BytePredicate lessThan(byte expected)
    {
        return new LessThanBytePredicate(expected);
    }

    public static BytePredicate greaterThan(byte expected)
    {
        return new GreaterThanBytePredicate(expected);
    }

    public static BytePredicate isEven()
    {
        return IS_EVEN;
    }

    public static BytePredicate isOdd()
    {
        return IS_ODD;
    }


    public static BytePredicate alwaysTrue()
    {
        return ALWAYS_TRUE;
    }

    public static BytePredicate alwaysFalse()
    {
        return ALWAYS_FALSE;
    }

    public static BytePredicate and(BytePredicate one, BytePredicate two)
    {
        return new AndBytePredicate(one, two);
    }

    public static BytePredicate or(BytePredicate one, BytePredicate two)
    {
        return new OrBytePredicate(one, two);
    }

    public static BytePredicate not(BytePredicate negate)
    {
        return new NotBytePredicate(negate);
    }

    private static final class EqualsBytePredicate implements BytePredicate
    {
        private static final long serialVersionUID = 1L;
        private final byte expected;

        private EqualsBytePredicate(byte expected)
        {
            this.expected = expected;
        }

        public boolean accept(byte actual)
        {
            return actual == this.expected;
        }
    }

    private static final class LessThanBytePredicate implements BytePredicate
    {
        private static final long serialVersionUID = 1L;

        private final byte expected;

        private LessThanBytePredicate(byte expected)
        {
            this.expected = expected;
        }

        public boolean accept(byte actual)
        {
            return actual < this.expected;
        }
    }

    private static final class GreaterThanBytePredicate implements BytePredicate
    {
        private static final long serialVersionUID = 1L;

        private final byte expected;

        private GreaterThanBytePredicate(byte expected)
        {
            this.expected = expected;
        }

        public boolean accept(byte actual)
        {
            return actual > this.expected;
        }
    }

    private static final class AndBytePredicate implements BytePredicate
    {
        private static final long serialVersionUID = 1L;

        private final BytePredicate one;
        private final BytePredicate two;

        private AndBytePredicate(BytePredicate one, BytePredicate two)
        {
            this.one = one;
            this.two = two;
        }

        public boolean accept(byte actual)
        {
            return this.one.accept(actual) && this.two.accept(actual);
        }
    }

    private static final class OrBytePredicate implements BytePredicate
    {
        private static final long serialVersionUID = 1L;

        private final BytePredicate one;
        private final BytePredicate two;

        private OrBytePredicate(BytePredicate one, BytePredicate two)
        {
            this.one = one;
            this.two = two;
        }

        public boolean accept(byte actual)
        {
            return this.one.accept(actual) || this.two.accept(actual);
        }
    }

    private static final class NotBytePredicate implements BytePredicate
    {
        private static final long serialVersionUID = 1L;

        private final BytePredicate negate;

        private NotBytePredicate(BytePredicate negate)
        {
            this.negate = negate;
        }

        public boolean accept(byte actual)
        {
            return !this.negate.accept(actual);
        }
    }

    private static final class ByteIsEvenPredicate implements BytePredicate
    {
        private static final long serialVersionUID = 1L;

        public boolean accept(byte integral)
        {
            return integral % 2 == 0;
        }
    }

    private static final class ByteIsOddPredicate implements BytePredicate
    {
        private static final long serialVersionUID = 1L;

        public boolean accept(byte integral)
        {
            return integral % 2 != 0;
        }
    }

    private static final class AlwaysTrueBytePredicate implements BytePredicate
    {
        private static final long serialVersionUID = 1L;

        public boolean accept(byte value)
        {
            return true;
        }
    }

    private static final class AlwaysFalseBytePredicate implements BytePredicate
    {
        private static final long serialVersionUID = 1L;

        public boolean accept(byte value)
        {
            return false;
        }
    }
}
