/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.lazy.primitive;

import java.io.IOException;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.DoubleIterable;
import com.gs.collections.api.LazyDoubleIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectDoubleToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.api.block.procedure.primitive.DoubleProcedure;
import com.gs.collections.api.iterator.DoubleIterator;
import com.gs.collections.api.bag.primitive.MutableDoubleBag;
import com.gs.collections.api.list.primitive.MutableDoubleList;
import com.gs.collections.api.set.primitive.MutableDoubleSet;
import com.gs.collections.impl.bag.mutable.primitive.DoubleHashBag;
import com.gs.collections.impl.set.mutable.primitive.DoubleHashSet;
import com.gs.collections.impl.list.mutable.primitive.DoubleArrayList;
import com.gs.collections.impl.block.factory.primitive.DoublePredicates;

/**
 * This file was automatically generated from template file selectPrimitiveIterable.stg.
 */
public class SelectDoubleIterable
    extends AbstractLazyDoubleIterable
{
    private final DoubleIterable delegate;
    private final DoublePredicate predicate;

    public SelectDoubleIterable(DoubleIterable delegate, DoublePredicate predicate)
    {
        this.delegate = delegate;
        this.predicate = predicate;
    }

    public DoubleIterator doubleIterator()
    {
        return new SelectDoubleIterator(this.delegate, this.predicate);
    }

    public void forEach(DoubleProcedure procedure)
    {
        this.delegate.forEach(new IfDoubleProcedure(procedure));
    }

    @Override
    public int size()
    {
        return this.delegate.count(this.predicate);
    }

    @Override
    public boolean isEmpty()
    {
        return !this.doubleIterator().hasNext();
    }

    @Override
    public boolean notEmpty()
    {
        return this.doubleIterator().hasNext();
    }

    @Override
    public int count(DoublePredicate predicate)
    {
        CountDoubleProcedure countDoubleProcedure = new CountDoubleProcedure(predicate);
        this.forEach(countDoubleProcedure);
        return countDoubleProcedure.getCount();
    }

    @Override
    public boolean anySatisfy(DoublePredicate predicate)
    {
        return this.delegate.anySatisfy(DoublePredicates.and(this.predicate, predicate));
    }

    @Override
    public boolean allSatisfy(DoublePredicate predicate)
    {
        return this.noneSatisfy(DoublePredicates.not(predicate));
    }

    @Override
    public boolean noneSatisfy(DoublePredicate predicate)
    {
        return !this.anySatisfy(predicate);
    }

        public double sum()
        {
            double sum = 0.0;
            for (DoubleIterator doubleIterator = this.doubleIterator(); doubleIterator.hasNext() ;)
            {
                sum += doubleIterator.next();
            }
            return sum;
        }

        public double max()
        {
            DoubleIterator doubleIterator = this.doubleIterator();
            double max = doubleIterator.next();
            while (doubleIterator.hasNext())
            {
                max = (double) Math.max(max, doubleIterator.next());
            }
            return max;
        }

        public double min()
        {
            DoubleIterator doubleIterator = this.doubleIterator();
            double min = doubleIterator.next();
            while (doubleIterator.hasNext())
            {
                min = (double) Math.min(min, doubleIterator.next());
            }
            return min;
        }

        public double minIfEmpty(double defaultValue)
        {
            try
            {
                return this.min();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public double maxIfEmpty(double defaultValue)
        {
            try
            {
                return this.max();
            }
            catch (NoSuchElementException ex)
            {
            }
            return defaultValue;
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            double[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                double first = sortedArray[middleIndex];
                double second = sortedArray[middleIndex - 1];
                return (first + second) / 2.0;
            }
            return sortedArray[middleIndex];
        }

        public double[] toSortedArray()
        {
            double[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableDoubleList toSortedList()
        {
            return DoubleArrayList.newList(this).sortThis();
        }

    @Override
    public double[] toArray()
    {
        final double[] array = new double[this.size()];
        this.forEach(new DoubleProcedure()
        {
            @SuppressWarnings("FieldMayBeFinal")
            private int index = 0;
            public void value(double each)
            {
                array[this.index++] = each;
            }
        });
        return array;
    }

    @Override
    public boolean containsAll(double... source)
    {
        for (double value : source)
        {
            if (!this.contains(value))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean containsAll(DoubleIterable source)
    {
        for (DoubleIterator iterator = source.doubleIterator(); iterator.hasNext(); )
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public MutableDoubleList toList()
    {
        return DoubleArrayList.newList(this);
    }

    @Override
    public MutableDoubleSet toSet()
    {
        return DoubleHashSet.newSet(this);
    }

    @Override
    public MutableDoubleBag toBag()
    {
        return DoubleHashBag.newBag(this);
    }

    private static final class CountDoubleProcedure implements DoubleProcedure
    {
        private static final long serialVersionUID = 1L;
        private final DoublePredicate predicate;
        private int counter = 0;

        private CountDoubleProcedure(DoublePredicate predicate)
        {
            this.predicate = predicate;
        }

        public void value(double each)
        {
            if (this.predicate.accept(each))
            {
                this.counter++;
            }
        }

        public int getCount()
        {
            return this.counter;
        }
    }

    private final class IfDoubleProcedure implements DoubleProcedure
    {
        private static final long serialVersionUID = 1L;
        private final DoubleProcedure procedure;

        private IfDoubleProcedure(DoubleProcedure procedure)
        {
            this.procedure = procedure;
        }

        public void value(double each)
        {
            if (SelectDoubleIterable.this.predicate.accept(each))
            {
                this.procedure.value(each);
            }
        }
    }

    private static final class SelectDoubleIterator
            implements DoubleIterator
    {
        private final DoubleIterator iterator;
        private final DoublePredicate predicate;
        private double next;
        private boolean verifiedHasNext = false;

        private SelectDoubleIterator(DoubleIterable iterable, DoublePredicate predicate)
        {
            this(iterable.doubleIterator(), predicate);
        }

        private SelectDoubleIterator(DoubleIterator iterator, DoublePredicate predicate)
        {
            this.iterator = iterator;
            this.predicate = predicate;
        }

        public boolean hasNext()
        {
            if (this.verifiedHasNext)
            {
                return true;
            }
            while (this.iterator.hasNext())
            {
                double temp = this.iterator.next();
                if (this.predicate.accept(temp))
                {
                    this.next = temp;
                    this.verifiedHasNext = true;
                    return true;
                }
            }
            return false;
        }

        public double next()
        {
            if (this.verifiedHasNext || this.hasNext())
            {
                this.verifiedHasNext = false;
                return this.next;
            }
            throw new NoSuchElementException();
        }
    }
}
