/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.list.immutable.primitive;

import java.io.IOException;
import java.io.Serializable;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.ByteIterable;
import com.gs.collections.api.LazyByteIterable;
import com.gs.collections.api.bag.primitive.MutableByteBag;
import com.gs.collections.api.block.function.primitive.ByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectByteIntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BytePredicate;
import com.gs.collections.api.block.procedure.primitive.ByteProcedure;
import com.gs.collections.api.block.procedure.primitive.ByteIntProcedure;
import com.gs.collections.api.iterator.ByteIterator;
import com.gs.collections.api.list.ImmutableList;
import com.gs.collections.api.list.primitive.ImmutableByteList;
import com.gs.collections.api.list.primitive.ByteList;
import com.gs.collections.api.list.primitive.MutableByteList;
import com.gs.collections.api.set.primitive.MutableByteSet;
import com.gs.collections.impl.bag.mutable.primitive.ByteHashBag;
import com.gs.collections.impl.factory.primitive.ByteLists;
import com.gs.collections.impl.lazy.primitive.LazyByteIterableAdapter;
import com.gs.collections.impl.lazy.primitive.ReverseByteIterable;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.ByteArrayList;
import com.gs.collections.impl.set.mutable.primitive.ByteHashSet;
import net.jcip.annotations.Immutable;

/**
 * ImmutableByteArrayList is the non-modifiable equivalent of {@link ByteArrayList}.
 * It wraps a Java byte array.
 * This file was automatically generated from template file immutablePrimitiveArrayList.stg.
 *
 * @since 3.2.
 */
@Immutable
final class ImmutableByteArrayList
        implements ImmutableByteList, Serializable
{
    private static final long serialVersionUID = 1L;
    private final byte[] items;

    private ImmutableByteArrayList(byte[] newElements)
    {
        if (newElements.length <= 1)
        {
            throw new IllegalArgumentException("Use ByteLists.immutable.with() to instantiate an optimized collection");
        }
        this.items = newElements;
    }

    public static ImmutableByteArrayList newList(ByteIterable iterable)
    {
        return new ImmutableByteArrayList(iterable.toArray());
    }

    public static ImmutableByteArrayList newListWith(byte... elements)
    {
        byte[] newArray = new byte[elements.length];
        System.arraycopy(elements, 0, newArray, 0, elements.length);
        return new ImmutableByteArrayList(newArray);
    }

    public byte get(int index)
    {
        return this.items[index];
    }

    public byte getFirst()
    {
        return this.items[0];
    }

    public byte getLast()
    {
        return this.items[this.items.length - 1];
    }

    public int indexOf(byte value)
    {
        for (int i = 0; i < this.items.length; i++)
        {
            if (this.items[i] == value)
            {
                return i;
            }
        }
        return -1;
    }

    public int lastIndexOf(byte value)
    {
        for (int i = this.items.length - 1; i >= 0; i--)
        {
            if (this.items[i] == value)
            {
                return i;
            }
        }
        return -1;
    }

    public ByteIterator byteIterator()
    {
        return new InternalByteIterator();
    }

    public void forEach(ByteProcedure procedure)
    {
        for (byte item : this.items)
        {
            procedure.value(item);
        }
    }

    public void forEachWithIndex(ByteIntProcedure procedure)
    {
        for (int i = 0; i < this.items.length; i++)
        {
            procedure.value(this.items[i], i);
        }
    }

    public int count(BytePredicate predicate)
    {
        int count = 0;
        for (byte item : this.items)
        {
            if (predicate.accept(item))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(BytePredicate predicate)
    {
        for (byte item : this.items)
        {
            if (predicate.accept(item))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(BytePredicate predicate)
    {
        for (byte item : this.items)
        {
            if (!predicate.accept(item))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(BytePredicate predicate)
    {
        for (byte item : this.items)
        {
            if (predicate.accept(item))
            {
                return false;
            }
        }
        return true;
    }

    public ImmutableByteList select(BytePredicate predicate)
    {
        ByteArrayList result = new ByteArrayList();
        for (byte item : this.items)
        {
            if (predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result.toImmutable();
    }

    public ImmutableByteList reject(BytePredicate predicate)
    {
        ByteArrayList result = new ByteArrayList();
        for (byte item : this.items)
        {
            if (!predicate.accept(item))
            {
                result.add(item);
            }
        }
        return result.toImmutable();
    }

    public byte detectIfNone(BytePredicate predicate, byte ifNone)
    {
        for (byte item : this.items)
        {
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public <V> ImmutableList<V> collect(ByteToObjectFunction<? extends V> function)
    {
        FastList<V> target = FastList.newList(this.items.length);
        for (byte item : this.items)
        {
            target.add(function.valueOf(item));
        }
        return target.toImmutable();
    }

    public long sum()
    {
        long result = 0L;
        for (byte item : this.items)
        {
            result += item;
        }
        return result;
    }

    public byte max()
    {
        byte max = this.items[0];
        for (int i = 1; i < this.items.length; i++)
        {
            byte value = this.items[i];
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public byte maxIfEmpty(byte defaultValue)
    {
        return this.max();
    }

    public byte min()
    {
        byte min = this.items[0];
        for (int i = 1; i < this.items.length; i++)
        {
            byte value = this.items[i];
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public byte minIfEmpty(byte defaultValue)
    {
        return this.min();
    }

    public double average()
    {
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        byte[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            byte first = sortedArray[middleIndex];
            byte second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public byte[] toSortedArray()
    {
        byte[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public long dotProduct(ByteList list)
    {
        if (this.size() != list.size())
        {
            throw new IllegalArgumentException("Lists used in dotProduct must be the same size");
        }
        long sum = 0L;
        for (int i = 0; i < this.size(); i++)
        {
            sum += (long) this.items[i] * list.get(i);
        }
        return sum;
    }

    public LazyByteIterable asReversed()
    {
        return ReverseByteIterable.adapt(this);
    }

    public MutableByteList toSortedList()
    {
        return ByteArrayList.newList(this).sortThis();
    }

    public byte[] toArray()
    {
        byte[] newItems = new byte[this.items.length];
        System.arraycopy(this.items, 0, newItems, 0, this.items.length);
        return newItems;
    }

    public boolean contains(byte value)
    {
        for (byte item : this.items)
        {
            if (item == value)
            {
                return true;
            }
        }
        return false;
    }

    public boolean containsAll(byte... source)
    {
        for (byte value : source)
        {
            if (!this.contains(value))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(ByteIterable source)
    {
        for (ByteIterator iterator = source.byteIterator(); iterator.hasNext(); )
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public MutableByteList toList()
    {
        return ByteArrayList.newList(this);
    }

    public MutableByteSet toSet()
    {
        return ByteHashSet.newSet(this);
    }

    public MutableByteBag toBag()
    {
        return ByteHashBag.newBag(this);
    }

    public LazyByteIterable asLazy()
    {
        return new LazyByteIterableAdapter(this);
    }

    public ImmutableByteList toImmutable()
    {
        return this;
    }

    public ImmutableByteArrayList toReversed()
    {
        return ImmutableByteArrayList.newList(this.asReversed());
    }

    public ImmutableByteList newWith(byte element)
    {
        byte[] newItems = new byte[this.items.length + 1];
        System.arraycopy(this.items, 0, newItems, 0, this.items.length);
        newItems[this.items.length] = element;
        return new ImmutableByteArrayList(newItems);
    }

    public ImmutableByteList newWithout(byte element)
    {
        int index = this.indexOf(element);
        if (index != -1)
        {
            byte[] newItems = new byte[this.items.length - 1];
            System.arraycopy(this.items, 0, newItems, 0, index);
            System.arraycopy(this.items, index + 1, newItems, index, this.items.length - index - 1);
            return ByteLists.immutable.with(newItems);
        }
        return this;
    }

    public ImmutableByteList newWithAll(ByteIterable elements)
    {
        byte[] newItems = new byte[this.items.length + elements.size()];
        System.arraycopy(this.items, 0, newItems, 0, this.items.length);
        int index = 0;
        for (ByteIterator iterator = elements.byteIterator(); iterator.hasNext(); index++)
        {
            newItems[this.items.length + index] = iterator.next();
        }
        return new ImmutableByteArrayList(newItems);
    }

    public ImmutableByteList newWithoutAll(ByteIterable elements)
    {
        MutableByteList mutableByteList = this.toList();
        mutableByteList.removeAll(elements);
        return mutableByteList.toImmutable();
    }

    public int size()
    {
        return this.items.length;
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    public <T> T injectInto(T injectedValue, ObjectByteToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.items.length; i++)
        {
            result = function.valueOf(result, this.items[i]);
        }
        return result;
    }

    public <T> T injectIntoWithIndex(T injectedValue, ObjectByteIntToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.items.length; i++)
        {
            result = function.valueOf(result, this.items[i], i);
        }
        return result;
    }

    @Override
    public boolean equals(Object otherList)
    {
        if (otherList == this)
        {
            return true;
        }
        if (!(otherList instanceof ByteList))
        {
            return false;
        }
        ByteList list = (ByteList) otherList;
        if (this.items.length != list.size())
        {
            return false;
        }
        for (int i = 0; i < this.items.length; i++)
        {
            if (this.items[i] != list.get(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        for (byte item : this.items)
        {
            hashCode = 31 * hashCode + (int) item;
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            for (int i = 0; i < this.items.length; i++)
            {
                if (i > 0)
                {
                    appendable.append(separator);
                }
                byte value = this.items[i];
                appendable.append(String.valueOf(value));
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public ImmutableByteList subList(int fromIndex, int toIndex)
    {
        throw new UnsupportedOperationException("subList not yet implemented!");
    }

    private class InternalByteIterator implements ByteIterator
    {
        /**
         * Index of element to be returned by subsequent call to next.
         */
        private int currentIndex;

        public boolean hasNext()
        {
            return this.currentIndex != ImmutableByteArrayList.this.items.length;
        }

        public byte next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }
            byte next = ImmutableByteArrayList.this.items[this.currentIndex];
            this.currentIndex++;
            return next;
        }
    }
}
