/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;
import java.util.Arrays;
import java.util.NoSuchElementException;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.bag.primitive.MutableIntBag;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectIntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.block.procedure.primitive.IntProcedure;
import com.gs.collections.api.iterator.IntIterator;
import com.gs.collections.api.list.primitive.MutableIntList;
import com.gs.collections.api.set.ImmutableSet;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ImmutableIntSet;
import com.gs.collections.api.set.primitive.MutableIntSet;
import com.gs.collections.api.set.primitive.IntSet;
import com.gs.collections.impl.bag.mutable.primitive.IntHashBag;
import com.gs.collections.impl.factory.primitive.IntSets;
import com.gs.collections.impl.lazy.primitive.LazyIntIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.IntArrayList;
import com.gs.collections.impl.set.immutable.primitive.ImmutableIntSetSerializationProxy;
import com.gs.collections.impl.set.mutable.UnifiedSet;

/**
 * This file was automatically generated from template file primitiveHashSet.stg.
 *
 * @since 3.0.
 */
public final class IntHashSet implements MutableIntSet, Externalizable
{
    private static final long serialVersionUID = 1L;
    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;
    private static final int EMPTY = 0;
    private static final int REMOVED = 1;

    private int[] table;
    private int occupiedWithData;
    private int occupiedWithSentinels;
    // The 32 bits of this integer indicate whether the items 0 to 31 are present in the set.
    private int zeroToThirtyOne;
    private int zeroToThirtyOneOccupied;
    private transient boolean copyOnWrite;

    public IntHashSet()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public IntHashSet(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    public IntHashSet(IntHashSet set)
    {
        this.occupiedWithData = set.occupiedWithData;
        this.occupiedWithSentinels = set.occupiedWithSentinels;
        this.zeroToThirtyOneOccupied = set.zeroToThirtyOneOccupied;
        this.zeroToThirtyOne = set.zeroToThirtyOne;
        this.allocateTable(set.table.length);

        System.arraycopy(set.table, 0, this.table, 0, set.table.length);
    }

    public static IntHashSet newSet(IntIterable source)
    {
        if (source instanceof IntHashSet)
        {
            return new IntHashSet((IntHashSet) source);
        }

        return IntHashSet.newSetWith(source.toArray());
    }

    public static IntHashSet newSetWith(int... source)
    {
        IntHashSet result = new IntHashSet();
        result.addAll(source);
        return result;
    }

    private static boolean isBetweenZeroAndThirtyOne(int value)
    {
        return value >= 0 && value <= 31;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof IntSet))
        {
            return false;
        }

        IntSet other = (IntSet) obj;
        return this.size() == other.size() && this.containsAll(other.toArray());
    }

    @Override
    public int hashCode()
    {
        int result = 0;
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            result += value;
            zeroToThirtyOne &= ~(1 << value);
        }
        if (this.table != null)
        {
            for (int i = 0; i < this.table.length; i++)
            {
                if (isNonSentinel(this.table[i]))
                {
                    result += this.table[i];
                }
            }
        }
        return result;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public int size()
    {
        return this.occupiedWithData + this.zeroToThirtyOneOccupied;
    }

    public boolean isEmpty()
    {
        return this.size() == 0;
    }

    public boolean notEmpty()
    {
        return this.size() != 0;
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            int count = 0;
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (count > 0)
                {
                    appendable.append(separator);
                }
                count++;
                appendable.append(String.valueOf(value));
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (count > 0)
                    {
                        appendable.append(separator);
                    }
                    count++;
                    appendable.append(String.valueOf(value));
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public boolean add(int element)
    {
        if (isBetweenZeroAndThirtyOne(element))
        {
            int initial = this.zeroToThirtyOne;
            this.zeroToThirtyOne |= 1 << element;
            if (this.zeroToThirtyOne != initial)
            {
                this.zeroToThirtyOneOccupied++;
                return true;
            }
            return false;
        }

        int index = this.probe(element);

        if (this.table[index] == element)
        {
            // element already present in set
            return false;
        }

        if (this.copyOnWrite)
        {
            this.copyTable();
        }
        if (this.table[index] == REMOVED)
        {
            --this.occupiedWithSentinels;
        }
        this.table[index] = element;
        ++this.occupiedWithData;
        if (this.occupiedWithData > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
        return true;
    }

    public boolean addAll(int... source)
    {
        int oldSize = this.size();
        for (int item : source)
        {
            this.add(item);
        }
        return this.size() != oldSize;
    }

    public boolean addAll(IntIterable source)
    {
        if (source.isEmpty())
        {
            return false;
        }
        int oldSize = this.size();
        if (source instanceof IntHashSet)
        {
            IntHashSet hashSet = (IntHashSet) source;
            this.zeroToThirtyOne |= hashSet.zeroToThirtyOne;
            this.zeroToThirtyOneOccupied = Integer.bitCount(this.zeroToThirtyOne);
            for (int item : hashSet.table)
            {
                if (isNonSentinel(item))
                {
                    this.add(item);
                }
            }
        }
        else
        {
            IntIterator iterator = source.intIterator();
            while (iterator.hasNext())
            {
                int item = iterator.next();
                this.add(item);
            }
        }
        return this.size() != oldSize;
    }

    public boolean remove(int value)
    {
        if (isBetweenZeroAndThirtyOne(value))
        {
            int initial = this.zeroToThirtyOne;
            this.zeroToThirtyOne &= ~(1 << value);
            if (this.zeroToThirtyOne == initial)
            {
                return false;
            }
            this.zeroToThirtyOneOccupied--;
            return true;
        }
        int index = this.probe(value);
        if (this.table[index] == value)
        {
            if (this.copyOnWrite)
            {
                this.copyTable();
            }
            this.table[index] = REMOVED;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
            if (this.occupiedWithSentinels > this.maxOccupiedWithSentinels())
            {
                this.rehash();
            }

            return true;
        }
        return false;
    }

    public boolean removeAll(IntIterable source)
    {
        if (source.isEmpty())
        {
            return false;
        }
        int oldSize = this.size();
        if (source instanceof IntHashSet)
        {
            IntHashSet hashSet = (IntHashSet) source;
            this.zeroToThirtyOne &= ~hashSet.zeroToThirtyOne;
            this.zeroToThirtyOneOccupied = Integer.bitCount(this.zeroToThirtyOne);
            for (int item : hashSet.table)
            {
                if (isNonSentinel(item))
                {
                    this.remove(item);
                }
            }
        }
        else
        {
            IntIterator iterator = source.intIterator();
            while (iterator.hasNext())
            {
                int item = iterator.next();
                this.remove(item);
            }
        }
        return this.size() != oldSize;
    }

    public boolean removeAll(int... source)
    {
        if (source.length == 0)
        {
            return false;
        }
        int oldSize = this.size();
        for (int item : source)
        {
            this.remove(item);
        }
        return this.size() != oldSize;
    }

    public boolean retainAll(IntIterable source)
    {
        int oldSize = this.size();
        final IntSet sourceSet = source instanceof IntSet ? (IntSet) source : source.toSet();
        IntHashSet retained = this.select(new IntPredicate()
        {
            public boolean accept(int value)
            {
                return sourceSet.contains(value);
            }
        });
        if (retained.size() != oldSize)
        {
            this.zeroToThirtyOne = retained.zeroToThirtyOne;
            this.zeroToThirtyOneOccupied = retained.zeroToThirtyOneOccupied;
            this.occupiedWithData = retained.occupiedWithData;
            this.occupiedWithSentinels = retained.occupiedWithSentinels;
            this.table = retained.table;
            return true;
        }
        return false;
    }

    public boolean retainAll(int... source)
    {
        return this.retainAll(IntHashSet.newSetWith(source));
    }

    public void clear()
    {
        this.zeroToThirtyOneOccupied = 0;
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        this.zeroToThirtyOne = 0;
        if (this.copyOnWrite)
        {
            this.table = new int[this.table.length];
            this.copyOnWrite = false;
        }
        else
        {
            Arrays.fill(this.table, EMPTY);
        }
    }

    public IntHashSet with(int element)
    {
        this.add(element);
        return this;
    }

    public IntHashSet without(int element)
    {
        this.remove(element);
        return this;
    }

    public IntHashSet withAll(IntIterable elements)
    {
        this.addAll(elements.toArray());
        return this;
    }

    public IntHashSet withoutAll(IntIterable elements)
    {
        this.removeAll(elements);
        return this;
    }

    public MutableIntSet asUnmodifiable()
    {
        return new UnmodifiableIntSet(this);
    }

    public MutableIntSet asSynchronized()
    {
        return new SynchronizedIntSet(this);
    }

    public ImmutableIntSet toImmutable()
    {
        if (this.size() == 0)
        {
            return IntSets.immutable.with();
        }
        if (this.size() == 1)
        {
            return IntSets.immutable.with(this.intIterator().next());
        }
        IntHashSet mutableSet = IntHashSet.newSetWith(this.toArray());
        return new ImmutableIntHashSet(mutableSet.table, mutableSet.occupiedWithData, mutableSet.zeroToThirtyOne, mutableSet.zeroToThirtyOneOccupied);
    }

    public IntIterator intIterator()
    {
        return new InternalIntIterator();
    }

    public int[] toArray()
    {
        int[] array = new int[this.size()];

        int j = 0;
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            array[j] = value;
            j++;
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int i = 0; i < this.table.length && j < this.size(); i++)
        {
            if (isNonSentinel(this.table[i]))
            {
                array[j] = this.table[i];
                j++;
            }
        }
        return array;
    }

    public boolean contains(int value)
    {
        if (isBetweenZeroAndThirtyOne(value))
        {
            int temp = this.zeroToThirtyOne;
            return ((temp >>> value) & 1) != 0;
        }
        return this.table[this.probe(value)] == value;
    }

    public boolean containsAll(int... source)
    {
        for (int item : source)
        {
            if (!this.contains(item))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(IntIterable source)
    {
        for (IntIterator iterator = source.intIterator(); iterator.hasNext();)
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public void forEach(IntProcedure procedure)
    {
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            procedure.value(value);
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                procedure.value(value);
            }
        }
    }

    public IntHashSet select(IntPredicate predicate)
    {
        IntHashSet result = new IntHashSet();

        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (predicate.accept(value))
            {
                result.add(value);
            }
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (predicate.accept(value))
                {
                    result.add(value);
                }
            }
        }
        return result;
    }

    public MutableIntSet reject(IntPredicate predicate)
    {
        IntHashSet result = new IntHashSet();

        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (!predicate.accept(value))
            {
                result.add(value);
            }
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (!predicate.accept(value))
                {
                    result.add(value);
                }
            }
        }
        return result;
    }

    public <V> MutableSet<V> collect(IntToObjectFunction<? extends V> function)
    {
        MutableSet<V> target = UnifiedSet.newSet(this.size());

        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            target.add(function.valueOf(value));
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                target.add(function.valueOf(value));
            }
        }
        return target;
    }

    public int detectIfNone(IntPredicate predicate, int ifNone)
    {
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (predicate.accept(value))
            {
                return value;
            }
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (predicate.accept(value))
                {
                    return value;
                }
            }
        }
        return ifNone;
    }

    public int count(IntPredicate predicate)
    {
        int count = 0;
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (predicate.accept(value))
            {
                count++;
            }
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (predicate.accept(value))
                {
                    count++;
                }
            }
        }
        return count;
    }

    public boolean anySatisfy(IntPredicate predicate)
    {
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (predicate.accept(value))
            {
                return true;
            }
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (predicate.accept(value))
                {
                    return true;
                }
            }
        }
        return false;
    }

    public boolean allSatisfy(IntPredicate predicate)
    {
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (!predicate.accept(value))
            {
                return false;
            }
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (!predicate.accept(value))
                {
                    return false;
                }
            }
        }
        return true;
    }

    public boolean noneSatisfy(IntPredicate predicate)
    {
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            if (predicate.accept(value))
            {
                return false;
            }
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                if (predicate.accept(value))
                {
                    return false;
                }
            }
        }
        return true;
    }

    public MutableIntList toList()
    {
        return IntArrayList.newList(this);
    }

    public MutableIntSet toSet()
    {
        return IntHashSet.newSet(this);
    }

    public MutableIntBag toBag()
    {
        return IntHashBag.newBag(this);
    }

    public LazyIntIterable asLazy()
    {
        return new LazyIntIterableAdapter(this);
    }

    public long sum()
    {
        long result = 0L;

        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            result += value;
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                result += value;
            }
        }
        return result;
    }

    public int max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        int max = 31 - Integer.numberOfLeadingZeros(this.zeroToThirtyOne);
        boolean isMaxSet = this.zeroToThirtyOneOccupied != 0;

        for (int value : this.table)
        {
            if (isNonSentinel(value) && (!isMaxSet || max < value))
            {
                max = value;
                isMaxSet = true;
            }
        }
        return max;
    }

    public int maxIfEmpty(int defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public int min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        int min = Integer.numberOfTrailingZeros(this.zeroToThirtyOne);
        boolean isMinSet = this.zeroToThirtyOneOccupied != 0;

        for (int value : this.table)
        {
            if (isNonSentinel(value) && (!isMinSet || value < min))
            {
                min = value;
                isMinSet = true;
            }
        }
        return min;
    }

    public int minIfEmpty(int defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        int[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            int first = sortedArray[middleIndex];
            int second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public int[] toSortedArray()
    {
        int[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableIntList toSortedList()
    {
        return IntArrayList.newList(this).sortThis();
    }

    public IntSet freeze()
    {
        if (this.size() == 0)
        {
            return IntSets.immutable.with();
        }
        if (this.size() == 1)
        {
            return IntSets.immutable.with(this.intIterator().next());
        }
        this.copyOnWrite = true;
        return new ImmutableIntHashSet(this.table, this.occupiedWithData, this.zeroToThirtyOne, this.zeroToThirtyOneOccupied);
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());

        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            out.writeInt(value);
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                out.writeInt(value);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        int size = in.readInt();

        for (int i = 0; i < size; i++)
        {
            this.add(in.readInt());
        }
    }

    public <T> T injectInto(T injectedValue, ObjectIntToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        int zeroToThirtyOne = this.zeroToThirtyOne;
        while (zeroToThirtyOne != 0)
        {
            int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
            result = function.valueOf(result, value);
            zeroToThirtyOne &= ~(1 << value);
        }

        for (int value : this.table)
        {
            if (isNonSentinel(value))
            {
                result = function.valueOf(result, value);
            }
        }
        return result;
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehash()
    {
        this.rehash(this.table.length);
    }

    private void rehashAndGrow()
    {
        this.rehash(this.table.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.table.length;
        int[] old = this.table;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.add(old[i]);
            }
        }
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.table = new int[sizeToAllocate];
    }

    // exposed for testing
    int probe(int element)
    {
        int index = this.spread(element);
        int valueAtIndex = this.table[index];

        if (valueAtIndex == element || valueAtIndex == EMPTY)
        {
            return index;
        }

        int removedIndex = valueAtIndex == REMOVED ? index : -1;
        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17 * n * (n+1) / 2 where n = number of collisions
            nextIndex = (nextIndex + probe) & this.table.length - 1;
            probe += 17;

            if (this.table[nextIndex] == element)
            {
                return nextIndex;
            }
            if (this.table[nextIndex] == REMOVED)
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (this.table[nextIndex] == EMPTY)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(int element)
    {
        int code = element;
        code ^= 61 ^ (code >> 16);
        code += code << 3;
        code ^= code >> 4;
        code *= 0x27d4eb2d;
        code ^= code >> 15;
        return code & (this.table.length - 1);
    }

    private void copyTable()
    {
        this.copyOnWrite = false;
        int[] copy = new int[this.table.length];
        System.arraycopy(this.table, 0, copy, 0,
        Math.min(this.table.length, this.table.length));
        this.table = copy;
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.table.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.table.length / OCCUPIED_SENTINEL_RATIO;
    }

    private static boolean isNonSentinel(int value)
    {
        return value != EMPTY && value != REMOVED;
    }

    private static final class ImmutableIntHashSet implements ImmutableIntSet, Serializable
    {
        private static final long serialVersionUID = 1L;
        private final int[] table;
        private final int occupied;
        // The 32 bits of this integer indicate whether the items 0 to 31 are present in the set.
        private final int zeroToThirtyOne;
        private final int zeroToThirtyOneOccupied;

        private ImmutableIntHashSet(int[] table, int occupied, int zeroToThirtyOne, int zeroToThirtyOneOccupied)
        {
            this.checkOptimizedSize(occupied + zeroToThirtyOneOccupied);
            this.occupied = occupied;
            this.zeroToThirtyOneOccupied = zeroToThirtyOneOccupied;
            this.zeroToThirtyOne = zeroToThirtyOne;
            this.table = table;
        }

        private void checkOptimizedSize(int length)
        {
            if (length <= 1)
            {
                throw new IllegalArgumentException("Use IntSets.immutable.with() to instantiate an optimized collection");
            }
        }

        public static ImmutableIntSet newSetWith(int... elements)
        {
            return IntHashSet.newSetWith(elements).toImmutable();
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }

            if (!(obj instanceof IntSet))
            {
                return false;
            }

            IntSet other = (IntSet) obj;
            return this.size() == other.size() && this.containsAll(other.toArray());
        }

        @Override
        public int hashCode()
        {
            int result = 0;
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                result += value;
                zeroToThirtyOne &= ~(1 << value);
            }
            if (this.table != null)
            {
                for (int i = 0; i < this.table.length; i++)
                {
                    if (isNonSentinel(this.table[i]))
                    {
                        result += this.table[i];
                    }
                }
            }
            return result;
        }

        @Override
        public String toString()
        {
            return this.makeString("[", ", ", "]");
        }

        public ImmutableIntSet newWith(int element)
        {
            return IntHashSet.newSet(this).with(element).toImmutable();
        }

        public ImmutableIntSet newWithout(int element)
        {
            return IntHashSet.newSet(this).without(element).toImmutable();
        }

        public ImmutableIntSet newWithAll(IntIterable elements)
        {
            return IntHashSet.newSet(this).withAll(elements).toImmutable();
        }

        public ImmutableIntSet newWithoutAll(IntIterable elements)
        {
            return IntHashSet.newSet(this).withoutAll(elements).toImmutable();
        }

        public int size()
        {
            return this.occupied + this.zeroToThirtyOneOccupied;
        }

        public boolean isEmpty()
        {
            return this.size() == 0;
        }

        public boolean notEmpty()
        {
            return this.size() != 0;
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                int count = 0;
                int zeroToThirtyOne = this.zeroToThirtyOne;
                while (zeroToThirtyOne != 0)
                {
                    int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                    if (count > 0)
                    {
                        appendable.append(separator);
                    }
                    count++;
                    appendable.append(String.valueOf(value));
                    zeroToThirtyOne &= ~(1 << value);
                }

                for (int value : this.table)
                {
                    if (isNonSentinel(value))
                    {
                        if (count > 0)
                        {
                            appendable.append(separator);
                        }
                        count++;
                        appendable.append(String.valueOf(value));
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public IntIterator intIterator()
        {
            return new InternalIntIterator();
        }

        public int[] toArray()
        {
            int[] array = new int[this.size()];

            int j = 0;
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                array[j] = value;
                j++;
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int i = 0; i < this.table.length && j < this.size(); i++)
            {
                if (isNonSentinel(this.table[i]))
                {
                    array[j] = this.table[i];
                    j++;
                }
            }
            return array;
        }

        public boolean contains(int value)
        {
            if (isBetweenZeroAndThirtyOne(value))
            {
                int temp = this.zeroToThirtyOne;
                return ((temp >>> value) & 1) != 0;
            }
            return this.table[this.probe(value)] == value;
        }

        public boolean containsAll(int... source)
        {
            for (int item : source)
            {
                if (!this.contains(item))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean containsAll(IntIterable source)
        {
            for (IntIterator iterator = source.intIterator(); iterator.hasNext();)
            {
                if (!this.contains(iterator.next()))
                {
                    return false;
                }
            }
            return true;
        }

        public void forEach(IntProcedure procedure)
        {
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                procedure.value(value);
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    procedure.value(value);
                }
            }
        }

        public ImmutableIntSet select(IntPredicate predicate)
        {
            IntHashSet result = new IntHashSet();

            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (predicate.accept(value))
                {
                    result.add(value);
                }
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (predicate.accept(value))
                    {
                        result.add(value);
                    }
                }
            }
            return result.toImmutable();
        }

        public ImmutableIntSet reject(IntPredicate predicate)
        {
            IntHashSet result = new IntHashSet();

            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (!predicate.accept(value))
                {
                    result.add(value);
                }
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (!predicate.accept(value))
                    {
                        result.add(value);
                    }
                }
            }
            return result.toImmutable();
        }

        public <V> ImmutableSet<V> collect(IntToObjectFunction<? extends V> function)
        {
            MutableSet<V> target = UnifiedSet.newSet(this.size());

            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                target.add(function.valueOf(value));
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    target.add(function.valueOf(value));
                }
            }
            return target.toImmutable();
        }

        public int detectIfNone(IntPredicate predicate, int ifNone)
        {
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (predicate.accept(value))
                {
                    return value;
                }
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (predicate.accept(value))
                    {
                        return value;
                    }
                }
            }
            return ifNone;
        }

        public int count(IntPredicate predicate)
        {
            int count = 0;
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (predicate.accept(value))
                {
                    count++;
                }
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (predicate.accept(value))
                    {
                        count++;
                    }
                }
            }
            return count;
        }

        public boolean anySatisfy(IntPredicate predicate)
        {
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (predicate.accept(value))
                {
                    return true;
                }
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (predicate.accept(value))
                    {
                        return true;
                    }
                }
            }
            return false;
        }

        public boolean allSatisfy(IntPredicate predicate)
        {
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (!predicate.accept(value))
                {
                    return false;
                }
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (!predicate.accept(value))
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        public boolean noneSatisfy(IntPredicate predicate)
        {
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                if (predicate.accept(value))
                {
                    return false;
                }
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    if (predicate.accept(value))
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        public MutableIntList toList()
        {
            return IntArrayList.newList(this);
        }

        public MutableIntSet toSet()
        {
            return IntHashSet.newSet(this);
        }

        public MutableIntBag toBag()
        {
            return IntHashBag.newBag(this);
        }

        public LazyIntIterable asLazy()
        {
            return new LazyIntIterableAdapter(this);
        }

        public long sum()
        {
            long result = 0L;

            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                result += value;
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    result += value;
                }
            }
            return result;
        }

        public int max()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            int max = 31 - Integer.numberOfLeadingZeros(this.zeroToThirtyOne);
            boolean isMaxSet = this.zeroToThirtyOneOccupied != 0;

            for (int value : this.table)
            {
                if (isNonSentinel(value) && (!isMaxSet || max < value))
                {
                    max = value;
                    isMaxSet = true;
                }
            }
            return max;
        }

        public int maxIfEmpty(int defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.max();
        }

        public int min()
        {
            if (this.isEmpty())
            {
                throw new NoSuchElementException();
            }
            int min = Integer.numberOfTrailingZeros(this.zeroToThirtyOne);
            boolean isMinSet = this.zeroToThirtyOneOccupied != 0;

            for (int value : this.table)
            {
                if (isNonSentinel(value) && (!isMinSet || value < min))
                {
                    min = value;
                    isMinSet = true;
                }
            }
            return min;
        }

        public int minIfEmpty(int defaultValue)
        {
            if (this.isEmpty())
            {
                return defaultValue;
            }
            return this.min();
        }

        public double average()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            return (double) this.sum() / (double) this.size();
        }

        public double median()
        {
            if (this.isEmpty())
            {
                throw new ArithmeticException();
            }
            int[] sortedArray = this.toSortedArray();
            int middleIndex = sortedArray.length >> 1;
            if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
            {
                int first = sortedArray[middleIndex];
                int second = sortedArray[middleIndex - 1];
                return ((double) first + (double) second) / 2.0;
            }
            return (double) sortedArray[middleIndex];
        }

        public int[] toSortedArray()
        {
            int[] array = this.toArray();
            Arrays.sort(array);
            return array;
        }

        public MutableIntList toSortedList()
        {
            return IntArrayList.newList(this).sortThis();
        }

        public <T> T injectInto(T injectedValue, ObjectIntToObjectFunction<? super T, ? extends T> function)
        {
            T result = injectedValue;
            int zeroToThirtyOne = this.zeroToThirtyOne;
            while (zeroToThirtyOne != 0)
            {
                int value = Integer.numberOfTrailingZeros(zeroToThirtyOne);
                result = function.valueOf(result, value);
                zeroToThirtyOne &= ~(1 << value);
            }

            for (int value : this.table)
            {
                if (isNonSentinel(value))
                {
                    result = function.valueOf(result, value);
                }
            }
            return result;
        }

        public IntSet freeze()
        {
            return this;
        }

        public ImmutableIntSet toImmutable()
        {
            return this;
        }

        private Object writeReplace()
        {
            return new ImmutableIntSetSerializationProxy(this);
        }

        // exposed for testing
        int probe(int element)
        {
            int index = this.spread(element);
            int valueAtIndex = this.table[index];

            if (valueAtIndex == element || valueAtIndex == EMPTY)
            {
                return index;
            }

            int removedIndex = valueAtIndex == REMOVED ? index : -1;
            int nextIndex = index;
            int probe = 17;

            // loop until an empty slot is reached
            while (true)
            {
                // Probe algorithm: 17 * n * (n+1) / 2 where n = number of collisions
                nextIndex = (nextIndex + probe) & this.table.length - 1;
                probe += 17;

                if (this.table[nextIndex] == element)
                {
                    return nextIndex;
                }
                if (this.table[nextIndex] == REMOVED)
                {
                    if (removedIndex == -1)
                    {
                        removedIndex = nextIndex;
                    }
                }
                else if (this.table[nextIndex] == EMPTY)
                {
                    return removedIndex == -1 ? nextIndex : removedIndex;
                }
            }
        }

        // exposed for testing
        int spread(int element)
        {
            int code = element;
            code ^= 61 ^ (code >> 16);
            code += code << 3;
            code ^= code >> 4;
            code *= 0x27d4eb2d;
            code ^= code >> 15;
            return code & (this.table.length - 1);
        }

        private class InternalIntIterator implements IntIterator
        {
            private int count;
            private int position;
            private int zeroToThirtyOne;

            public boolean hasNext()
            {
                return this.count < ImmutableIntHashSet.this.size();
            }

            public int next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException("next() called, but the iterator is exhausted");
                }
                this.count++;

                while (this.zeroToThirtyOne < 32)
                {
                    if (ImmutableIntHashSet.this.contains(this.zeroToThirtyOne))
                    {
                        int result = this.zeroToThirtyOne;
                        this.zeroToThirtyOne++;
                        return result;
                    }
                    this.zeroToThirtyOne++;
                }

                int[] table = ImmutableIntHashSet.this.table;
                while (!isNonSentinel(table[this.position]))
                {
                    this.position++;
                }
                int result = table[this.position];
                this.position++;
                return result;
            }
        }
    }

    private class InternalIntIterator implements IntIterator
    {
        private int count;
        private int position;
        private int zeroToThirtyOne;

        public boolean hasNext()
        {
            return this.count < IntHashSet.this.size();
        }

        public int next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException("next() called, but the iterator is exhausted");
            }
            this.count++;

            while (this.zeroToThirtyOne < 32)
            {
                if (IntHashSet.this.contains(this.zeroToThirtyOne))
                {
                    int result = this.zeroToThirtyOne;
                    this.zeroToThirtyOne++;
                    return result;
                }
                this.zeroToThirtyOne++;
            }

            int[] table = IntHashSet.this.table;
            while (!isNonSentinel(table[this.position]))
            {
                this.position++;
            }
            int result = table[this.position];
            this.position++;
            return result;
        }
    }
}
