/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.firestore.local;

import android.util.SparseArray;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import com.google.firebase.Timestamp;
import com.google.firebase.database.collection.ImmutableSortedMap;
import com.google.firebase.database.collection.ImmutableSortedSet;
import com.google.firebase.firestore.auth.User;
import com.google.firebase.firestore.bundle.BundleCallback;
import com.google.firebase.firestore.bundle.BundleMetadata;
import com.google.firebase.firestore.bundle.NamedQuery;
import com.google.firebase.firestore.core.Query;
import com.google.firebase.firestore.core.Target;
import com.google.firebase.firestore.core.TargetIdGenerator;
import com.google.firebase.firestore.local.BundleCache;
import com.google.firebase.firestore.local.IndexBackfiller;
import com.google.firebase.firestore.local.IndexManager;
import com.google.firebase.firestore.local.LocalDocumentsView;
import com.google.firebase.firestore.local.LocalViewChanges;
import com.google.firebase.firestore.local.LocalWriteResult;
import com.google.firebase.firestore.local.LruGarbageCollector;
import com.google.firebase.firestore.local.MutationQueue;
import com.google.firebase.firestore.local.Persistence;
import com.google.firebase.firestore.local.QueryEngine;
import com.google.firebase.firestore.local.QueryPurpose;
import com.google.firebase.firestore.local.QueryResult;
import com.google.firebase.firestore.local.ReferenceSet;
import com.google.firebase.firestore.local.RemoteDocumentCache;
import com.google.firebase.firestore.local.TargetCache;
import com.google.firebase.firestore.local.TargetData;
import com.google.firebase.firestore.model.Document;
import com.google.firebase.firestore.model.DocumentKey;
import com.google.firebase.firestore.model.FieldIndex;
import com.google.firebase.firestore.model.MutableDocument;
import com.google.firebase.firestore.model.ObjectValue;
import com.google.firebase.firestore.model.ResourcePath;
import com.google.firebase.firestore.model.SnapshotVersion;
import com.google.firebase.firestore.model.mutation.Mutation;
import com.google.firebase.firestore.model.mutation.MutationBatch;
import com.google.firebase.firestore.model.mutation.MutationBatchResult;
import com.google.firebase.firestore.model.mutation.PatchMutation;
import com.google.firebase.firestore.model.mutation.Precondition;
import com.google.firebase.firestore.remote.RemoteEvent;
import com.google.firebase.firestore.remote.TargetChange;
import com.google.firebase.firestore.util.Assert;
import com.google.firebase.firestore.util.Logger;
import com.google.protobuf.ByteString;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

public final class LocalStore
implements BundleCallback {
    private static final long RESUME_TOKEN_MAX_AGE_SECONDS = TimeUnit.MINUTES.toSeconds(5L);
    private final Persistence persistence;
    private final IndexManager indexManager;
    private MutationQueue mutationQueue;
    private final RemoteDocumentCache remoteDocuments;
    private LocalDocumentsView localDocuments;
    private QueryEngine queryEngine;
    private final ReferenceSet localViewReferences;
    private final TargetCache targetCache;
    private final BundleCache bundleCache;
    private final SparseArray<TargetData> queryDataByTarget;
    private final Map<Target, Integer> targetIdByTarget;
    private final TargetIdGenerator targetIdGenerator;

    public LocalStore(Persistence persistence, QueryEngine queryEngine, User initialUser) {
        Assert.hardAssert(persistence.isStarted(), "LocalStore was passed an unstarted persistence implementation", new Object[0]);
        this.persistence = persistence;
        this.targetCache = persistence.getTargetCache();
        this.bundleCache = persistence.getBundleCache();
        this.targetIdGenerator = TargetIdGenerator.forTargetCache(this.targetCache.getHighestTargetId());
        this.mutationQueue = persistence.getMutationQueue(initialUser);
        this.remoteDocuments = persistence.getRemoteDocumentCache();
        this.indexManager = persistence.getIndexManager();
        this.localDocuments = new LocalDocumentsView(this.remoteDocuments, this.mutationQueue, this.indexManager);
        this.queryEngine = queryEngine;
        queryEngine.setLocalDocumentsView(this.localDocuments);
        this.localViewReferences = new ReferenceSet();
        persistence.getReferenceDelegate().setInMemoryPins(this.localViewReferences);
        this.queryDataByTarget = new SparseArray();
        this.targetIdByTarget = new HashMap<Target, Integer>();
    }

    public void start() {
        this.startMutationQueue();
    }

    private void startMutationQueue() {
        this.persistence.runTransaction("Start MutationQueue", () -> this.mutationQueue.start());
    }

    public ImmutableSortedMap<DocumentKey, Document> handleUserChange(User user) {
        List<MutationBatch> oldBatches = this.mutationQueue.getAllMutationBatches();
        this.mutationQueue = this.persistence.getMutationQueue(user);
        this.startMutationQueue();
        List<MutationBatch> newBatches = this.mutationQueue.getAllMutationBatches();
        this.localDocuments = new LocalDocumentsView(this.remoteDocuments, this.mutationQueue, this.indexManager);
        this.queryEngine.setLocalDocumentsView(this.localDocuments);
        ImmutableSortedSet changedKeys = DocumentKey.emptyKeySet();
        for (List batches : Arrays.asList(oldBatches, newBatches)) {
            for (MutationBatch batch : batches) {
                for (Mutation mutation : batch.getMutations()) {
                    changedKeys = changedKeys.insert((Object)mutation.getKey());
                }
            }
        }
        return this.localDocuments.getDocuments((Iterable<DocumentKey>)changedKeys);
    }

    public LocalWriteResult writeLocally(List<Mutation> mutations) {
        Timestamp localWriteTime = Timestamp.now();
        HashSet<DocumentKey> keys = new HashSet<DocumentKey>();
        for (Mutation mutation : mutations) {
            keys.add(mutation.getKey());
        }
        return this.persistence.runTransaction("Locally write mutations", () -> {
            ImmutableSortedMap<DocumentKey, Document> documents = this.localDocuments.getDocuments(keys);
            ArrayList<Mutation> baseMutations = new ArrayList<Mutation>();
            for (Mutation mutation : mutations) {
                ObjectValue baseValue = mutation.extractTransformBaseValue((Document)documents.get((Object)mutation.getKey()));
                if (baseValue == null) continue;
                baseMutations.add(new PatchMutation(mutation.getKey(), baseValue, baseValue.getFieldMask(), Precondition.exists(true)));
            }
            MutationBatch batch = this.mutationQueue.addMutationBatch(localWriteTime, baseMutations, mutations);
            batch.applyToLocalDocumentSet(documents);
            return new LocalWriteResult(batch.getBatchId(), documents);
        });
    }

    public ImmutableSortedMap<DocumentKey, Document> acknowledgeBatch(MutationBatchResult batchResult) {
        return this.persistence.runTransaction("Acknowledge batch", () -> {
            MutationBatch batch = batchResult.getBatch();
            this.mutationQueue.acknowledgeBatch(batch, batchResult.getStreamToken());
            this.applyWriteToRemoteDocuments(batchResult);
            this.mutationQueue.performConsistencyCheck();
            return this.localDocuments.getDocuments(batch.getKeys());
        });
    }

    public ImmutableSortedMap<DocumentKey, Document> rejectBatch(int batchId) {
        return this.persistence.runTransaction("Reject batch", () -> {
            MutationBatch toReject = this.mutationQueue.lookupMutationBatch(batchId);
            Assert.hardAssert(toReject != null, "Attempt to reject nonexistent batch!", new Object[0]);
            this.mutationQueue.removeMutationBatch(toReject);
            this.mutationQueue.performConsistencyCheck();
            return this.localDocuments.getDocuments(toReject.getKeys());
        });
    }

    public int getHighestUnacknowledgedBatchId() {
        return this.mutationQueue.getHighestUnacknowledgedBatchId();
    }

    public ByteString getLastStreamToken() {
        return this.mutationQueue.getLastStreamToken();
    }

    public void setLastStreamToken(ByteString streamToken) {
        this.persistence.runTransaction("Set stream token", () -> this.mutationQueue.setLastStreamToken(streamToken));
    }

    public SnapshotVersion getLastRemoteSnapshotVersion() {
        return this.targetCache.getLastRemoteSnapshotVersion();
    }

    public ImmutableSortedMap<DocumentKey, Document> applyRemoteEvent(RemoteEvent remoteEvent) {
        SnapshotVersion remoteVersion = remoteEvent.getSnapshotVersion();
        return this.persistence.runTransaction("Apply remote event", () -> {
            Map<Integer, TargetChange> targetChanges = remoteEvent.getTargetChanges();
            long sequenceNumber = this.persistence.getReferenceDelegate().getCurrentSequenceNumber();
            for (Map.Entry<Integer, TargetChange> entry : targetChanges.entrySet()) {
                Integer boxedTargetId = entry.getKey();
                int targetId = boxedTargetId;
                TargetChange change = entry.getValue();
                TargetData oldTargetData = (TargetData)this.queryDataByTarget.get(targetId);
                if (oldTargetData == null) continue;
                this.targetCache.removeMatchingKeys(change.getRemovedDocuments(), targetId);
                this.targetCache.addMatchingKeys(change.getAddedDocuments(), targetId);
                ByteString resumeToken = change.getResumeToken();
                if (resumeToken.isEmpty()) continue;
                TargetData newTargetData = oldTargetData.withResumeToken(resumeToken, remoteEvent.getSnapshotVersion()).withSequenceNumber(sequenceNumber);
                this.queryDataByTarget.put(targetId, (Object)newTargetData);
                if (!LocalStore.shouldPersistTargetData(oldTargetData, newTargetData, change)) continue;
                this.targetCache.updateTargetData(newTargetData);
            }
            Map<DocumentKey, MutableDocument> documentUpdates = remoteEvent.getDocumentUpdates();
            Set<DocumentKey> limboDocuments = remoteEvent.getResolvedLimboDocuments();
            for (DocumentKey key : documentUpdates.keySet()) {
                if (!limboDocuments.contains(key)) continue;
                this.persistence.getReferenceDelegate().updateLimboDocument(key);
            }
            Map<DocumentKey, MutableDocument> changedDocs = this.populateDocumentChanges(documentUpdates, null, remoteEvent.getSnapshotVersion());
            SnapshotVersion lastRemoteVersion = this.targetCache.getLastRemoteSnapshotVersion();
            if (!remoteVersion.equals(SnapshotVersion.NONE)) {
                Assert.hardAssert(remoteVersion.compareTo(lastRemoteVersion) >= 0, "Watch stream reverted to previous snapshot?? (%s < %s)", remoteVersion, lastRemoteVersion);
                this.targetCache.setLastRemoteSnapshotVersion(remoteVersion);
            }
            return this.localDocuments.getLocalViewOfDocuments(changedDocs);
        });
    }

    private Map<DocumentKey, MutableDocument> populateDocumentChanges(Map<DocumentKey, MutableDocument> documents, @Nullable Map<DocumentKey, SnapshotVersion> documentVersions, SnapshotVersion globalVersion) {
        HashMap<DocumentKey, MutableDocument> changedDocs = new HashMap<DocumentKey, MutableDocument>();
        Map<DocumentKey, MutableDocument> existingDocs = this.remoteDocuments.getAll(documents.keySet());
        for (Map.Entry<DocumentKey, MutableDocument> entry : documents.entrySet()) {
            SnapshotVersion readTime;
            DocumentKey key = entry.getKey();
            MutableDocument doc = entry.getValue();
            MutableDocument existingDoc = existingDocs.get(key);
            SnapshotVersion snapshotVersion = readTime = documentVersions != null ? documentVersions.get(key) : globalVersion;
            if (doc.isNoDocument() && doc.getVersion().equals(SnapshotVersion.NONE)) {
                this.remoteDocuments.remove(doc.getKey());
                changedDocs.put(key, doc);
                continue;
            }
            if (!existingDoc.isValidDocument() || doc.getVersion().compareTo(existingDoc.getVersion()) > 0 || doc.getVersion().compareTo(existingDoc.getVersion()) == 0 && existingDoc.hasPendingWrites()) {
                Assert.hardAssert(!SnapshotVersion.NONE.equals(readTime), "Cannot add a document when the remote version is zero", new Object[0]);
                this.remoteDocuments.add(doc, readTime);
                changedDocs.put(key, doc);
                continue;
            }
            Logger.debug("LocalStore", "Ignoring outdated watch update for %s.Current version: %s  Watch version: %s", key, existingDoc.getVersion(), doc.getVersion());
        }
        return changedDocs;
    }

    private static boolean shouldPersistTargetData(TargetData oldTargetData, TargetData newTargetData, TargetChange change) {
        long oldSeconds;
        Assert.hardAssert(!newTargetData.getResumeToken().isEmpty(), "Attempted to persist query data with empty resume token", new Object[0]);
        if (oldTargetData.getResumeToken().isEmpty()) {
            return true;
        }
        long newSeconds = newTargetData.getSnapshotVersion().getTimestamp().getSeconds();
        long timeDelta = newSeconds - (oldSeconds = oldTargetData.getSnapshotVersion().getTimestamp().getSeconds());
        if (timeDelta >= RESUME_TOKEN_MAX_AGE_SECONDS) {
            return true;
        }
        int changes = change.getAddedDocuments().size() + change.getModifiedDocuments().size() + change.getRemovedDocuments().size();
        return changes > 0;
    }

    public void notifyLocalViewChanges(List<LocalViewChanges> viewChanges) {
        this.persistence.runTransaction("notifyLocalViewChanges", () -> {
            for (LocalViewChanges viewChange : viewChanges) {
                int targetId = viewChange.getTargetId();
                this.localViewReferences.addReferences(viewChange.getAdded(), targetId);
                ImmutableSortedSet<DocumentKey> removed = viewChange.getRemoved();
                for (DocumentKey key : removed) {
                    this.persistence.getReferenceDelegate().removeReference(key);
                }
                this.localViewReferences.removeReferences(removed, targetId);
                if (viewChange.isFromCache()) continue;
                TargetData targetData = (TargetData)this.queryDataByTarget.get(targetId);
                Assert.hardAssert(targetData != null, "Can't set limbo-free snapshot version for unknown target: %s", targetId);
                SnapshotVersion lastLimboFreeSnapshotVersion = targetData.getSnapshotVersion();
                TargetData updatedTargetData = targetData.withLastLimboFreeSnapshotVersion(lastLimboFreeSnapshotVersion);
                this.queryDataByTarget.put(targetId, (Object)updatedTargetData);
            }
        });
    }

    @Nullable
    public MutationBatch getNextMutationBatch(int afterBatchId) {
        return this.mutationQueue.getNextMutationBatchAfterBatchId(afterBatchId);
    }

    public Document readDocument(DocumentKey key) {
        return this.localDocuments.getDocument(key);
    }

    public TargetData allocateTarget(Target target) {
        int targetId;
        TargetData cached = this.targetCache.getTargetData(target);
        if (cached != null) {
            targetId = cached.getTargetId();
        } else {
            AllocateQueryHolder holder = new AllocateQueryHolder();
            this.persistence.runTransaction("Allocate target", () -> {
                holder.targetId = this.targetIdGenerator.nextId();
                holder.cached = new TargetData(target, holder.targetId, this.persistence.getReferenceDelegate().getCurrentSequenceNumber(), QueryPurpose.LISTEN);
                this.targetCache.addTargetData(holder.cached);
            });
            targetId = holder.targetId;
            cached = holder.cached;
        }
        if (this.queryDataByTarget.get(targetId) == null) {
            this.queryDataByTarget.put(targetId, (Object)cached);
            this.targetIdByTarget.put(target, targetId);
        }
        return cached;
    }

    @VisibleForTesting
    @Nullable
    TargetData getTargetData(Target target) {
        Integer targetId = this.targetIdByTarget.get(target);
        if (targetId != null) {
            return (TargetData)this.queryDataByTarget.get(targetId.intValue());
        }
        return this.targetCache.getTargetData(target);
    }

    public boolean hasNewerBundle(BundleMetadata bundleMetadata) {
        return this.persistence.runTransaction("Has newer bundle", () -> {
            BundleMetadata cachedMetadata = this.bundleCache.getBundleMetadata(bundleMetadata.getBundleId());
            return cachedMetadata != null && cachedMetadata.getCreateTime().compareTo(bundleMetadata.getCreateTime()) >= 0;
        });
    }

    @Override
    public void saveBundle(BundleMetadata bundleMetadata) {
        this.persistence.runTransaction("Save bundle", () -> this.bundleCache.saveBundleMetadata(bundleMetadata));
    }

    @Override
    public ImmutableSortedMap<DocumentKey, Document> applyBundledDocuments(ImmutableSortedMap<DocumentKey, MutableDocument> documents, String bundleId) {
        TargetData umbrellaTargetData = this.allocateTarget(LocalStore.newUmbrellaTarget(bundleId));
        return this.persistence.runTransaction("Apply bundle documents", () -> {
            ImmutableSortedSet documentKeys = DocumentKey.emptyKeySet();
            HashMap<DocumentKey, MutableDocument> documentMap = new HashMap<DocumentKey, MutableDocument>();
            HashMap<DocumentKey, SnapshotVersion> versionMap = new HashMap<DocumentKey, SnapshotVersion>();
            for (Map.Entry entry : documents) {
                DocumentKey documentKey = (DocumentKey)entry.getKey();
                MutableDocument document = (MutableDocument)entry.getValue();
                if (document.isFoundDocument()) {
                    documentKeys = documentKeys.insert((Object)documentKey);
                }
                documentMap.put(documentKey, document);
                versionMap.put(documentKey, document.getVersion());
            }
            this.targetCache.removeMatchingKeysForTargetId(umbrellaTargetData.getTargetId());
            this.targetCache.addMatchingKeys(documentKeys, umbrellaTargetData.getTargetId());
            Map<DocumentKey, MutableDocument> changedDocs = this.populateDocumentChanges(documentMap, versionMap, SnapshotVersion.NONE);
            return this.localDocuments.getLocalViewOfDocuments(changedDocs);
        });
    }

    @Override
    public void saveNamedQuery(NamedQuery namedQuery, ImmutableSortedSet<DocumentKey> documentKeys) {
        TargetData existingTargetData = this.allocateTarget(namedQuery.getBundledQuery().getTarget());
        int targetId = existingTargetData.getTargetId();
        this.persistence.runTransaction("Saved named query", () -> {
            if (namedQuery.getReadTime().compareTo(existingTargetData.getSnapshotVersion()) > 0) {
                TargetData newTargetData = existingTargetData.withResumeToken(ByteString.EMPTY, namedQuery.getReadTime());
                this.queryDataByTarget.append(targetId, (Object)newTargetData);
                this.targetCache.updateTargetData(newTargetData);
                this.targetCache.removeMatchingKeysForTargetId(targetId);
                this.targetCache.addMatchingKeys(documentKeys, targetId);
            }
            this.bundleCache.saveNamedQuery(namedQuery);
        });
    }

    @Nullable
    public NamedQuery getNamedQuery(String queryName) {
        return this.persistence.runTransaction("Get named query", () -> this.bundleCache.getNamedQuery(queryName));
    }

    public void configureIndices(List<FieldIndex> fieldIndices) {
        this.persistence.runTransaction("Configure indices", () -> {
            for (FieldIndex fieldIndex : fieldIndices) {
                this.indexManager.addFieldIndex(fieldIndex);
            }
        });
    }

    public void releaseTarget(int targetId) {
        this.persistence.runTransaction("Release target", () -> {
            TargetData targetData = (TargetData)this.queryDataByTarget.get(targetId);
            Assert.hardAssert(targetData != null, "Tried to release nonexistent target: %s", targetId);
            ImmutableSortedSet<DocumentKey> removedReferences = this.localViewReferences.removeReferencesForId(targetId);
            for (DocumentKey key : removedReferences) {
                this.persistence.getReferenceDelegate().removeReference(key);
            }
            this.persistence.getReferenceDelegate().removeTarget(targetData);
            this.queryDataByTarget.remove(targetId);
            this.targetIdByTarget.remove(targetData.getTarget());
        });
    }

    public QueryResult executeQuery(Query query, boolean usePreviousResults) {
        TargetData targetData = this.getTargetData(query.toTarget());
        SnapshotVersion lastLimboFreeSnapshotVersion = SnapshotVersion.NONE;
        ImmutableSortedSet<DocumentKey> remoteKeys = DocumentKey.emptyKeySet();
        if (targetData != null) {
            lastLimboFreeSnapshotVersion = targetData.getLastLimboFreeSnapshotVersion();
            remoteKeys = this.targetCache.getMatchingKeysForTargetId(targetData.getTargetId());
        }
        ImmutableSortedMap<DocumentKey, Document> documents = this.queryEngine.getDocumentsMatchingQuery(query, usePreviousResults ? lastLimboFreeSnapshotVersion : SnapshotVersion.NONE, usePreviousResults ? remoteKeys : DocumentKey.emptyKeySet());
        return new QueryResult(documents, remoteKeys);
    }

    public ImmutableSortedSet<DocumentKey> getRemoteDocumentKeys(int targetId) {
        return this.targetCache.getMatchingKeysForTargetId(targetId);
    }

    private void applyWriteToRemoteDocuments(MutationBatchResult batchResult) {
        MutationBatch batch = batchResult.getBatch();
        Set<DocumentKey> docKeys = batch.getKeys();
        for (DocumentKey docKey : docKeys) {
            MutableDocument doc = this.remoteDocuments.get(docKey);
            SnapshotVersion ackVersion = (SnapshotVersion)batchResult.getDocVersions().get((Object)docKey);
            Assert.hardAssert(ackVersion != null, "docVersions should contain every doc in the write.", new Object[0]);
            if (doc.getVersion().compareTo(ackVersion) >= 0) continue;
            batch.applyToRemoteDocument(doc, batchResult);
            if (!doc.isValidDocument()) continue;
            this.remoteDocuments.add(doc, batchResult.getCommitVersion());
        }
        this.mutationQueue.removeMutationBatch(batch);
    }

    public LruGarbageCollector.Results collectGarbage(LruGarbageCollector garbageCollector) {
        return this.persistence.runTransaction("Collect garbage", () -> garbageCollector.collect(this.queryDataByTarget));
    }

    public IndexBackfiller.Results backfillIndexes(IndexBackfiller indexBackfiller) {
        return this.persistence.runTransaction("Backfill Indexes", () -> indexBackfiller.backfill());
    }

    private static Target newUmbrellaTarget(String bundleName) {
        return Query.atPath(ResourcePath.fromString("__bundle__/docs/" + bundleName)).toTarget();
    }

    private static class AllocateQueryHolder {
        TargetData cached;
        int targetId;

        private AllocateQueryHolder() {
        }
    }
}

