/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.firestore.local;

import androidx.annotation.VisibleForTesting;
import com.google.firebase.database.collection.ImmutableSortedMap;
import com.google.firebase.database.collection.ImmutableSortedSet;
import com.google.firebase.firestore.core.Query;
import com.google.firebase.firestore.core.Target;
import com.google.firebase.firestore.local.IndexManager;
import com.google.firebase.firestore.local.LocalDocumentsView;
import com.google.firebase.firestore.local.QueryContext;
import com.google.firebase.firestore.model.Document;
import com.google.firebase.firestore.model.DocumentKey;
import com.google.firebase.firestore.model.FieldIndex;
import com.google.firebase.firestore.model.SnapshotVersion;
import com.google.firebase.firestore.util.Assert;
import com.google.firebase.firestore.util.Logger;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.annotation.Nullable;

public class QueryEngine {
    private static final String LOG_TAG = "QueryEngine";
    private static final int DEFAULT_INDEX_AUTO_CREATION_MIN_COLLECTION_SIZE = 100;
    private static final double DEFAULT_RELATIVE_INDEX_READ_COST_PER_DOCUMENT = 2.0;
    private LocalDocumentsView localDocumentsView;
    private IndexManager indexManager;
    private boolean initialized;
    private boolean indexAutoCreationEnabled = false;
    private int indexAutoCreationMinCollectionSize = 100;
    private double relativeIndexReadCostPerDocument = 2.0;

    public void initialize(LocalDocumentsView localDocumentsView, IndexManager indexManager) {
        this.localDocumentsView = localDocumentsView;
        this.indexManager = indexManager;
        this.initialized = true;
    }

    public void setIndexAutoCreationEnabled(boolean isEnabled) {
        this.indexAutoCreationEnabled = isEnabled;
    }

    public ImmutableSortedMap<DocumentKey, Document> getDocumentsMatchingQuery(Query query, SnapshotVersion lastLimboFreeSnapshotVersion, ImmutableSortedSet<DocumentKey> remoteKeys) {
        Assert.hardAssert(this.initialized, "initialize() not called", new Object[0]);
        ImmutableSortedMap<DocumentKey, Document> result = this.performQueryUsingIndex(query);
        if (result != null) {
            return result;
        }
        result = this.performQueryUsingRemoteKeys(query, remoteKeys, lastLimboFreeSnapshotVersion);
        if (result != null) {
            return result;
        }
        QueryContext context = new QueryContext();
        result = this.executeFullCollectionScan(query, context);
        if (result != null && this.indexAutoCreationEnabled) {
            this.createCacheIndexes(query, context, result.size());
        }
        return result;
    }

    private void createCacheIndexes(Query query, QueryContext context, int resultSize) {
        if (context.getDocumentReadCount() < this.indexAutoCreationMinCollectionSize) {
            Logger.debug(LOG_TAG, "SDK will not create cache indexes for query: %s, since it only creates cache indexes for collection contains more than or equal to %s documents.", query.toString(), this.indexAutoCreationMinCollectionSize);
            return;
        }
        Logger.debug(LOG_TAG, "Query: %s, scans %s local documents and returns %s documents as results.", query.toString(), context.getDocumentReadCount(), resultSize);
        if ((double)context.getDocumentReadCount() > this.relativeIndexReadCostPerDocument * (double)resultSize) {
            this.indexManager.createTargetIndexes(query.toTarget());
            Logger.debug(LOG_TAG, "The SDK decides to create cache indexes for query: %s, as using cache indexes may help improve performance.", query.toString());
        }
    }

    @Nullable
    private ImmutableSortedMap<DocumentKey, Document> performQueryUsingIndex(Query query) {
        if (query.matchesAllDocuments()) {
            return null;
        }
        Target target = query.toTarget();
        IndexManager.IndexType indexType = this.indexManager.getIndexType(target);
        if (indexType.equals((Object)IndexManager.IndexType.NONE)) {
            return null;
        }
        if (query.hasLimit() && indexType.equals((Object)IndexManager.IndexType.PARTIAL)) {
            return this.performQueryUsingIndex(query.limitToFirst(-1L));
        }
        List<DocumentKey> keys = this.indexManager.getDocumentsMatchingTarget(target);
        Assert.hardAssert(keys != null, "index manager must return results for partial and full indexes.", new Object[0]);
        ImmutableSortedMap<DocumentKey, Document> indexedDocuments = this.localDocumentsView.getDocuments(keys);
        FieldIndex.IndexOffset offset = this.indexManager.getMinOffset(target);
        ImmutableSortedSet<Document> previousResults = this.applyQuery(query, indexedDocuments);
        if (this.needsRefill(query, keys.size(), previousResults, offset.getReadTime())) {
            return this.performQueryUsingIndex(query.limitToFirst(-1L));
        }
        return this.appendRemainingResults((Iterable<Document>)previousResults, query, offset);
    }

    @Nullable
    private ImmutableSortedMap<DocumentKey, Document> performQueryUsingRemoteKeys(Query query, ImmutableSortedSet<DocumentKey> remoteKeys, SnapshotVersion lastLimboFreeSnapshotVersion) {
        if (query.matchesAllDocuments()) {
            return null;
        }
        if (lastLimboFreeSnapshotVersion.equals(SnapshotVersion.NONE)) {
            return null;
        }
        ImmutableSortedMap<DocumentKey, Document> documents = this.localDocumentsView.getDocuments((Iterable<DocumentKey>)remoteKeys);
        ImmutableSortedSet<Document> previousResults = this.applyQuery(query, documents);
        if (this.needsRefill(query, remoteKeys.size(), previousResults, lastLimboFreeSnapshotVersion)) {
            return null;
        }
        if (Logger.isDebugEnabled()) {
            Logger.debug(LOG_TAG, "Re-using previous result from %s to execute query: %s", lastLimboFreeSnapshotVersion.toString(), query.toString());
        }
        return this.appendRemainingResults((Iterable<Document>)previousResults, query, FieldIndex.IndexOffset.createSuccessor(lastLimboFreeSnapshotVersion, -1));
    }

    private ImmutableSortedSet<Document> applyQuery(Query query, ImmutableSortedMap<DocumentKey, Document> documents) {
        ImmutableSortedSet queryResults = new ImmutableSortedSet(Collections.emptyList(), query.comparator());
        for (Map.Entry entry : documents) {
            Document document = (Document)entry.getValue();
            if (!query.matches(document)) continue;
            queryResults = queryResults.insert((Object)document);
        }
        return queryResults;
    }

    private boolean needsRefill(Query query, int expectedDocumentCount, ImmutableSortedSet<Document> sortedPreviousResults, SnapshotVersion limboFreeSnapshotVersion) {
        Document documentAtLimitEdge;
        if (!query.hasLimit()) {
            return false;
        }
        if (expectedDocumentCount != sortedPreviousResults.size()) {
            return true;
        }
        Document document = documentAtLimitEdge = query.getLimitType() == Query.LimitType.LIMIT_TO_FIRST ? (Document)sortedPreviousResults.getMaxEntry() : (Document)sortedPreviousResults.getMinEntry();
        if (documentAtLimitEdge == null) {
            return false;
        }
        return documentAtLimitEdge.hasPendingWrites() || documentAtLimitEdge.getVersion().compareTo(limboFreeSnapshotVersion) > 0;
    }

    private ImmutableSortedMap<DocumentKey, Document> executeFullCollectionScan(Query query, QueryContext context) {
        if (Logger.isDebugEnabled()) {
            Logger.debug(LOG_TAG, "Using full collection scan to execute query: %s", query.toString());
        }
        return this.localDocumentsView.getDocumentsMatchingQuery(query, FieldIndex.IndexOffset.NONE, context);
    }

    private ImmutableSortedMap<DocumentKey, Document> appendRemainingResults(Iterable<Document> indexedResults, Query query, FieldIndex.IndexOffset offset) {
        ImmutableSortedMap remainingResults = this.localDocumentsView.getDocumentsMatchingQuery(query, offset);
        for (Document entry : indexedResults) {
            remainingResults = remainingResults.insert((Object)entry.getKey(), (Object)entry);
        }
        return remainingResults;
    }

    @VisibleForTesting
    void setIndexAutoCreationMinCollectionSize(int newMin) {
        this.indexAutoCreationMinCollectionSize = newMin;
    }

    @VisibleForTesting
    void setRelativeIndexReadCostPerDocument(double newCost) {
        this.relativeIndexReadCostPerDocument = newCost;
    }
}

