/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.firestore.remote;

import android.util.Base64;
import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;
import com.google.protobuf.ByteString;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

public final class BloomFilter {
    private final int bitCount;
    private final ByteString bitmap;
    private final int hashCount;
    private final MessageDigest md5HashMessageDigest;

    public BloomFilter(@NonNull ByteString bitmap, int padding, int hashCount) {
        if (padding < 0 || padding >= 8) {
            throw new IllegalArgumentException("Invalid padding: " + padding);
        }
        if (hashCount < 0) {
            throw new IllegalArgumentException("Invalid hash count: " + hashCount);
        }
        if (bitmap.size() > 0 && hashCount == 0) {
            throw new IllegalArgumentException("Invalid hash count: " + hashCount);
        }
        if (bitmap.size() == 0 && padding != 0) {
            throw new IllegalArgumentException("Expected padding of 0 when bitmap length is 0, but got " + padding);
        }
        this.bitmap = bitmap;
        this.hashCount = hashCount;
        this.bitCount = bitmap.size() * 8 - padding;
        this.md5HashMessageDigest = BloomFilter.createMd5HashMessageDigest();
    }

    public static BloomFilter create(@NonNull ByteString bitmap, int padding, int hashCount) throws BloomFilterCreateException {
        if (padding < 0 || padding >= 8) {
            throw new BloomFilterCreateException("Invalid padding: " + padding);
        }
        if (hashCount < 0) {
            throw new BloomFilterCreateException("Invalid hash count: " + hashCount);
        }
        if (bitmap.size() > 0 && hashCount == 0) {
            throw new BloomFilterCreateException("Invalid hash count: " + hashCount);
        }
        if (bitmap.size() == 0 && padding != 0) {
            throw new BloomFilterCreateException("Expected padding of 0 when bitmap length is 0, but got " + padding);
        }
        return new BloomFilter(bitmap, padding, hashCount);
    }

    @VisibleForTesting
    int getBitCount() {
        return this.bitCount;
    }

    public boolean mightContain(@NonNull String value) {
        if (this.bitCount == 0) {
            return false;
        }
        byte[] hashedValue = this.md5HashDigest(value);
        if (hashedValue.length != 16) {
            throw new RuntimeException("Invalid md5 hash array length: " + hashedValue.length + " (expected 16)");
        }
        long hash1 = BloomFilter.getLongLittleEndian(hashedValue, 0);
        long hash2 = BloomFilter.getLongLittleEndian(hashedValue, 8);
        for (int i = 0; i < this.hashCount; ++i) {
            int index = this.getBitIndex(hash1, hash2, i);
            if (this.isBitSet(index)) continue;
            return false;
        }
        return true;
    }

    @NonNull
    private byte[] md5HashDigest(@NonNull String value) {
        return this.md5HashMessageDigest.digest(value.getBytes(StandardCharsets.UTF_8));
    }

    @NonNull
    private static MessageDigest createMd5HashMessageDigest() {
        try {
            return MessageDigest.getInstance("MD5");
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Missing MD5 MessageDigest provider: ", e);
        }
    }

    private static long getLongLittleEndian(@NonNull byte[] bytes, int offset) {
        long result = 0L;
        for (int i = 0; i < 8; ++i) {
            result |= ((long)bytes[offset + i] & 0xFFL) << i * 8;
        }
        return result;
    }

    private int getBitIndex(long hash1, long hash2, int hashIndex) {
        long combinedHash = hash1 + hash2 * (long)hashIndex;
        long modulo = BloomFilter.unsignedRemainder(combinedHash, this.bitCount);
        return (int)modulo;
    }

    private static long unsignedRemainder(long dividend, long divisor) {
        long remainder;
        long quotient = (dividend >>> 1) / divisor << 1;
        return remainder - ((remainder = dividend - quotient * divisor) >= divisor ? divisor : 0L);
    }

    private boolean isBitSet(int index) {
        int offset;
        byte byteAtIndex = this.bitmap.byteAt(index / 8);
        return (byteAtIndex & 1 << (offset = index % 8)) != 0;
    }

    public String toString() {
        return "BloomFilter{hashCount=" + this.hashCount + ", size=" + this.bitCount + ", bitmap=\"" + Base64.encodeToString((byte[])this.bitmap.toByteArray(), (int)2) + "\"}";
    }

    public static final class BloomFilterCreateException
    extends Exception {
        public BloomFilterCreateException(String message) {
            super(message);
        }
    }
}

