package com.gradle.enterprise.gradleplugin;

import com.gradle.obfuscation.Keep;
import com.gradle.scan.plugin.BuildScanExtension;
import org.gradle.api.Action;
import org.gradle.api.NonExtensible;
import org.gradle.caching.configuration.BuildCacheConfiguration;

import javax.annotation.Nullable;

/**
 * The DSL extension for configuring Gradle Enterprise, with the name {@code "gradleEnterprise"}.
 * <pre>{@code
 * gradleEnterprise {
 *   // Gradle Enterprise configuration
 * }
 * }</pre>
 *
 * @since 3.0
 */
@Keep
@NonExtensible
public interface GradleEnterpriseExtension {

    /**
     * The name of the extension added to the root project (for Gradle 5.x) or settings (for Gradle 6.0 and later).
     */
    String NAME = "gradleEnterprise";

    /**
     * The build scan extension.
     *
     * @return the build scan extension
     */
    BuildScanExtension getBuildScan();

    /**
     * Configures the build scan extension.
     *
     * @param action the configuration action
     */
    void buildScan(Action<? super BuildScanExtension> action);

    /**
     * Sets the URL of the Gradle Enterprise server.
     *
     * @param server the server URL
     * @since 3.2
     */
    void setServer(String server);

    /**
     * Returns the URL of the Gradle Enterprise server.
     *
     * @return null when no Gradle Enterprise server is configured
     * @since 3.2
     */
    @Nullable
    String getServer();

    /**
     * Specifies whether it is acceptable to communicate with a Gradle Enterprise server using an untrusted SSL certificate.
     * <p>
     * The default (public) build scan server uses SSL certificates that are trusted by default by standard modern Java environments.
     * If you are using a different build scan server via Gradle Enterprise, it may use an untrusted certificate.
     * This may be due to the use of an internally provisioned or self-signed certificate.
     * <p>
     * In such a scenario, you can either configure the build JVM environment to trust the certificate,
     * or call this method with {@code true} to disable verification of the server's identity.
     * Alternatively, you may disable SSL completely for Gradle Enterprise installation but this is not recommended.
     * <p>
     * Allowing communication with untrusted servers keeps data encrypted during transmission,
     * but makes it easy for a man-in-the-middle to impersonate the intended server and capture data.
     * <p>
     * This value has no effect if a server is specified using the HTTP protocol (i.e. has SSL disabled).
     *
     * @param allow whether to allow communication with a HTTPS server with an untrusted certificate
     * @since 3.2
     */
    void setAllowUntrustedServer(boolean allow);

    /**
     * Whether it is acceptable to communicate with a build scan server with an untrusted SSL certificate.
     *
     * @return <code>true</code> it is acceptable to communicate with a build scan server with an untrusted SSL certificate
     * @since 3.2
     **/
    boolean getAllowUntrustedServer();

    /**
     * Sets the access key for authenticating with the Gradle Enterprise server.
     * <p>
     * An access key configured this way will take precedence over the {@code GRADLE_ENTERPRISE_ACCESS_KEY}
     * environment variable or access key file entry associated with the server.
     *
     * @param accessKey a Gradle Enterprise server access key without any hostname prefix
     * @since 3.7
     */
    void setAccessKey(String accessKey);

    /**
     * Returns the access key for authenticating with the Gradle Enterprise server.
     * <p>
     * Only the value of the explicitly configured access key (via {@link #setAccessKey(String)}) is returned but
     * not the value of an access key configured via the {@code GRADLE_ENTERPRISE_ACCESS_KEY} environment variable
     * or access key file entry.
     *
     * @return the configured Gradle Enterprise server access key, if available; otherwise, {@code null}
     * @since 3.7
     */
    @Nullable
    String getAccessKey();

    /**
     * The type to be used for registering the Gradle Enterprise build cache with Gradle.
     * <p>
     * This should be used with {@link BuildCacheConfiguration#remote(Class, Action)} to enable
     * use of the Gradle Enterprise remote build caching.
     * <p>
     * The following example demonstrates usage in a build's settings script:
     * <p>
     * <pre>{@code
     * buildCache {
     *   remote(gradleEnterprise.buildCache) {
     *     // configure
     *   }
     * }
     * }</pre>
     *
     * @return the type to be used for registering the Gradle Enterprise build cache with Gradle
     * @see GradleEnterpriseBuildCache
     * @since 3.11
     */
    Class<? extends GradleEnterpriseBuildCache> getBuildCache();

}
