package com.gradle.scan.plugin;

import com.gradle.obfuscation.Keep;
import org.gradle.api.Action;
import org.gradle.api.NonExtensible;

import javax.annotation.Nullable;

/**
 * The DSL extension for configuring build scans, with the name {@code "buildScan"}.
 * <pre>{@code
 * buildScan {
 *     // build scan configuration
 * }
 * }</pre>
 */
@NonExtensible
@Keep
public interface BuildScanExtension {

    /**
     * Executes the given action in a background thread, allowing the current Gradle work to continue.
     * <p>
     * This method is useful for capturing values, tags and links that are expensive to compute.
     * By capturing them in the background, Gradle can continue doing other work, making your build faster.
     * <p>
     * For example, if you are capturing the Git commit ID as a custom value you should invoke Git in the background:
     * <pre>
     * buildScan.background {
     *   def commitId = 'git rev-parse --verify HEAD'.execute().text.trim()
     *   value "Git Commit ID", commitId
     * }
     * </pre>
     * <p>
     * All background work will be completed before finishing the build and publishing the build scan.
     * <p>
     * Any errors that are thrown by the background action will be logged and captured in the build scan.
     *
     * @param action the action to execute in the background
     * @since 1.15
     */
    void background(Action<? super BuildScanExtension> action);

    /**
     * Captures a tag for the current build. The tag is not required to be unique for a given build.
     *
     * @param tag the name of the tag, must not be null
     * @since 1.1
     */
    void tag(String tag);

    /**
     * Captures a named value for the current build. The name is not required to be unique for a given build.
     *
     * @param name  the name, must not be null
     * @param value the value, may be null
     * @since 1.1
     */
    void value(String name, String value);

    /**
     * Captures a named link for the current build. The name is not required to be unique for a given build.
     *
     * @param name the name, must not be null
     * @param url  the url, must not be null
     * @since 1.1
     */
    void link(String name, String url);

    /**
     * Registers a callback that is invoked when the build has finished, but <i>before</i> this extension stops
     * accepting to be called.
     *
     * @param action the action to execute when the build has finished
     * @since 1.2
     */
    void buildFinished(Action<? super BuildResult> action);

    /**
     * Registers a callback that is invoked when a build scan has been published successfully.
     * <p>
     * If {@link #isUploadInBackground()} is {@code true}, the callback is executed before the build scan is uploaded.
     * If the build scan is accessed immediately, it may not be available yet as it is still being uploaded to the server.
     *
     * @param action the action to execute when the build scan has been published successfully
     * @since 1.8
     */
    void buildScanPublished(Action<? super PublishedBuildScan> action);

    /**
     * The location of the Gradle Terms of Service that are agreed to when creating a build scan.
     *
     * @param termsOfServiceUrl the location of the Gradle Terms of Service
     * @since 1.12
     */
    void setTermsOfServiceUrl(String termsOfServiceUrl);

    /**
     * The location of the Gradle Terms of Service that are agreed to when creating a build scan.
     *
     * @return the location of the Gradle Terms of Service
     * @since 2.0
     */
    String getTermsOfServiceUrl();

    /**
     * Indicates whether the Gradle Terms of Service specified under {@link #setTermsOfServiceUrl(String)} are agreed to.
     *
     * @param agree <i>true</i> if agreeing to the Gradle Terms of Service, false otherwise
     * @since 1.12
     */
    void setTermsOfServiceAgree(String agree);

    /**
     * The agreement of the Gradle Terms of Service specified under {@link #setTermsOfServiceUrl(String)}.
     *
     * @return the agreement of the Gradle Terms of Service
     * @since 2.0
     */
    String getTermsOfServiceAgree();

    /**
     * Sets the URL of the Gradle Enterprise server to which the build scans are published.
     *
     * @param server the server URL
     * @since 1.0
     */
    void setServer(String server);

    /**
     * Returns the URL of the Gradle Enterprise server to which the build scans are published.
     *
     * @return null when no enterprise server is configured
     * @since 2.0
     */
    @Nullable
    String getServer();

    /**
     * Specifies whether it is acceptable to communicate with a Gradle Enterprise server using an untrusted SSL certificate.
     * <p>
     * The default (public) build scan server uses SSL certificates that are trusted by default by standard modern Java environments.
     * If you are using a different build scan server via Gradle Enterprise, it may use an untrusted certificate.
     * This may be due to the use of an internally provisioned or self-signed certificate.
     * <p>
     * In such a scenario, you can either configure the build JVM environment to trust the certificate,
     * or call this method with {@code true} to disable verification of the server's identity.
     * Alternatively, you may disable SSL completely for Gradle Enterprise installation but this is not recommended.
     * <p>
     * Allowing communication with untrusted servers keeps data encrypted during transmission,
     * but makes it easy for a man-in-the-middle to impersonate the intended server and capture data.
     * <p>
     * This value has no effect if a server is specified using the HTTP protocol (i.e. has SSL disabled).
     *
     * @param allow whether to allow communication with a HTTPS server with an untrusted certificate
     * @since 1.1
     */
    void setAllowUntrustedServer(boolean allow);

    /**
     * Whether it is acceptable to communicate with a build scan server with an untrusted SSL certificate.
     *
     * @return <code>true</code> it is acceptable to communicate with a build scan server with an untrusted SSL certificate
     * @since 2.0
     **/
    boolean getAllowUntrustedServer();

    /**
     * Indicates that a build scan should be published at the end of the build, regardless of whether
     * the build succeeded or failed.
     *
     * @since 1.1
     */
    void publishAlways();

    /**
     * Indicates that, if the given condition is true, a build scan should be published at the end of the build, regardless
     * of whether the build succeeded or failed.
     *
     * @param condition whether to publish
     * @since 1.1
     */
    void publishAlwaysIf(boolean condition);

    /**
     * Indicates that a build scan should be published at the end of the build, if and only if the build failed.
     *
     * @since 1.1
     */
    void publishOnFailure();

    /**
     * Indicates that, if the given condition is true, a build scan should be published at the end of the build, if
     * and only if the build failed.
     *
     * @param condition whether to publish on failure
     * @since 1.1
     */
    void publishOnFailureIf(boolean condition);

    /**
     * Specifies whether to upload the build scan in background after the build has finished.
     * <p>
     * Defaults to {@code true}.
     * <p>
     * This allows the build to finish sooner, but can be problematic in build environments that
     * terminate as soon as the build is finished as the upload may be terminated before it completes.
     * Background uploading should be disabled for such environments.
     * <p>
     * This setting can also be set by the {@code scan.uploadInBackground} system property,
     * which if set takes precedence over any value set by this method and the default.
     *
     * @param uploadInBackground whether to upload the build scan in background
     * @since 3.3
     */
    void setUploadInBackground(boolean uploadInBackground);

    /**
     * See {@link #setUploadInBackground(boolean)}.
     *
     * @return whether to upload build scan in background
     * @since 3.3
     */
    boolean isUploadInBackground();

    /**
     * Specifies whether to capture information about each file used as an input to a task.
     * <p>
     * Defaults to {@code false}.
     * <p>
     * Enabling this feature may increase the size of the build scan data.
     * This requires more time to transmit to the server, and more storage space at the server.
     * Most builds will not incur a noticeable difference when this feature is enabled.
     * Large builds may increase the build scan data by a handful of megabytes.
     * For most builds, the increase will be negligible.
     * <p>
     * If using Gradle Enterprise with a good connection to the server this capture should be enabled,
     * as it allows comparing task inputs at a file level when comparing builds.
     * <p>
     * This property may also be set by the {@code "scan.capture-task-input-files"} system property.
     * If this is set to any value other than {@code "false"}, the capture will be enabled.
     * If this is set to {@code "false"}, the capture will be disabled.
     * If the capture is enabled or disabled via system property, calling this method has no effect.
     * That is, the system property takes precedence over the value set via this method.
     * <p>
     * This method cannot be called after the root project has finished configuring.
     * Doing so will produce a build time error.
     *
     * @param capture whether to capture information about each file use as an input to a task
     * @since 2.1
     * @deprecated Please use {@link BuildScanCaptureSettings#setTaskInputFiles(boolean)}
     */
    @Deprecated
    void setCaptureTaskInputFiles(boolean capture);

    /**
     * See {@link #setCaptureTaskInputFiles(boolean)}.
     *
     * @return whether information about each file used as an input to a task will be captured
     * @since 2.1
     * @deprecated Please use {@link BuildScanCaptureSettings#isTaskInputFiles()}
     */
    @Deprecated
    boolean isCaptureTaskInputFiles();

    /**
     * Allows registering functions for obfuscating certain identifying information within build scans.
     *
     * @return the register of obfuscation functions
     * @since 2.4.2
     */
    BuildScanDataObfuscation getObfuscation();

    /**
     * Allows registering functions for obfuscating certain identifying information within build scans.
     *
     * @param action a function to be applied to the register of obfuscation functions
     * @since 2.4.2
     */
    void obfuscation(Action<? super BuildScanDataObfuscation> action);

    /**
     * Allows configuring what data will be captured as part of the build scan.
     *
     * @return the capture settings
     * @since 3.7
     */
    BuildScanCaptureSettings getCapture();

    /**
     * Allows configuring what data will be captured as part of the build scan.
     *
     * @param action a function to be applied to the capture settings
     * @since 3.7
     */
    void capture(Action<? super BuildScanCaptureSettings> action);

}
