/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.annoy;

import com.spotify.annoy.AnnoyIndex;
import com.spotify.annoy.IndexType;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.PriorityQueue;

public class ANNIndex
implements AnnoyIndex {
    private final ArrayList<Long> roots;
    private MappedByteBuffer[] buffers;
    private final int DIMENSION;
    private final int MIN_LEAF_SIZE;
    private final IndexType INDEX_TYPE;
    private final int INDEX_TYPE_OFFSET;
    private final int K_NODE_HEADER_STYLE;
    private final long NODE_SIZE;
    private final int INT_SIZE = 4;
    private final int FLOAT_SIZE = 4;
    private final int MAX_NODES_IN_BUFFER;
    private final int BLOCK_SIZE;
    private RandomAccessFile memoryMappedFile;

    public ANNIndex(int dimension, String filename, IndexType indexType) throws IOException {
        this(dimension, filename, indexType, 0);
    }

    public ANNIndex(int dimension, String filename) throws IOException {
        this(dimension, filename, IndexType.ANGULAR);
    }

    ANNIndex(int dimension, String filename, IndexType indexType, int blockSize) throws IOException {
        this.DIMENSION = dimension;
        this.INDEX_TYPE = indexType;
        this.INDEX_TYPE_OFFSET = this.INDEX_TYPE == IndexType.ANGULAR ? 4 : 8;
        this.K_NODE_HEADER_STYLE = this.INDEX_TYPE == IndexType.ANGULAR ? 12 : 16;
        this.MIN_LEAF_SIZE = this.DIMENSION + 2;
        this.NODE_SIZE = this.K_NODE_HEADER_STYLE + 4 * this.DIMENSION;
        this.MAX_NODES_IN_BUFFER = (int)(blockSize == 0 ? Integer.MAX_VALUE / this.NODE_SIZE : (long)blockSize * this.NODE_SIZE);
        this.BLOCK_SIZE = (int)((long)this.MAX_NODES_IN_BUFFER * this.NODE_SIZE);
        this.roots = new ArrayList();
        this.load(filename);
    }

    private void load(String filename) throws IOException {
        int blockSize;
        this.memoryMappedFile = new RandomAccessFile(filename, "r");
        long fileSize = this.memoryMappedFile.length();
        if (fileSize == 0L) {
            throw new IOException("Index is a 0-byte file?");
        }
        int numNodes = (int)(fileSize / this.NODE_SIZE);
        int buffIndex = (numNodes - 1) / this.MAX_NODES_IN_BUFFER;
        int rest = (int)(fileSize % (long)this.BLOCK_SIZE);
        int n = blockSize = rest > 0 ? rest : this.BLOCK_SIZE;
        if ((long)rest % this.NODE_SIZE != 0L || (fileSize - (long)rest) % this.NODE_SIZE != 0L) {
            throw new RuntimeException("ANNIndex initiated with wrong dimension size");
        }
        this.buffers = new MappedByteBuffer[buffIndex + 1];
        boolean process = true;
        int m = -1;
        long index = fileSize;
        for (long position = fileSize - (long)blockSize; position >= 0L; position -= (long)blockSize) {
            MappedByteBuffer annBuf = this.memoryMappedFile.getChannel().map(FileChannel.MapMode.READ_ONLY, position, blockSize);
            annBuf.order(ByteOrder.LITTLE_ENDIAN);
            this.buffers[buffIndex--] = annBuf;
            int i = blockSize - (int)this.NODE_SIZE;
            while (process && i >= 0) {
                index -= this.NODE_SIZE;
                int k = annBuf.getInt(i);
                if (m == -1 || k == m) {
                    this.roots.add(index);
                    m = k;
                } else {
                    process = false;
                }
                i = (int)((long)i - this.NODE_SIZE);
            }
            blockSize = this.BLOCK_SIZE;
        }
    }

    private float getFloatInAnnBuf(long pos) {
        int b = (int)(pos / (long)this.BLOCK_SIZE);
        int f = (int)(pos % (long)this.BLOCK_SIZE);
        return this.buffers[b].getFloat(f);
    }

    private int getIntInAnnBuf(long pos) {
        int b = (int)(pos / (long)this.BLOCK_SIZE);
        int i = (int)(pos % (long)this.BLOCK_SIZE);
        return this.buffers[b].getInt(i);
    }

    @Override
    public void getNodeVector(long nodeOffset, float[] v) {
        MappedByteBuffer nodeBuf = this.buffers[(int)(nodeOffset / (long)this.BLOCK_SIZE)];
        int offset = (int)(nodeOffset % (long)this.BLOCK_SIZE + (long)this.K_NODE_HEADER_STYLE);
        for (int i = 0; i < this.DIMENSION; ++i) {
            v[i] = nodeBuf.getFloat(offset + i * 4);
        }
    }

    @Override
    public void getItemVector(int itemIndex, float[] v) {
        this.getNodeVector((long)itemIndex * this.NODE_SIZE, v);
    }

    private float getNodeBias(long nodeOffset) {
        return this.getFloatInAnnBuf(nodeOffset + 4L);
    }

    @Override
    public final float[] getItemVector(int itemIndex) {
        return this.getNodeVector((long)itemIndex * this.NODE_SIZE);
    }

    public float[] getNodeVector(long nodeOffset) {
        float[] v = new float[this.DIMENSION];
        this.getNodeVector(nodeOffset, v);
        return v;
    }

    private static float norm(float[] u) {
        float n = 0.0f;
        for (float x : u) {
            n += x * x;
        }
        return (float)Math.sqrt(n);
    }

    private static float euclideanDistance(float[] u, float[] v) {
        float[] diff = new float[u.length];
        for (int i = 0; i < u.length; ++i) {
            diff[i] = u[i] - v[i];
        }
        return ANNIndex.norm(diff);
    }

    public static float cosineMargin(float[] u, float[] v) {
        double d = 0.0;
        for (int i = 0; i < u.length; ++i) {
            d += (double)(u[i] * v[i]);
        }
        return (float)(d / (double)(ANNIndex.norm(u) * ANNIndex.norm(v)));
    }

    public static float euclideanMargin(float[] u, float[] v, float bias) {
        float d = bias;
        for (int i = 0; i < u.length; ++i) {
            d += u[i] * v[i];
        }
        return d;
    }

    @Override
    public void close() throws IOException {
        this.memoryMappedFile.close();
    }

    private static boolean isZeroVec(float[] v) {
        for (int i = 0; i < v.length; ++i) {
            if (v[i] == 0.0f) continue;
            return false;
        }
        return true;
    }

    @Override
    public final List<Integer> getNearest(float[] queryVector, int nResults) {
        if (queryVector.length != this.DIMENSION) {
            throw new RuntimeException(String.format("queryVector must be size of %d, but was %d", this.DIMENSION, queryVector.length));
        }
        PriorityQueue<PQEntry> pq = new PriorityQueue<PQEntry>(this.roots.size() * 4);
        float kMaxPriority = 1.0E30f;
        for (long r : this.roots) {
            pq.add(new PQEntry(1.0E30f, r));
        }
        HashSet<Integer> nearestNeighbors = new HashSet<Integer>();
        while (nearestNeighbors.size() < this.roots.size() * nResults && !pq.isEmpty()) {
            PQEntry top = (PQEntry)pq.poll();
            long topNodeOffset = top.nodeOffset;
            int nDescendants = this.getIntInAnnBuf(topNodeOffset);
            float[] v = this.getNodeVector(topNodeOffset);
            if (nDescendants == 1) {
                if (ANNIndex.isZeroVec(v)) continue;
                nearestNeighbors.add((int)(topNodeOffset / this.NODE_SIZE));
                continue;
            }
            if (nDescendants <= this.MIN_LEAF_SIZE) {
                for (int i = 0; i < nDescendants; ++i) {
                    int j = this.getIntInAnnBuf(topNodeOffset + (long)this.INDEX_TYPE_OFFSET + (long)(i * 4));
                    if (ANNIndex.isZeroVec(this.getNodeVector((long)j * this.NODE_SIZE))) continue;
                    nearestNeighbors.add(j);
                }
                continue;
            }
            float margin = this.INDEX_TYPE == IndexType.ANGULAR ? ANNIndex.cosineMargin(v, queryVector) : ANNIndex.euclideanMargin(v, queryVector, this.getNodeBias(topNodeOffset));
            long childrenMemOffset = topNodeOffset + (long)this.INDEX_TYPE_OFFSET;
            long lChild = this.NODE_SIZE * (long)this.getIntInAnnBuf(childrenMemOffset);
            long rChild = this.NODE_SIZE * (long)this.getIntInAnnBuf(childrenMemOffset + 4L);
            pq.add(new PQEntry(-margin, lChild));
            pq.add(new PQEntry(margin, rChild));
        }
        ArrayList<PQEntry> sortedNNs = new ArrayList<PQEntry>();
        Iterator topNodeOffset = nearestNeighbors.iterator();
        while (topNodeOffset.hasNext()) {
            int nn = (Integer)topNodeOffset.next();
            float[] v = this.getItemVector(nn);
            if (ANNIndex.isZeroVec(v)) continue;
            sortedNNs.add(new PQEntry(this.INDEX_TYPE == IndexType.ANGULAR ? ANNIndex.cosineMargin(v, queryVector) : -ANNIndex.euclideanDistance(v, queryVector), nn));
        }
        Collections.sort(sortedNNs);
        ArrayList<Integer> result = new ArrayList<Integer>(nResults);
        for (int i = 0; i < nResults && i < sortedNNs.size(); ++i) {
            result.add((int)((PQEntry)sortedNNs.get(i)).nodeOffset);
        }
        return result;
    }

    public static void main(String[] args) throws IOException {
        String indexPath = args[0];
        int dimension = Integer.parseInt(args[1]);
        IndexType indexType = null;
        if (args[2].toLowerCase().equals("angular")) {
            indexType = IndexType.ANGULAR;
        } else if (args[2].toLowerCase().equals("euclidean")) {
            indexType = IndexType.EUCLIDEAN;
        } else {
            throw new RuntimeException("wrong index type specified");
        }
        int queryItem = Integer.parseInt(args[3]);
        ANNIndex annIndex = new ANNIndex(dimension, indexPath, indexType);
        float[] u = annIndex.getItemVector(queryItem);
        System.out.printf("vector[%d]: ", queryItem);
        for (float x : u) {
            System.out.printf("%2.2f ", Float.valueOf(x));
        }
        System.out.printf("\n", new Object[0]);
        List<Integer> nearestNeighbors = annIndex.getNearest(u, 10);
        for (int nn : nearestNeighbors) {
            float[] v = annIndex.getItemVector(nn);
            System.out.printf("%d %d %f\n", queryItem, nn, Float.valueOf(indexType == IndexType.ANGULAR ? ANNIndex.cosineMargin(u, v) : ANNIndex.euclideanDistance(u, v)));
        }
    }

    private class PQEntry
    implements Comparable<PQEntry> {
        private float margin;
        private long nodeOffset;

        PQEntry(float margin, long nodeOffset) {
            this.margin = margin;
            this.nodeOffset = nodeOffset;
        }

        @Override
        public int compareTo(PQEntry o) {
            return Float.compare(o.margin, this.margin);
        }
    }
}

