package com.xebialabs.deployit.plugin.apache.httpd.step;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.springframework.core.io.ClassPathResource;

import com.xebialabs.deployit.ResolutionException;
import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.util.ExtendedResourceUtils;
import com.xebialabs.deployit.util.TemplateResolver;

/**
 * Used to create a String representation of a VirtualHost definition for Apache Httpd. Also computes it's filename. e.g. www.xebialabs.com:443
 * 
 * NameVirtualHost will be *:443 DocumentRoot will be htdocsLocation + www.xebialabs.com_443 ServerName will be www.xebialabs.com
 * 
 */
public class ApacheVirtualHostDefinition {

	public static String DEFAULT_APACHE_VHOST_TEMPLATE = "com/xebialabs/deployit/plugin/apache/httpd/step/apache_httpd_server_vhost_template.conf";

	private String documentRoot;
	private String host;
	private String port;
	private String nameVirtualHost;
	private String serverName;
	private String fileName;
	private String vhostDefinition;
	private Collection<? extends Object> existingContext;

	/**
	 * Constructs a VirtualHostDefinition based on a single vhostDefinition.
	 * 
	 * @param vhostDefinition
	 */
	public ApacheVirtualHostDefinition(String vhostDefinition, String htdocsLocation, OperatingSystemFamily osf, Collection<?> existingContext)
			throws ResolutionException {
		this.existingContext = existingContext;
		if (vhostDefinition == null || vhostDefinition.equals("")) {
			throw new ResolutionException("No vhost definition provided");
		}
		if (htdocsLocation == null || htdocsLocation.equals("")) {
			throw new ResolutionException("No htdocs provided");
		}

		// Verify that the vhostDefinition is correct.
		int colonIndex = vhostDefinition.indexOf(':');
		if (colonIndex == -1 || colonIndex - 1 <= 0 || colonIndex + 1 >= vhostDefinition.length())
			throw new ResolutionException("VHostDefinition has a syntax error. Should be <hostname>:<port>. Currently is " + vhostDefinition);

		host = vhostDefinition.substring(0, vhostDefinition.indexOf(':'));
		port = vhostDefinition.substring(vhostDefinition.indexOf(':') + 1);

		nameVirtualHost = "*:" + port;
		serverName = host;

		String fileNameFriendlyPort = port.equals("*") ? "alports" : port;
		String fileNameFriendlyHost = host.equals("*") ? "allhosts" : host;
		fileName = fileNameFriendlyHost + "_" + fileNameFriendlyPort + ".conf";

		documentRoot = htdocsLocation;

		if (existingContext != null) {
			this.vhostDefinition = resolveVhostDefinition();
		}
	}

	protected String resolveVhostDefinition() {
		String template;
		List<Object> context = new ArrayList<Object>();
		context.addAll(existingContext);
		context.add(this);
		template = ExtendedResourceUtils.toString(new ClassPathResource(DEFAULT_APACHE_VHOST_TEMPLATE));
		TemplateResolver r = new TemplateResolver(context);
		return r.resolveStrict(template);
	}

	public String toVirtualHostDefinition() {
		return vhostDefinition;
	}

	public String getDocumentRoot() {
		return documentRoot;
	}

	public String getHost() {
		return host;
	}

	public void setHost(String host) {
		this.host = host;
	}

	public String getPort() {
		return port;
	}

	public void setPort(String port) {
		this.port = port;
	}

	public String getNameVirtualHost() {
		return nameVirtualHost;
	}

	public String getServerName() {
		return serverName;
	}

	public String getFileName() {
		return fileName;
	}

	public void setDocumentRoot(String documentRoot) {
		this.documentRoot = documentRoot;
	}

	public void setNameVirtualHost(String nameVirtualHost) {
		this.nameVirtualHost = nameVirtualHost;
	}

	public void setServerName(String serverName) {
		this.serverName = serverName;
	}

	public void setFileName(String fileName) {
		this.fileName = fileName;
	}

}
