package com.xebialabs.deployit.core.rest.api

import com.xebialabs.deployit.core.rest.api.support.PaginationSupport
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.engine.api.RoleService
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.engine.api.security.RolePrincipals
import com.xebialabs.deployit.engine.spi.event._
import com.xebialabs.deployit.event.EventBusHolder
import com.xebialabs.deployit.security.permission.PlatformPermissions.{EDIT_SECURITY, VIEW_SECURITY}
import com.xebialabs.deployit.security.{Permissions, Role}
import grizzled.slf4j.Logging
import org.jboss.resteasy.spi.HttpResponse
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import java.util
import java.util.stream.Collectors
import java.util.stream.Collectors.toList
import javax.ws.rs.core.Context
import scala.jdk.CollectionConverters._

@Controller class RoleResource @Autowired()(service: com.xebialabs.deployit.security.RoleService)
  extends AbstractSecuredResource with RoleService with Logging {

  private val dtoWriter = new DtoWriter

  @Context val response: HttpResponse = null

  override def exists(roleName: String): Boolean = service.roleExists(roleName)

  override def list(id: String, rolePattern: String, paging: Paging, order: Ordering): util.List[com.xebialabs.deployit.engine.api.security.Role] = {
    if (rolePattern == null && (paging == null || paging.resultsPerPage == -1) && order == null) {
      dtoWriter.writeRoles(service.getRoles(id))
    }
    else {
      PaginationSupport.addTotalCountHeader(service.countRoles(id, rolePattern), response)
      dtoWriter.writeRoles(service.getRoles(id, rolePattern, paging, order))
    }
  }

  override def list(rolePattern: String, paging: Paging, order: Ordering): util.List[String] = {
    if (rolePattern == null && (paging == null || paging.resultsPerPage == -1) && order == null) {
      service.getRoles().stream.map[String](_.getName).collect(toList[String])
    }
    else {
      PaginationSupport.addTotalCountHeader(service.countRoles(-1, rolePattern), response)
      service.getRoles(rolePattern, paging, order).stream.map[String](_.getName).collect(toList[String])
    }
  }

  override def listMyRoles(rolePattern: String, paging: Paging, order: Ordering): util.List[String] =
    service.getRolesFor(Permissions.getAuthentication, rolePattern, paging, order).stream.map[String](_.getName).collect(toList[String])

  override def listRoles(username: String, rolePattern: String, paging: Paging, order: Ordering): util.List[String] = {
    checkPermissions(EDIT_SECURITY, VIEW_SECURITY)
    service.getRolesFor(username, rolePattern, paging, order).stream.map[String](_.getName).collect(toList[String])
  }

  override def create(name: String): Unit = {
    checkPermission(EDIT_SECURITY)
    sanityCheckRole(Option(name))
    if (!service.roleExists(name)) {
      service.create(name)
      EventBusHolder.publish(new RoleCreatedEvent(name))
    }
  }

  override def updatePrincipals(rolePrincipals: RolePrincipals): Unit = {
    updatePrincipals(rolePrincipals.getRole.getId, rolePrincipals.getRole.getName, rolePrincipals.getPrincipals)
  }

  override def updatePrincipals(name: String, principals: util.List[String]): Unit = {
    updatePrincipals(null, name, principals)
  }

  private def updatePrincipals(id: String, name: String, principals: util.List[String]): Unit = {
    checkPermission(EDIT_SECURITY)
    sanityCheckRole(Option(name))
    val role = new Role(id, name, principals)
    service.createOrUpdateRole(role)
    fireRolePrincipalsChangedEvent(service.readRoleAssignments())
  }

  override def assign(name: String, principal: String): Unit = {
    val role: Role = service.getRoleForRoleName(name)
    val newPrincipals = role.getPrincipals.asScala :+ principal
    updatePrincipals(name, newPrincipals.asJava)
    EventBusHolder.publish(new PrincipalRoleAssignmentCreatedEvent(principal, name))
  }

  override def unassign(name: String, principal: String): Unit = {
    val role: Role = service.getRoleForRoleName(name)
    val newPrincipals = role.getPrincipals.asScala.filter(p => p != principal);
    updatePrincipals(name, newPrincipals.asJava)
    EventBusHolder.publish(new PrincipalRoleAssignmentDeletedEvent(principal, name))
  }

  override def rename(name: String, newName: String): Unit = {
    checkPermission(EDIT_SECURITY)
    sanityCheckRole(Option(newName))
    service.rename(name, newName)
    EventBusHolder.publish(new RoleRenamedEvent(name, newName))
  }

  override def delete(name: String): Unit = {
    checkPermission(EDIT_SECURITY)
    service.deleteByName(name)
    EventBusHolder.publish(new RoleDeletedEvent(name))
  }

  override def readRolePrincipals(rolePattern: String, paging: Paging, order: Ordering): util.List[RolePrincipals] = {
    checkPermissions(EDIT_SECURITY, VIEW_SECURITY)
    var multimap: util.List[Role] = null
    if (rolePattern == null && (paging == null || paging.resultsPerPage == -1) && order == null) {
      multimap = service.readRoleAssignments()
    }
    else {
      multimap = service.readRoleAssignments(rolePattern, paging, order)
      val total = service.countRoles(-1, rolePattern)
      PaginationSupport.addTotalCountHeader(total, response)
    }
    logger.debug(s"com.xebialabs.deployit.security.RoleService delivered roles: $multimap")
    val assignments = dtoWriter.writeRoleAssignments(multimap)
    logger.debug(s"Going to return the following roles: $assignments")
    assignments
  }

  private def fireRolePrincipalsChangedEvent(rolePrincipals: util.List[Role]): Unit = {
    EventBusHolder.publish(new RolePrincipalsChangedEvent(rolePrincipals.stream.map[String]((r: Role) => r.getName + " => " + r.getPrincipals).collect(Collectors.toList[String])))
  }

  private def sanityCheckRole(roleName: Option[String]): Unit = {
    roleName.filter(_.trim.nonEmpty).getOrElse(throw new IllegalArgumentException("Role name cannot be empty"))
  }
}
