package com.xebialabs.deployit.core.rest.util;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.engine.api.RoleService;
import com.xebialabs.deployit.engine.spi.command.RepositoryBaseCommand;
import com.xebialabs.deployit.engine.spi.command.UpdateCisCommand;
import com.xebialabs.deployit.engine.spi.command.util.Update;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.repository.CiRepositoryUtils;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData;
import nl.javadude.t2bus.event.strategy.ThrowingRuntimeExceptionHandlerStrategy;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import java.util.*;
import java.util.function.Predicate;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.READ;
import static java.util.stream.Collectors.*;

public class RepositoryHelper {
    private RepositoryService repositoryService;
    private List<Type> allowedConvertibleTypes;
    private RoleService roleService;

    @Autowired
    public RepositoryHelper(RepositoryService repositoryService,
                            RoleService roleService,
                            @Value("#{allowedConvertibleTypes}") List<Type> allowedConvertibleTypes) {
        this.repositoryService = repositoryService;
        this.roleService = roleService;
        this.allowedConvertibleTypes = allowedConvertibleTypes;
    }

    @SuppressWarnings("unchecked")
    private <T> T getOrNull(ConfigurationItem ci, PropertyDescriptor pd) {
        if (ci == null) {
            return null;
        }
        return (T) pd.get(ci);
    }

    private boolean hasPermission(Permission permission, String id) {
        return permission.getPermissionHandler().hasPermission(id);
    }

    public boolean isFileUriNotChanged(final ConfigurationItem ci, ConfigurationItem previous) {
        if (ci instanceof SourceArtifact && previous instanceof SourceArtifact) {
            return ((SourceArtifact) ci).getFileUri() == null ||
                    ((SourceArtifact) ci).getFileUri().equals(((SourceArtifact) previous).getFileUri());
        }
        return true;
    }


    public void checkIfUpdatedReadonlyProperty(final ConfigurationItem ci, ConfigurationItem previous) {
        String propertyDescriptors = ci.getType().getDescriptor().getPropertyDescriptors()
                .stream()
                .filter(PropertyDescriptor::isReadonly)
                .filter(pd -> !pd.areEqual(ci, previous))
                .map(PropertyDescriptor::getName).collect(joining(","));
        if (!propertyDescriptors.isEmpty()) {
            throw new Checks.IncorrectArgumentException("Readonly properties [%s] updated. You can only set readonly properties when creating the configuration item.", propertyDescriptors);
        }
    }

    public void checkIfConvertibleType(ConfigurationItem ci, ConfigurationItem previous) {
        if (!previous.getType().equals(ci.getType())) {
            Predicate<Type> instanceOfAllowedType = type -> previous.getType().instanceOf(type) && ci.getType().instanceOf(type);
            Checks.checkArgument(allowedConvertibleTypes.stream().anyMatch(instanceOfAllowedType),
                    "Type of the Configuration Item cannot be changed from [%s] to [%s]", previous.getType(), ci.getType());
        }
    }

    public void publishCommand(final RepositoryBaseCommand event) {
        List<String> roles = roleService.listMyRoles(null, null, null);
        event.setSecurityContext(Permissions.getAuthenticatedUserName(), roles);
        EventBusHolder.publish(event, new ThrowingRuntimeExceptionHandlerStrategy());
    }

    public List<ConfigurationItem> reloadEntities(Collection<ConfigurationItem> cis) {
        return cis.stream()
                .map(ConfigurationItem::getId)
                .map(id -> (ConfigurationItem) repositoryService.read(id, 1))
                .collect(toList());
    }

    @SuppressWarnings("unchecked")
    public void checkReadAccessOnRelations(ConfigurationItem existingCi, ConfigurationItem updatedCi, List<String> nonReadIdAggregator, Set<String> otherCiIdsInTransaction) {
        Collection<PropertyDescriptor> propertyDescriptors = updatedCi.getType().getDescriptor().getPropertyDescriptors();
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            if (propertyDescriptor.isAsContainment()) {
                continue;
            }

            switch (propertyDescriptor.getKind()) {
                case CI:
                    ConfigurationItem linkedCi = (ConfigurationItem) propertyDescriptor.get(updatedCi);
                    if (linkedCi == null) {
                        break;
                    }
                    String id = linkedCi.getId();
                    ConfigurationItem existingLinkedCi = getOrNull(existingCi, propertyDescriptor);
                    if (
                        // Don't check if the link was pre-existing...
                            (existingLinkedCi != null && existingLinkedCi.getId().equals(id)) ||
                                    // ... or the linked CI is part of this transaction.
                                    (otherCiIdsInTransaction.contains(id))
                            ) {
                        break;
                    } else if (!hasPermission(READ, id)) {
                        nonReadIdAggregator.add(id);
                    }
                    break;
                case SET_OF_CI:
                case LIST_OF_CI:
                    Collection<ConfigurationItem> cis = (Collection<ConfigurationItem>) propertyDescriptor.get(updatedCi);
                    Collection<ConfigurationItem> existingCis = getOrNull(existingCi, propertyDescriptor);
                    Collection<String> existingLinkedCiIds = existingCis == null ? new ArrayList<>() : existingCis.stream().map(ConfigurationItem::getId).collect(toList());
                    for (ConfigurationItem configurationItem : cis) {
                        if (configurationItem != null
                                // Don't check if the link was pre-existing...
                                && !existingLinkedCiIds.contains(configurationItem.getId())
                                // ... or the linked CI is part of this transaction.
                                && !otherCiIdsInTransaction.contains(configurationItem.getId())
                                && !hasPermission(READ, configurationItem.getId())) {
                            nonReadIdAggregator.add(configurationItem.getId());
                        }
                    }
                    break;
            }
        }
    }

    public List<Update> createOrUpdateCis(List<ConfigurationItem> cis,
                                          Optional<Set<String>> maybeExistingIds,
                                          SCMTraceabilityData traceabilityData) {
        List<Update> updates = new ArrayList<>();

        Set<String> ciIds = cis.stream().map(ConfigurationItem::getId).collect(toSet());
        Set<String> existingIds = maybeExistingIds.orElseGet(() -> ciIds.stream().filter(repositoryService::exists).collect(toSet()));
        List<String> aggregator = new ArrayList<>();
        cis.stream().filter((configurationItem) -> {
            // Add to updates if needed
            if (existingIds.contains(configurationItem.getId())) {
                ConfigurationItem previous = repositoryService.read(configurationItem.getId(), 1, null, false);
                if (isFileUriNotChanged(configurationItem, previous)) {
                    checkIfConvertibleType(configurationItem, previous);
                    checkReadAccessOnRelations(previous, configurationItem, aggregator, ciIds);
                    checkIfUpdatedReadonlyProperty(configurationItem, previous);
                    updates.add(new Update(previous, configurationItem));
                } else {
                    return false;
                }
            } else {
                CiRepositoryUtils.checkApplicationOnUniqueness(configurationItem);
                checkReadAccessOnRelations(null, configurationItem, aggregator, ciIds);
                updates.add(new Update(null, configurationItem));
            }
            return true;
        }).collect(toList());

        if (!aggregator.isEmpty()) {
            throw new PermissionDeniedException("Permission READ is not granted on the following linked CIs: " + aggregator);
        }

        UpdateCisCommand command = new UpdateCisCommand(updates, traceabilityData);
        publishCommand(command);
        return updates;
    }

    public List<ConfigurationItem> createOrUpdateAndReloadCis(List<ConfigurationItem> cis, Optional<Set<String>> maybeExistingIds) {
        List<Update> updates = createOrUpdateCis(cis, maybeExistingIds, null);
        return reloadEntities(updates.stream().map(Update::getNewCi).collect(toList()));
    }
}
