/**
 * Copyright 2014-2015 XebiaLabs B.V. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.jetty;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.xebialabs.deployit.security.Permissions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;

import com.xebialabs.deployit.ServerState;
import com.xebialabs.deployit.security.permission.PermissionHelper;

/**
 * Intercept all requests configured via Spring MVC.
 * <p/>
 * If server is put into MAINTENANCE mode it will reject requests.
 */
public class MaintenanceModeFilter implements Filter {
    public static final String PARAMETER_MAINTENANCE_FORBIDDEN_REQUEST = "maintenance.forbidden_paths";
    public static final String PARAMETER_CONTEXT_ROOT = "maintenance.context_root";

    private static final Logger logger = LoggerFactory.getLogger(MaintenanceModeFilter.class);

    private List<PathAndMethod> forbiddenPaths = new ArrayList<>();

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        // load server config
        String contextRoot = filterConfig.getInitParameter(PARAMETER_CONTEXT_ROOT);
        if (!contextRoot.endsWith("/")) {
            contextRoot = contextRoot + "/";
        }
        String pathsParam = filterConfig.getInitParameter(PARAMETER_MAINTENANCE_FORBIDDEN_REQUEST);
        forbiddenPaths = PathAndMethod.fromRequestParam(contextRoot, pathsParam);
    }

    @Override
    public void destroy() {
        // nothing
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        HttpServletResponse httpResponse = (HttpServletResponse) response;

        String method = httpRequest.getMethod();
        String requestUri = httpRequest.getRequestURI();
        ServerState serverState = ServerState.getInstance();
        boolean inMaintenanceMode = ServerState.Mode.MAINTENANCE == serverState.getCurrentMode();
        if (inMaintenanceMode) {
            logger.trace("Processing '{}' request for '{}' in MAINTENANCE mode.", method, requestUri);
            Authentication authentication = Permissions.getAuthentication();
            boolean hasAuthentication = null != authentication;
            // admins are allowed to execute anything in MAINTENANCE mode
            if (hasAuthentication && !PermissionHelper.isCurrentUserAdmin()) {
                if (isForbiddenInMaintenanceMode(httpRequest)) {
                    httpResponse.getWriter().write("Server is in MAINTENANCE mode (shutdown pending).");
                    httpResponse.setContentType("text/html");
                    httpResponse.setStatus(HttpServletResponse.SC_SERVICE_UNAVAILABLE);
                    return;
                }
            }
        }
        chain.doFilter(request, response);
    }

    private boolean isForbiddenInMaintenanceMode(HttpServletRequest request) {
        boolean isRequestAllowed = false;
        final String uri = request.getRequestURI();
        final String method = request.getMethod();
        for (PathAndMethod pam : forbiddenPaths) {
            if (pam.matches(uri, method)) {
                isRequestAllowed = true;
                break;
            }
        }

        return isRequestAllowed;
    }

    static class PathAndMethod {
        private String path;
        private String method;

        private PathAndMethod() {
        }

        private PathAndMethod(String path, String method) {
            this.path = path;
            this.method = method;
        }

        static public PathAndMethod parse(final String pathAndMethod) {
            String[] paths = pathAndMethod.split(":");
            if (paths.length == 2) {
                String p = paths[0];
                String m = paths[1];
                return new PathAndMethod(p, m);
            } else {
                return new PathAndMethod();
            }
        }

        public static List<PathAndMethod> fromRequestParam(final String contextRoot, final String paths) {
            List<PathAndMethod> pathList = new ArrayList<>();
            if (null != paths) {
                String[] pathAndMethodArray = paths.split(",");
                for (String pathAndMethodValue : pathAndMethodArray) {
                    pathAndMethodValue = pathAndMethodValue.trim();
                    if (pathAndMethodValue.length() > 0) {
                        pathAndMethodValue = contextRoot + pathAndMethodValue;
                        PathAndMethod pathAndMethod = PathAndMethod.parse(pathAndMethodValue);
                        pathList.add(pathAndMethod);
                    }
                }
            }
            return pathList;
        }

        public boolean matches(String requestUri, String requestMethod) {
            boolean retval = false;

            if (null != method && null != path) {
                boolean methodMatches = "*".equals(method) || method.equals(requestMethod);
                boolean requestUriStartsWithPath = requestUri.startsWith(path);
                if (methodMatches && requestUriStartsWithPath) {
                    retval = true;
                }
            }

            return retval;
        }

    }

}
