package com.xebialabs.deployit.jetty;

import com.xebialabs.deployit.rest.mappers.DeployitExceptionMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * Filter that makes Rest compatible with the broken Flex implementation
 * - Return 20X status regardless of what the status was, original status stored in X-Original-Status Header
 * -
 */
public class FlexRestCompatibilityFilter implements Filter {
    protected static final String X_FLEX_REST = "X-Flex-Rest";
    public static final String X_ORIGINAL_STATUS_HEADER = "X-Original-Status";
    public static final String X_METHOD_OVERRIDE_HEADER = "X-HTTP-Method-Override";

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        logger.info("Added FlexRestCompatibilityFilter to the mix.");
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        boolean installResponseCompatibilityLayer = false;
        if (request instanceof HttpServletRequest) {
            HttpServletRequest hsr = (HttpServletRequest) request;
            String brokenRest = hsr.getHeader(X_FLEX_REST);
            installResponseCompatibilityLayer = (brokenRest != null && !"".equals(brokenRest));
        }

        request = new FlexRestCompatibilityServletRequest((HttpServletRequest) request, X_METHOD_OVERRIDE_HEADER);

        if (installResponseCompatibilityLayer) {
            logger.debug("Setting Flex Compatibility mode.");
            response = new FlexRestCompatibilityServletResponse((HttpServletResponse) response);
            response.setContentType("application/xml; charset=UTF-8");
            ((FlexRestCompatibilityServletResponse) response).writePreamble();
        }

        try {
            DeployitExceptionMapper.doCDataWrapping(installResponseCompatibilityLayer);
            chain.doFilter(request, response);
        } finally {
            DeployitExceptionMapper.doCDataWrapping(false);
        }

        if (installResponseCompatibilityLayer) {
            ((FlexRestCompatibilityServletResponse) response).writePostamble();
        }
    }

    @Override
    public void destroy() {
        // Empty
    }

    private static final Logger logger = LoggerFactory.getLogger(FlexRestCompatibilityFilter.class);
}
