package com.xebialabs.deployit.jetty;

import java.util.Collections;
import java.util.Enumeration;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FlexRestCompatibilityServletRequest extends HttpServletRequestWrapper {

    public static final String X_HEADER_PREFIX = "X-Header-";

    private final String methodOverrideHeader;

    private transient String method;

    /**
     * @param request
     *            the original request
     * @param methodOverrideHeader
     *            the name of the header that can be used to specify an
     *            overriding method
     */
    public FlexRestCompatibilityServletRequest(HttpServletRequest request, String methodOverrideHeader) {
        super(request);
        this.methodOverrideHeader = methodOverrideHeader;
    }

    @Override
    public String getMethod() {
        if (method == null) {
            method = resolveMethod();
        }
        return method;
    }

    protected String resolveMethod() {
        String headerValue = getHeader(methodOverrideHeader);
        String method;

        if (headerValue != null) {
            method = headerValue;
            logger.debug("Overriding " + super.getMethod() + " request to be a " + method + " request because a " + methodOverrideHeader
                    + " header was present");
        } else {
            method = super.getMethod();
            logger.debug("Not overriding a " + method + " request");
        }
        return method;
    }

    //
    // Header overrides for Flex calls that can't set custom headers, like
    // FileReference.download()
    //

    @Override
    public String getHeader(String name) {
        String header = getHeaderFromParameters(name);
        if (header != null) {
            logger.debug("Header {} resolved from request parameters", name);
        } else {
            header = super.getHeader(name);
        }

        return header;
    }

    private String getHeaderFromParameters(String name) {
        String headerAsParam = X_HEADER_PREFIX + name;
        return getParameter(headerAsParam);
    }

    @Override
    public String getContentType() {
        String contentType = getHeaderFromParameters("Content-Type");
        if (contentType == null) {
            contentType = super.getContentType();
        }
        return contentType;
    }

    @Override
    @SuppressWarnings("unchecked")
    public Enumeration<String> getHeaders(String name) {
        List<String> headers = Collections.list((Enumeration<String>) super.getHeaders(name));
        String headerFromParam = getHeaderFromParameters(name);
        if (headerFromParam != null) {
            logger.debug("Header {} resolved from request parameters", name);
            headers.add(headerFromParam);
        }
        return Collections.enumeration(headers);
    }

    @Override
    @SuppressWarnings("unchecked")
    public Enumeration<String> getHeaderNames() {
        List<String> headers = Collections.list((Enumeration<String>) super.getHeaderNames());
        Enumeration<String> params = getParameterNames();
        while (params.hasMoreElements()) {
            String param = params.nextElement();
            if (param.startsWith(X_HEADER_PREFIX)) {
                String headerInParam = param.substring(X_HEADER_PREFIX.length());
                if (!headers.contains(headerInParam)) {
                    logger.debug("Header {} resolved from request parameters", headerInParam);
                    headers.add(headerInParam);
                }
            }
        }
        return Collections.enumeration(headers);
    }

    private static Logger logger = LoggerFactory.getLogger(FlexRestCompatibilityServletRequest.class);

}
