/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.jetty;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.util.Objects;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.checks.Checks;

import org.eclipse.jetty.ee10.servlet.DefaultServlet;
import org.eclipse.jetty.util.resource.Resource;
import org.eclipse.jetty.util.resource.ResourceFactory;
import org.eclipse.jetty.util.resource.URLResourceFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

// TODO jetty12 - this class is used by Deploy only
@SuppressWarnings("serial")
public class ClassPathResourceContentServlet extends DefaultServlet {
    private static final String FALLBACK_SYSTEM_PROPERTY_NAME = "com.xebialabs.deployit.jetty.ClassPathResourceContentServlet.fallbackDirectory";

    private final String packageName;
    private final boolean fallbackToDevelopmentResources;
    private final String fallbackDirectory;

    public ClassPathResourceContentServlet(String packageName) {
        Checks.checkNotNull(packageName, "packageName");
        this.packageName = packageName.replace('.', '/');
        fallbackDirectory = System.getProperty(FALLBACK_SYSTEM_PROPERTY_NAME);
        fallbackToDevelopmentResources = Strings.isNotEmpty(fallbackDirectory);
        if (logger.isDebugEnabled()) {
            logger.debug("Using {} as the document root on the classpath (fallbackToDevelopmentResources={}).", this.packageName, fallbackToDevelopmentResources);
        }
    }


    public Resource getResource(String pathInContext) {
        pathInContext = pathInContext.replace("-@project.version@", "");
        final String resourcePath = packageName + pathInContext;
        if (logger.isDebugEnabled()){
            logger.debug("Requested resource {} to load", resourcePath);
        }
        URL url = Thread.currentThread().getContextClassLoader().getResource(resourcePath);
        try {
            Resource r = null;
            ResourceFactory resourceFactory = new URLResourceFactory();
            if (url != null) {
                if(Objects.equals(url.getProtocol(), "xlp")) {
                    r = new XlpUrlResource(url);
                } else {
                    r = resourceFactory.newResource(url);
                }
            } else {
                if (fallbackToDevelopmentResources) {
                    String fallbackUrl = "file:" + fallbackDirectory + pathInContext;
                    logger.info("Cannot find {} in classpath, falling back to URL {}", resourcePath, fallbackUrl);
                    r = resourceFactory.newResource(fallbackUrl);
                }
            }
            if (r == null || !r.exists()) {
                if (resourcePath.endsWith(".gz")) {
                    logger.debug("Cannot find {} in classpath", resourcePath);
                } else {
                    logger.error("Cannot find {} in classpath", resourcePath);
                }
            }
            return r;
        } catch (IOException exc) {
            throw new RuntimeException(exc);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(ClassPathResourceContentServlet.class);

    private static class XlpUrlResource extends Resource {
        private final Resource delegate;

        public XlpUrlResource(final URL url) throws IOException {
            super();
            URLResourceFactory resourceFactory = new URLResourceFactory();
            try {
                delegate = resourceFactory.newResource(url.toURI());
            } catch (URISyntaxException e) {
                throw new IOException(e);
            }
        }

        public File getFile() throws IOException {
            return null;
        }

        @Override
        public Path getPath() {
            return delegate.getPath();
        }

        @Override
        public boolean isDirectory() {
            return delegate.isDirectory();
        }

        @Override
        public boolean isReadable() {
            return delegate.isReadable();
        }

        @Override
        public URI getURI() {
            return delegate.getURI();
        }

        @Override
        public String getName() {
            return delegate.getName();
        }

        @Override
        public String getFileName() {
            return delegate.getFileName();
        }

        @Override
        public Resource resolve(final String s) {
            return delegate.resolve(s);
        }
    }
}
