package com.xebialabs.ascode.yaml.parser.support


import com.fasterxml.jackson.databind.{JsonNode, ObjectMapper}
import com.xebialabs.ascode.yaml.parser.util.JsonNodeSugar._
import com.xebialabs.ascode.yaml.parser.util.{AlwaysValidNodeValidator, IdResolver, IdentityIdResolver, NodeValidator}
import com.xebialabs.deployit.plugin.api.udm.lookup.LookupValueKey
import com.xebialabs.deployit.plugin.api.udm.{CiAttributes, ExternalProperty}
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage
import com.xebialabs.xltype.serialization.{CiListReader, CiReader}

import scala.jdk.CollectionConverters._
import java.util

case class ReaderConfig(overrides: Map[String, String] = Map(),
                        reserved: Set[String] = Set("id", "type"),
                        idResolver: IdResolver = IdentityIdResolver,
                        nodeValidator: NodeValidator = AlwaysValidNodeValidator) {
  val reversedOverrides: Map[String, String] = overrides.map { case (key, value) => value -> key }
}

case class ReaderField(internalField: String, ciField: String)

class JacksonCiReader(root: JsonNode,
                      parentReader: Option[CiReader],
                      config: ReaderConfig)
                     (implicit mapper: ObjectMapper) extends CiReader {
  config.nodeValidator.validate(config, Option(getId), Option(getType), root)

  override def getExternalProperties: util.Map[String, ExternalProperty] = {
    val result = new util.HashMap[String, ExternalProperty]
    val externalProperty =  root.mapStringAnyRef("external-properties").asJava
    if (!externalProperty.isEmpty) {
      externalProperty.forEach((key, valu) => {
        val value = valu.asInstanceOf[util.Map[String, String]]
          if(value.containsKey("kind") && value.get("kind").equals("lookup")) {
            val lookupValueKey = new LookupValueKey
            lookupValueKey.setKey(value.get("key"))
            lookupValueKey.setProviderId(value.get("provider"))
            result.put(key, lookupValueKey)
          } else {
            throw new IllegalArgumentException("Unknown external property kind: '" + value.get("kind") + "'.")
          }
        })
    }
    result
  }

  private val fieldNamesIterator = root
    .fieldNames()
    .asScala
    .withFilter(key => !config.reserved.contains(key) && !key.startsWith("$"))

  private var currentField: ReaderField = _

  private def staticStringField(name: String)(implicit context: JsonNode = root) = {
    val field = config
      .reversedOverrides
      .getOrElse(name, name)
    context.string(field)
  }

  private def resolveId(id: String) = config.idResolver.resolveCiId(id, getType, parentReader.map(_.getId))

  private def resolveRefId(id: String) = config.idResolver.resolveReferenceId(id, parentReader.map(_.getId))

  override def getType: String = staticStringField("type").orNull

  override def getId: String = staticStringField("id").map(resolveId).orNull

  override def getToken: String = root.string("$token").orNull

  override def getCiAttributes: CiAttributes = {
    val createdBy = root.string("$createdBy").orNull
    val createdAt = root.dateTime("$createdAt").orNull
    val lastModifiedBy = root.string("$lastModifiedBy").orNull
    val lastModifiedAt = root.dateTime("$lastModifiedAt").orNull
    new CiAttributes(createdBy, createdAt, lastModifiedBy, lastModifiedAt, null)
  }

  override def hasMoreProperties: Boolean = fieldNamesIterator.hasNext

  override def moveIntoProperty(): Unit = {
    val internalField = fieldNamesIterator.next()
    val field = ReaderField(internalField, config.overrides.getOrElse(internalField, internalField))
    this.currentField = field
  }

  override def moveIntoNestedProperty(): CiReader = new JacksonCiReader(
    root.property(currentField.internalField).get, Some(this), config
  )

  override def moveOutOfProperty(): Unit = {}

  override def getCurrentPropertyName: String = currentField.ciField

  override def getStringValue: String = root.string(currentField.internalField).get

  override def getStringValues: util.List[String] = root.listOfStrings(currentField.internalField).asJava

  override def getStringMap: util.Map[String, String] = root.mapStringString(currentField.internalField).asJava

  override def isCiReference: Boolean = currentField != null && root.property(currentField.internalField).exists(_.isTextual)

  override def getCiReference: String = root.string(currentField.internalField).map(resolveRefId).get

  override def getCiReferences: util.List[String] = root.listOfStrings(currentField.internalField).map(resolveRefId).asJava

  override def getCurrentCiListReader: CiListReader =
    new JacksonCiListReader(root.property(currentField.internalField).get, Some(this), config)

  override def getValidationMessages: util.List[ValidationMessage] = root.property("validation-messages")
    .get
    .iterator()
    .asScala
    .toList
    .map(message => new ValidationMessage(
        message.string("ci").get,
        message.string("property").orNull,
        message.string("message").get,
        message.string("level").orNull
    ))
    .asJava
}
