package com.xebialabs.deployit.ascode.service.spec

import com.xebialabs.ascode.service.spec.{InterpreterContext, SpecInterpreter}
import com.xebialabs.ascode.yaml.dto.AsCodeResponse
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.TaskInfo
import com.xebialabs.deployit.ascode.service.spec.util.DeploymentPreparationService
import com.xebialabs.deployit.ascode.yaml.model.DeploymentSpec
import com.xebialabs.deployit.engine.api.dto.Deployment
import com.xebialabs.deployit.engine.api.dto.Deployment.DeploymentType
import com.xebialabs.deployit.service.deployment.DeploymentTaskService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
@Autowired
class DeploymentSpecInterpreter(deploymentTaskService: DeploymentTaskService,
                                deploymentPreparationService: DeploymentPreparationService) extends SpecInterpreter {
  override def isDefinedAt(context: InterpreterContext): Boolean = context.definition.spec.isInstanceOf[DeploymentSpec]

  private def prepareAndStartTask(deployment: Deployment): String = {
    val taskId = deploymentTaskService.createTask(deployment)
    deploymentTaskService.startDeploymentTask(taskId)
    taskId
  }

  def apply(context: InterpreterContext): AsCodeResponse = {
    val spec = context.definition.spec.asInstanceOf[DeploymentSpec]
    val deploymentType = DeploymentType.UNDEPLOYMENT.toString
    val deploymentWithDeployeds = Option(spec.deploymentType) match {
      case Some(spec.deploymentType) if deploymentType.equalsIgnoreCase(spec.deploymentType) =>
        deploymentPreparationService.deploymentSpecToUnDeployment(spec)
      case _ =>  deploymentPreparationService.deploymentSpecToDeployment(spec)
    }
    val taskId = prepareAndStartTask(deploymentWithDeployeds)
    val taskType = deploymentWithDeployeds.getDeploymentType.toString.toLowerCase.capitalize

    AsCodeResponse.task(TaskInfo(
      id = taskId,
      description = Option(spec.deploymentType) match {
        case Some(spec.deploymentType) if deploymentType.equalsIgnoreCase(spec.deploymentType) =>
          s"$taskType of '${spec.version}'"
        case _ => s"$taskType deployment of '${spec.version}' to '${spec.environment}'"
      },
      started = true
    ))
  }
}
