package com.xebialabs.deployit.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.PreviewResponse
import com.xebialabs.ascode.yaml.dto.PreviewResponse.{TaskInfo, TaskPreviewItem}
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.deployit.ascode.service.spec.util.DeploymentPreparationService
import com.xebialabs.deployit.ascode.service.validation.DeploymentValidator
import com.xebialabs.deployit.ascode.yaml.model.Constants.Kinds._
import com.xebialabs.deployit.ascode.yaml.model.DeploymentSpec
import com.xebialabs.deployit.core.rest.{DryRunResource, PreviewResource}
import com.xebialabs.deployit.engine.api.DeploymentService
import com.xebialabs.deployit.engine.api.dto.Deployment.DeploymentType
import com.xebialabs.deployit.engine.api.dto.{ChangeSetDto, Deployment, DryRunDeploymentDto}
import com.xebialabs.deployit.engine.api.execution._
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
@Autowired
class PreviewService(deploymentService: DeploymentService,
                     stitchPreviewResource: PreviewResource,
                     stitchPreviewService: StitchPreviewService,
                     dryRunResource: DryRunResource,
                     deploymentPreparationService: DeploymentPreparationService,
                     deploymentValidator: DeploymentValidator) {
  private def transformStepState(stepState: StepState) = TaskPreviewItem(stepState.getDescription, Nil)

  private def transformBlockState(blockState: BlockState, isParallel: Boolean = false): List[TaskPreviewItem] = {
    blockState match {
      case container: PhaseContainer =>
        val children = container.phases.toList.flatMap(transformBlockState(_))
        Option(container.description)
          .filter(_.nonEmpty)
          .map(desc => TaskPreviewItem(desc, children, isParallel) :: Nil)
          .getOrElse(children)
      case ParallelBlock(_, description, _, blocks) =>
        TaskPreviewItem(description, blocks.toList.flatMap(transformBlockState(_, isParallel = true)), isParallel) :: Nil
      case SerialBlock(_, description, _, blocks) =>
        TaskPreviewItem(description, blocks.toList.flatMap(transformBlockState(_)), isParallel) :: Nil
      case state: StepBlockState =>
        TaskPreviewItem(state.getDescription, state.getSteps.asScala.toList.map(transformStepState), isParallel) :: Nil
      case phase: Phase =>
        transformBlockState(phase.block)
      case blockState: Block =>
        TaskPreviewItem(blockState.description, blockState.getStepList().asScala.toList.map(transformStepState), isParallel) :: Nil
      case _ => Nil
    }
  }

  private def getTaskDescription(deployment: Deployment): String = {
    val typeRelatedPart = deployment.getDeploymentType match {
      case DeploymentType.INITIAL => "deployed to"
      case DeploymentType.UPDATE => "updated on"
      case DeploymentType.UNDEPLOYMENT => "un-deployed from"
    }
    deployment.getDeployedApplication match {
      case deployedApp: DeployedApplication => s"Application '${deployedApp.getVersion.getId}' will be $typeRelatedPart '${deployedApp.getEnvironment.getId}'"
    }
  }

  def preview(definition: Definition): PreviewResponse = {
    if (definition.kind == DEPLOYMENT) {
      val spec = definition.spec.asInstanceOf[DeploymentSpec]
      val deployment = deploymentPreparationService.deploymentSpecToDeployment(spec)
      val validatedDeployment = deploymentValidator.validate(deployment)
      val taskPreviewBlock = deploymentService.taskPreviewBlock(validatedDeployment)
      val taskPreviewItems = transformBlockState(taskPreviewBlock.getBlock)
      PreviewResponse(Some(TaskInfo(getTaskDescription(deployment), taskPreviewItems)))
    } else if (definition.kind == STITCH_PREVIEW) {
      val spec = definition.spec.asInstanceOf[DeploymentSpec]
      val deployment = deploymentPreparationService.deploymentSpecToDeployment(spec)
      val stitchPreviewBlock = stitchPreviewResource.preview(deployment)
      val stitchPreviewItems = stitchPreviewService.transformStitchPreviewResult(stitchPreviewBlock)
      PreviewResponse(Some(TaskInfo(stitchPreviewService.getStitchTaskDescription(deployment, "Generating stitch preview for the deployment of application"), stitchPreviewItems)))
    } else if (definition.kind == STITCH_DRYRUN) {
      val spec = definition.spec.asInstanceOf[DeploymentSpec]
      val deployment = deploymentPreparationService.deploymentSpecToDeployment(spec)
      val dryRunPreviewResult = dryRunResource.dryRun(new DryRunDeploymentDto(new ChangeSetDto(spec.stitchSource, spec.branchName), deployment))
      val stitchPreviewItems = stitchPreviewService.transformStitchPreviewResult(dryRunPreviewResult)
      PreviewResponse(Some(TaskInfo(stitchPreviewService.getStitchTaskDescription(deployment, "Generating DryRun preview " +
        s"on CI '${spec.stitchSource}' and branch '${spec.branchName}' for the deployment of application"), stitchPreviewItems)))
    }
    else {
      throw new AsCodeException(s"Cannot preview '${definition.kind}' spec. Only '$DEPLOYMENT' and '$STITCH_PREVIEW' spec kind are supported.")
    }
  }
}
