package com.xebialabs.deployit.core.rest;

import com.xebialabs.ascode.service.spec.InterpreterContext;
import com.xebialabs.ascode.yaml.model.Definition;
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig;
import com.xebialabs.deployit.ascode.service.DefinitionInterpreterService;
import com.xebialabs.deployit.ascode.service.generator.DefinitionGeneratorService;
import com.xebialabs.deployit.ascode.service.generator.DefinitionGeneratorService.GeneratorConfig;
import com.xebialabs.deployit.core.rest.resteasy.Workdir;
import com.xebialabs.deployit.core.rest.resteasy.Workdir.Clean;
import com.xebialabs.deployit.engine.api.XLDAsCodeService;
import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import com.xebialabs.deployit.ascode.yaml.parser.XLDDefinitionParser;
import com.xebialabs.deployit.ascode.yaml.sugar.XLDSugar;
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import java.io.ByteArrayInputStream;

/**
 * REST Controller implementing XLDAsCodeService for DevOps-as-Code operations.
 * Provides REST endpoints for GitOps plugin and external workers to apply/generate YAML.
 * 
 * Following the same pattern as TaskBlockResource, this class:
 * - Annotated with @Controller (not @Service) to enable REST endpoint registration
 * - Uses @Workdir annotation to inject WorkDir context for file operations
 * - Implements the XLDAsCodeService interface defined in engine-api
 */
@Controller
public class XLDAsCodeServiceImpl implements XLDAsCodeService {
    
    private static final String AS_CODE_APPLY_PREFIX = "as-code-apply";
    private static final String AS_CODE_GENERATE_PREFIX = "as-code-generate";
    
    private static final Logger logger = LoggerFactory.getLogger(XLDAsCodeServiceImpl.class);
    
    private final DefinitionInterpreterService asCodeInterpretationService;
    private final DefinitionGeneratorService generatorService;
    private final XLDDefinitionParser parser;

    @Autowired
    public XLDAsCodeServiceImpl(DefinitionInterpreterService asCodeInterpretationService,
                               DefinitionGeneratorService generatorService,
                               XLDDefinitionParser parser) {
        this.asCodeInterpretationService = asCodeInterpretationService;
        this.generatorService = generatorService;
        this.parser = parser;
        logger.debug("XLDAsCodeServiceImpl bean created successfully");
    }    
    
    @Override
    @Workdir(prefix = AS_CODE_APPLY_PREFIX, clean = Clean.ONLY_ON_EXCEPTION)
    public XLDAsCodeResult apply(String yamlContent) {
        logger.debug("XLDAsCodeServiceImpl.apply() called with YAML content length: {}", yamlContent != null ? yamlContent.length() : 0);
        
        // Log user information for master node verification
        try {
            String currentUser = System.getProperty("user.name");
            String javaVersion = System.getProperty("java.version");
            String processId = String.valueOf(ProcessHandle.current().pid());
            String workingDir = System.getProperty("user.dir");
            
            logger.info("=== MASTER NODE EXECUTION INFO ===");
            logger.info("XLDAsCodeServiceImpl.apply() - User: {}, PID: {}, Java: {}", currentUser, processId, javaVersion);
            logger.info("Working Directory: {}", workingDir);
            logger.info("Thread: {}", Thread.currentThread().getName());
            logger.info("==================================");
            
        } catch (Exception e) {
            logger.warn("Could not retrieve master node execution info: {}", e.getMessage());
        }
        
        logger.debug("Applying DevOps-as-Code YAML via XLDAsCodeService");
        
        try {
            // Parse YAML content into Definition using same logic as XLDAsCodeResource
            Definition definition = parseYamlContent(yamlContent);
            
            // Use same pattern as XLDAsCodeResource.interpret()
            // No SCM traceability data for plugin usage
            var response = asCodeInterpretationService.interpret(new InterpreterContext(definition, scala.Option.empty()));
            
            logger.debug("DevOps-as-Code apply successful");
            
            // Log detailed response information at debug level
            if (response != null && logger.isDebugEnabled()) {
                if (response.changes().isDefined()) {
                    var changes = response.changes().get();
                    logger.debug("Changes detected: {}", changes);
                } else {
                    logger.debug("No changes detected in response");
                }
                
                if (response.errors().isDefined()) {
                    logger.warn("Errors in response: {}", response.errors().get());
                }
            }
            
            return new XLDAsCodeResult(true, "DevOps-as-Code YAML applied successfully", response != null ? response.toString() : "null response", null);
            
        } catch (Exception e) {
            logger.error("Failed to apply DevOps-as-Code YAML: {}", e.getMessage(), e);
            return new XLDAsCodeResult(false, "Failed to apply YAML: " + e.getMessage(), null, getStackTrace(e));
        }
    }

    @Override
    @Workdir(prefix = AS_CODE_GENERATE_PREFIX, clean = Clean.ONLY_ON_EXCEPTION)
    public XLDAsCodeResult generate(String path, boolean globalPermissions, boolean roles, boolean users, 
                                   boolean includeSecrets, boolean includeDefaults) {
        logger.debug("Generating DevOps-as-Code YAML via XLDAsCodeService for path: {}", path);
        
        try {
            // Use same pattern as XLDAsCodeResource.generate()
            GeneratorConfig config = new GeneratorConfig(
                scala.Option.apply(path).filter(p -> p != null && !p.isEmpty()), 
                globalPermissions, roles, users, includeSecrets
            );
            
            var definitions = generatorService.generate(config);
            WriterConfig writerConfig = new WriterConfig(definitions, includeSecrets, includeDefaults);
            
            logger.debug("DevOps-as-Code generation successful");
            
            // Convert WriterConfig to string representation for the result
            String generatedContent = writerConfig.toString();
            
            return new XLDAsCodeResult(true, "DevOps-as-Code YAML generated successfully", generatedContent, null);
            
        } catch (Exception e) {
            logger.error("Failed to generate DevOps-as-Code YAML: {}", e.getMessage(), e);
            return new XLDAsCodeResult(false, "Failed to generate YAML: " + e.getMessage(), null, getStackTrace(e));
        }
    }

    /**
     * Parses YAML content into Definition object using XLDDefinitionParser.
     */
    private Definition parseYamlContent(String yamlContent) {
        try {
            // Use XLDDefinitionParser with default config, same as XLDAsCodeResource
            return parser.parse(new ByteArrayInputStream(yamlContent.getBytes()), XLDSugar.defaultConfig());
            
        } catch (Exception e) {
            throw new IllegalArgumentException("Failed to parse YAML content: " + e.getMessage(), e);
        }
    }

    /**
     * Gets stack trace as string for error reporting.
     */
    private String getStackTrace(Exception e) {
        java.io.StringWriter sw = new java.io.StringWriter();
        java.io.PrintWriter pw = new java.io.PrintWriter(sw);
        e.printStackTrace(pw);
        return sw.toString();
    }
}