package com.xebialabs.deployit.ascode.service.generator

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.utils.TypeSugar._
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.deployit.ascode.service.generator.DefinitionGeneratorService.{GeneratorConfig, _}
import com.xebialabs.deployit.ascode.service.validation.PermissionValidator
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.internal.Root
import com.xebialabs.deployit.repository.{RepositoryService, SearchParameters}
import com.xebialabs.deployit.security.permission.PermissionHelper
import com.xebialabs.deployit.security.permission.PlatformPermissions.{ADMIN, READ}
import com.xebialabs.deployit.security.{PermissionDeniedException, PermissionEnforcer}
import com.xebialabs.xlplatform.config.ProductConfiguration
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.{List => JavaList}
import scala.jdk.CollectionConverters._
import scala.util.Try

object DefinitionGeneratorService {
  val GENERATE_CI_LIMIT_KEY: String = "xl.deploy.devops-as-code.ci-limit"
  val GENERATE_CI_LIMIT_DEFAULT: Int = 256
  private var GENERATE_CI_LIMIT: Int = _

  case class GeneratorConfig(ciId: Option[String],
                             globalPermissions: Boolean = false,
                             roles: Boolean = false,
                             users: Boolean = false,
                             includeSecrets: Boolean = false)

  def getGenerateCiLimit: Int = GENERATE_CI_LIMIT
}

@Service
@Autowired
class DefinitionGeneratorService(repository: RepositoryService,
                                 permissionEnforcer: PermissionEnforcer,
                                 permissionValidator: PermissionValidator,
                                 generators: JavaList[DefinitionGenerator],
                                 xlConfiguration: ProductConfiguration) extends Logging {

  private def getConfigInt(key: String, defaultValue: Int) = {
    if (xlConfiguration.configuration.hasPath(key)) {
      Try(xlConfiguration.configuration.getInt(key)).getOrElse(defaultValue)
    } else {
      defaultValue
    }
  }

  private def init(): Unit = {
    GENERATE_CI_LIMIT = getConfigInt(GENERATE_CI_LIMIT_KEY, GENERATE_CI_LIMIT_DEFAULT)
    logger.debug(s"Devops As Code CI generate limit set to $GENERATE_CI_LIMIT")
  }

  init()

  private def validate(config: GeneratorConfig): Unit = {
    if (config.ciId.isEmpty && !config.globalPermissions && !config.roles && !config.users) {
      throw new AsCodeException("Generate path cannot be empty")
    }
    if (config.includeSecrets) {
      if (!PermissionHelper.isCurrentUserAdmin) throw PermissionDeniedException.forPermission(ADMIN, null: String)
    }

    config.ciId.foreach { ciId =>
      if (ciId.startsWith("/") || ciId.endsWith("/")) {
        throw new AsCodeException(s"Cannot generate definition for path `$ciId`. It should not start or end with `/`")
      }

      if (!repository.exists(ciId)) {
        throw new AsCodeException(s"A CI with id `$ciId` does not exist")
      }

      permissionValidator.checkPermissions(READ, ciId :: Nil)
    }
  }

  private def retrieveTargetedCi(ciId: String): Option[ConfigurationItem] =
    Option(repository.read[ConfigurationItem](ciId)).filterNot(_.getType.instanceOf(typeOf[Root]))

  private def retrieveCis(config: GeneratorConfig): List[ConfigurationItem] = {
    config.ciId.map { ciId =>
      val ci = retrieveTargetedCi(ciId)

      val params = new SearchParameters()
      params.setAncestor(ciId)
      params.setResultsPerPage(GENERATE_CI_LIMIT + 1)
      permissionEnforcer.applyLoggedInUserPermission(params, READ)
      val cis = ci.toList ::: repository.listEntities[ConfigurationItem](params).asScala.toList

      if (cis.length > GENERATE_CI_LIMIT) {
        throw new AsCodeException(s"The result contains more then $GENERATE_CI_LIMIT configuration items. Please narrow down the amount of CIs by specifying a more specific path.")
      }
      cis
    }.toList.flatten
  }

  def generate(config: GeneratorConfig): List[Definition] = {
    validate(config)
    val cis = retrieveCis(config)
    generators.asScala.toList.flatMap(svc =>
      try {
        svc.generate(config, cis)
      } catch {
        case _:PermissionDeniedException => Nil
      })
  }
}
