package com.xebialabs.deployit.core.rest;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.xebialabs.ascode.service.spec.InterpreterContext;
import com.xebialabs.ascode.yaml.dto.AsCodeResponse;
import com.xebialabs.ascode.yaml.model.Definition;
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig;
import com.xebialabs.deployit.ascode.service.DefinitionInterpreterService;
import com.xebialabs.deployit.ascode.service.generator.DefinitionGeneratorService;
import com.xebialabs.deployit.ascode.service.generator.DefinitionGeneratorService.GeneratorConfig;
import com.xebialabs.deployit.core.rest.resteasy.Workdir;
import com.xebialabs.deployit.core.rest.resteasy.Workdir.Clean;
import com.xebialabs.deployit.engine.api.XLDAsCodeService;
import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import com.xebialabs.deployit.ascode.yaml.parser.XLDDefinitionParser;
import com.xebialabs.deployit.ascode.yaml.sugar.XLDSugar;
import com.xebialabs.deployit.ascode.yaml.writer.XLDDefinitionWriter;
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import scala.jdk.javaapi.CollectionConverters;

import java.io.ByteArrayInputStream;

/**
 * REST Controller implementing XLDAsCodeService for DevOps-as-Code operations.
 * Provides REST endpoints for GitOps plugin and external workers to apply/generate YAML.
 * 
 * Following the same pattern as TaskBlockResource, this class:
 * - Annotated with @Controller (not @Service) to enable REST endpoint registration
 * - Uses @Workdir annotation to inject WorkDir context for file operations
 * - Implements the XLDAsCodeService interface defined in engine-api
 */
@Controller
public class XLDAsCodeServiceImpl implements XLDAsCodeService {
    
    private static final String AS_CODE_APPLY_PREFIX = "as-code-apply";
    private static final String AS_CODE_GENERATE_PREFIX = "as-code-generate";
    
    private static final Logger logger = LoggerFactory.getLogger(XLDAsCodeServiceImpl.class);
    private static final ObjectMapper objectMapper = new ObjectMapper();
    
    private final DefinitionInterpreterService asCodeInterpretationService;
    private final DefinitionGeneratorService generatorService;
    private final XLDDefinitionParser parser;

    @Autowired
    public XLDAsCodeServiceImpl(DefinitionInterpreterService asCodeInterpretationService,
                               DefinitionGeneratorService generatorService,
                               XLDDefinitionParser parser) {
        this.asCodeInterpretationService = asCodeInterpretationService;
        this.generatorService = generatorService;
        this.parser = parser;
        logger.debug("XLDAsCodeServiceImpl bean created successfully");
    }    
    
    @Override
    @Workdir(prefix = AS_CODE_APPLY_PREFIX, clean = Clean.ONLY_ON_EXCEPTION)
    public XLDAsCodeResult apply(String yamlContent) {
        logger.debug("XLDAsCodeServiceImpl.apply() called with YAML content length: {}", yamlContent != null ? yamlContent.length() : 0);
        
        // Log user information for master node verification
        try {
            String currentUser = System.getProperty("user.name");
            String javaVersion = System.getProperty("java.version");
            String processId = String.valueOf(ProcessHandle.current().pid());
            String workingDir = System.getProperty("user.dir");
            
            logger.info("=== MASTER NODE EXECUTION INFO ===");
            logger.info("XLDAsCodeServiceImpl.apply() - User: {}, PID: {}, Java: {}", currentUser, processId, javaVersion);
            logger.info("Working Directory: {}", workingDir);
            logger.info("Thread: {}", Thread.currentThread().getName());
            logger.info("==================================");
            
        } catch (Exception e) {
            logger.error("Could not retrieve master node execution info: {}", e.getMessage());
        }
        
        logger.debug("Applying DevOps-as-Code YAML via XLDAsCodeService");
        
        try {
            // Parse YAML content into Definition using same logic as XLDAsCodeResource
            Definition definition = parseYamlContent(yamlContent);
            
            // Use same pattern as XLDAsCodeResource.interpret()
            // No SCM traceability data for plugin usage
            var response = asCodeInterpretationService.interpret(new InterpreterContext(definition, scala.Option.empty()));
            
            logger.debug("DevOps-as-Code apply successful");
            
            // Log detailed response information at debug level
            if (response != null && logger.isDebugEnabled()) {
                if (response.changes().isDefined()) {
                    var changes = response.changes().get();
                    logger.debug("Changes detected: {}", changes);
                } else {
                    logger.debug("No changes detected in response");
                }
                
                if (response.errors().isDefined()) {
                    logger.warn("Errors in response: {}", response.errors().get());
                }
            }
            
            // Serialize response to JSON for proper parsing by clients
            String jsonContent = serializeAsCodeResponse(response);
            return new XLDAsCodeResult(true, "DevOps-as-Code YAML applied successfully", jsonContent, null);
            
        } catch (Exception e) {
            logger.error("Failed to apply DevOps-as-Code YAML: {}", e.getMessage(), e);
            return new XLDAsCodeResult(false, "Failed to apply YAML: " + e.getMessage(), null, getStackTrace(e));
        }
    }

    @Override
    @Workdir(prefix = AS_CODE_APPLY_PREFIX, clean = Clean.ONLY_ON_EXCEPTION)
    public XLDAsCodeResult applyWithScm(String yamlContent, String scmType, String scmCommit, 
                                        String scmAuthor, String scmDate, String scmMessage, 
                                        String scmRemote, String scmFileName) {
        logger.debug("XLDAsCodeServiceImpl.applyWithScm() called with YAML content length: {}, SCM commit: {}", 
                     yamlContent != null ? yamlContent.length() : 0, scmCommit != null ? scmCommit : "null");
        
        if (yamlContent == null || yamlContent.isEmpty()) {
            return new XLDAsCodeResult(false, "YAML content cannot be null or empty", null, null);
        }
        
        logger.debug("Applying DevOps-as-Code YAML via XLDAsCodeService with SCM traceability");
        
        try {
            // Parse YAML content into Definition using same logic as XLDAsCodeResource
            Definition definition = parseYamlContent(yamlContent);
            
            // Construct SCMTraceabilityData from individual parameters if commit is provided
            scala.Option<SCMTraceabilityData> scmOption;
            if (scmCommit != null && !scmCommit.isEmpty()) {
                // Parse date string to DateTime (null if parsing fails or date is empty)
                org.joda.time.DateTime dateTime = null;
                if (scmDate != null && !scmDate.isEmpty()) {
                    try {
                        dateTime = org.joda.time.DateTime.parse(scmDate);
                    } catch (Exception e) {
                        logger.debug("Could not parse SCM date '{}': {}", scmDate, e.getMessage());
                    }
                }
                
                SCMTraceabilityData scmData = new SCMTraceabilityData(
                    scmType != null ? scmType : "git",
                    scmCommit,
                    scmAuthor != null ? scmAuthor : "",
                    dateTime,
                    scmMessage != null ? scmMessage : "",
                    scmRemote != null ? scmRemote : "",
                    scmFileName != null ? scmFileName : ""
                );
                scmOption = scala.Option.apply(scmData);
                logger.debug("SCM Traceability - Commit: {}, Author: {}, Remote: {}", 
                            scmCommit, scmAuthor, scmRemote);
            } else {
                scmOption = scala.Option.<SCMTraceabilityData>empty();
            }
            
            var response = asCodeInterpretationService.interpret(new InterpreterContext(definition, scmOption));
            
            logger.debug("DevOps-as-Code apply with SCM traceability successful");
            
            // Log detailed response information at debug level
            if (response != null && logger.isDebugEnabled()) {
                if (response.changes().isDefined()) {
                    var changes = response.changes().get();
                    logger.debug("Changes detected: {}", changes);
                } else {
                    logger.debug("No changes detected in response");
                }
                
                if (response.errors().isDefined()) {
                    logger.warn("Errors in response: {}", response.errors().get());
                }
            }
            
            // Serialize response to JSON for proper parsing by clients
            String jsonContent = serializeAsCodeResponse(response);
            return new XLDAsCodeResult(true, "DevOps-as-Code YAML applied successfully with SCM traceability", jsonContent, null);
            
        } catch (Exception e) {
            logger.error("Failed to apply DevOps-as-Code YAML with SCM traceability: {}", e.getMessage(), e);
            return new XLDAsCodeResult(false, "Failed to apply YAML: " + e.getMessage(), null, getStackTrace(e));
        }
    }

    @Override
    @Workdir(prefix = AS_CODE_GENERATE_PREFIX, clean = Clean.ONLY_ON_EXCEPTION)
    public XLDAsCodeResult generate(String path, boolean globalPermissions, boolean roles, boolean users, 
                                   boolean includeSecrets, boolean includeDefaults) {
        logger.debug("Generating DevOps-as-Code YAML via XLDAsCodeService for path: {}", path);
        
        try {
            // Use same pattern as XLDAsCodeResource.generate()
            GeneratorConfig config = new GeneratorConfig(
                scala.Option.apply(path).filter(p -> p != null && !p.isEmpty()), 
                globalPermissions, roles, users, includeSecrets
            );
            
            var definitions = generatorService.generate(config);
            WriterConfig writerConfig = new WriterConfig(definitions, includeSecrets, includeDefaults);
            
            logger.debug("DevOps-as-Code generation successful");
            
            // Write the ZIP content to a ByteArrayOutputStream
            java.io.ByteArrayOutputStream zipOutputStream = new java.io.ByteArrayOutputStream();
            XLDDefinitionWriter.write(zipOutputStream, writerConfig, XLDSugar.defaultConfig());
            
            // Extract index.yaml from the ZIP
            String generatedContent = extractYamlFromZip(zipOutputStream.toByteArray());
            
            return new XLDAsCodeResult(true, "DevOps-as-Code YAML generated successfully", generatedContent, null);
            
        } catch (Exception e) {
            logger.error("Failed to generate DevOps-as-Code YAML: {}", e.getMessage(), e);
            return new XLDAsCodeResult(false, "Failed to generate YAML: " + e.getMessage(), null, getStackTrace(e));
        }
    }

    /**
     * Parses YAML content into Definition object using XLDDefinitionParser.
     */
    private Definition parseYamlContent(String yamlContent) {
        try {
            // Use XLDDefinitionParser with default config, same as XLDAsCodeResource
            return parser.parse(new ByteArrayInputStream(yamlContent.getBytes()), XLDSugar.defaultConfig());
            
        } catch (Exception e) {
            throw new IllegalArgumentException("Failed to parse YAML content: " + e.getMessage(), e);
        }
    }

    /**
     * Gets stack trace as string for error reporting.
     */
    private String getStackTrace(Exception e) {
        java.io.StringWriter sw = new java.io.StringWriter();
        java.io.PrintWriter pw = new java.io.PrintWriter(sw);
        e.printStackTrace(pw);
        return sw.toString();
    }
    
    /**
     * Extracts the index.yaml content from a ZIP byte array.
     * The XLDDefinitionWriter produces a ZIP containing index.yaml and optionally secrets.xlvals.
     */
    private String extractYamlFromZip(byte[] zipBytes) throws Exception {
        try (java.util.zip.ZipInputStream zipIn = new java.util.zip.ZipInputStream(
                new java.io.ByteArrayInputStream(zipBytes))) {
            java.util.zip.ZipEntry entry;
            while ((entry = zipIn.getNextEntry()) != null) {
                if ("index.yaml".equals(entry.getName())) {
                    java.io.ByteArrayOutputStream yamlOut = new java.io.ByteArrayOutputStream();
                    byte[] buffer = new byte[4096];
                    int len;
                    while ((len = zipIn.read(buffer)) > 0) {
                        yamlOut.write(buffer, 0, len);
                    }
                    return yamlOut.toString("UTF-8");
                }
                zipIn.closeEntry();
            }
        }
        throw new IllegalStateException("index.yaml not found in generated ZIP");
    }
    
    /**
     * Serializes AsCodeResponse to JSON format for parsing by clients.
     * Format matches the REST API response structure:
     * {
     *   "changes": {
     *     "ids": [{ "kind": "CI", "created": [...], "updated": [...], "deleted": [...] }]
     *   },
     *   "errors": { ... }
     * }
     */
    private String serializeAsCodeResponse(AsCodeResponse response) {
        if (response == null) {
            return "{}";
        }
        
        try {
            ObjectNode root = objectMapper.createObjectNode();
            
            // Serialize changes
            if (response.changes().isDefined()) {
                var changes = response.changes().get();
                ObjectNode changesNode = objectMapper.createObjectNode();
                ArrayNode idsArray = objectMapper.createArrayNode();
                
                for (var changedIds : CollectionConverters.asJava(changes.ids())) {
                    ObjectNode idNode = objectMapper.createObjectNode();
                    idNode.put("kind", changedIds.kind());
                    
                    ArrayNode createdArray = objectMapper.createArrayNode();
                    for (String id : CollectionConverters.asJava(changedIds.created())) {
                        createdArray.add(id);
                    }
                    idNode.set("created", createdArray);
                    
                    ArrayNode updatedArray = objectMapper.createArrayNode();
                    for (String id : CollectionConverters.asJava(changedIds.updated())) {
                        updatedArray.add(id);
                    }
                    idNode.set("updated", updatedArray);
                    
                    ArrayNode deletedArray = objectMapper.createArrayNode();
                    for (String id : CollectionConverters.asJava(changedIds.deleted())) {
                        deletedArray.add(id);
                    }
                    idNode.set("deleted", deletedArray);
                    
                    idsArray.add(idNode);
                }
                
                changesNode.set("ids", idsArray);
                root.set("changes", changesNode);
            }
            
            // Serialize errors if present
            if (response.errors().isDefined()) {
                var errors = response.errors().get();
                ObjectNode errorsNode = objectMapper.createObjectNode();
                
                if (!CollectionConverters.asJava(errors.validation()).isEmpty()) {
                    ArrayNode validationArray = objectMapper.createArrayNode();
                    for (var error : CollectionConverters.asJava(errors.validation())) {
                        ObjectNode errorNode = objectMapper.createObjectNode();
                        errorNode.put("ciId", error.ciId());
                        errorNode.put("propertyName", error.propertyName());
                        errorNode.put("message", error.message());
                        validationArray.add(errorNode);
                    }
                    errorsNode.set("validation", validationArray);
                }
                
                if (!CollectionConverters.asJava(errors.permission()).isEmpty()) {
                    ArrayNode permissionArray = objectMapper.createArrayNode();
                    for (var error : CollectionConverters.asJava(errors.permission())) {
                        ObjectNode errorNode = objectMapper.createObjectNode();
                        errorNode.put("ciId", error.ciId());
                        errorNode.put("permission", error.permission());
                        permissionArray.add(errorNode);
                    }
                    errorsNode.set("permission", permissionArray);
                }
                
                if (errors.document().isDefined()) {
                    var docError = errors.document().get();
                    ObjectNode docNode = objectMapper.createObjectNode();
                    docNode.put("field", docError.field());
                    docNode.put("problem", docError.problem());
                    errorsNode.set("document", docNode);
                }
                
                if (errors.generic().isDefined()) {
                    errorsNode.put("generic", errors.generic().get());
                }
                
                root.set("errors", errorsNode);
            }
            
            return objectMapper.writeValueAsString(root);
            
        } catch (Exception e) {
            logger.error("Failed to serialize AsCodeResponse to JSON, falling back to toString: {}", e.getMessage());
            return response.toString();
        }
    }
}