package com.xebialabs.deployit.cli.api.internal;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import javax.ws.rs.core.Response;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.io.Files;

import com.xebialabs.deployit.cli.rest.ResponseExtractor;
import com.xebialabs.deployit.core.api.ImportablePackageProxy;
import com.xebialabs.deployit.core.api.dto.FileUpload;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;


/**
 * Helper that can both upload packages from the client and import packages on the server.
 */
public class ImportHelper {

    /**
     * Check whether the packageToImport parameter refers to a file by checking for its existence.
     * If it is not a file, it should be a package in the importablePackages directory on the server, and should be
     * importable from there.
     *
     * @param proxy
     * @param packageToImport
     * @return
     * @throws IOException 
     */
    public static RepositoryObject doImport(ImportablePackageProxy proxy, String packageToImport) {
        Response response;
        try {
            URL url = new URL(packageToImport);
            if (url.getProtocol().equals("file")) {
                response = processFilePath(proxy,url.getFile());
            } else {
                response = proxy.importPackageFromUrl(url.toString());
            }
        } catch (MalformedURLException e) {
           response = processFilePath(proxy,packageToImport);
        }

        return new ResponseExtractor(response).getEntity();
    }

    private static Response processFilePath(ImportablePackageProxy proxy, String path) {
    	Response response;
    	final File file = new File(path);        
        if (file.exists()) {
            final FileUpload fileUpload = new FileUpload();
            try {
                fileUpload.setFileData(Files.toByteArray(file));
                response = proxy.importUploadedPackage(file.getName(), fileUpload);
            } catch (IOException e) {
                logger.error("The file \'{}\' exists, but could not be read for upload; perhaps it's not an archive.", file.toString());
                return null;                
            }
        } else {
        	if (!file.isAbsolute()) {
        		response = proxy.importPackage(path);
        	} else {
        		logger.error("The file \'{}\' does not exist at the given absolute path", file.toString());
        		return null;
        	}
        }
        return response;

    }

    private static final Logger logger = LoggerFactory.getLogger(ImportHelper.class);
}
