package com.xebialabs.deployit.client;

import com.google.common.base.*;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.booter.remote.BooterConfig;
import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.booter.remote.RemoteBooter;
import com.xebialabs.deployit.client.logger.Slf4jDeploymentListener;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.engine.api.dto.ServerInfo;
import com.xebialabs.deployit.engine.api.dto.ValidatedConfigurationItem;
import com.xebialabs.deployit.engine.api.execution.StepExecutionState;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import org.apache.commons.lang.StringUtils;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.any;
import static com.google.common.collect.Iterators.filter;
import static java.lang.String.format;

/**
 * Client for Deployit.
 */
public class DeployitCli {

	private final DeploymentListener listener;

	private ConnectionOptions options;

    private DeployitCommunicator communicator;

	private final DeploymentClient deploymentClient;
	private final DeployitClient deployitClient;

	private AtomicReference<Descriptors> descriptors = new AtomicReference<Descriptors>();

	public DeployitCli(ConnectionOptions options) {
		this(options, null);
	}

	public DeployitCli(ConnectionOptions options, DeploymentListener listener) {
		this.listener = (listener != null ? listener : new Slf4jDeploymentListener(DeployitCli.class));
		this.options = options;

		try {
			connect();
			deploymentClient = new DeploymentClient(communicator.getProxies().getDeploymentService());
			deployitClient = new DeployitClient(this.listener, communicator.getProxies());
		} catch (Exception e) {
			throw new RuntimeException("Initialization failed", e);
		}
	}

	private void connect() {
		getListener().info("Connecting to the Deployit server at " + options.getHost() + "...");

        communicator = RemoteBooter.boot(
                BooterConfig.builder()
                        .withProtocol(BooterConfig.Protocol.HTTP)
                        .withCredentials(options.getUsername(), options.getPassword())
                        .withHost(options.getHost())
                        .withPort(options.getPort())
                        .build()
        );
    }

	public ConfigurationItem create(Container ci) {
		final String id = ci.getId();
		try {
			return get(id);
		} catch (Exception e) {
			getListener().debug(format("%s does not exist, create it", id));

			BaseConfigurationItem baseConfigurationItem = new BaseConfigurationItem();
			baseConfigurationItem.setId(id);
			baseConfigurationItem.setType(Type.valueOf(ci.getType()));
			for (Map.Entry<String, Object> entry : ci.getProperties().entrySet()) {
				baseConfigurationItem.setProperty(entry.getKey(), entry.getValue());
			}


			return communicator.getProxies().getRepositoryService().create(id, baseConfigurationItem);
		}
	}

	public void delete(String id) {
		try {
			getListener().debug("Delete " + id);
			communicator.getProxies().getRepositoryService().delete(id);
		} catch (Exception e) {
			getListener().debug(format("delete fails %s", id));
		}
	}

	public List<String> search(String type) {
		getListener().debug("search " + type);
		try {
			List<ConfigurationItemId> result = communicator.getProxies().getRepositoryService().query(Type.valueOf(type), null, null, null, null, 0, -1);

			return Lists.transform(result, new Function<ConfigurationItemId, String>() {
				@Override
				public String apply(ConfigurationItemId input) {
					return input.getId();
				}
			});
		} catch (Exception e) {
			getListener().debug(format("search fails for %s %s", type, e.getMessage()));
		}
		return Collections.emptyList();
	}

	public ServerInfo info() {
		return communicator.getProxies().getServerService().getInfo();
	}

	public ConfigurationItem get(String ciId) {
		return communicator.getProxies().getRepositoryService().read(ciId);
	}

	public Descriptors getDescriptors() {
		if (descriptors.get() == null) {
			synchronized (this) {
				List<Descriptor> retrievedDescriptors = communicator.getProxies().getMetadataService().listDescriptors();
				descriptors.set(new Descriptors(retrievedDescriptors));
			}
		}
		return descriptors.get();
	}

	public ConfigurationItem importPackage(String darFileLocation, DeploymentListener listener) {
		getListener().setActionListener(listener);
		try {
			return new ImportHelper(communicator.getProxies()).doImport(darFileLocation);
		} finally {
			getListener().setActionListener(null);
		}
	}

	public String deploy(String source, String target, List<? extends Deployed> configuredDeployeds, DeploymentOptions deploymentOptions, DeploymentListener listener) {
		getListener().setActionListener(listener);

		getListener().debug(deploymentOptions.toString());

		Deployment deployment;
		String previouslyDeployedApplicationId = null;
		boolean initialDeployment = isInitialDeployment(source, target);
		if (initialDeployment) {
			getListener().info("initial Deployment");
			deployment = deploymentClient.prepareInitial(source, target);
			if (!deploymentOptions.isExplicitMappings()) {
				getListener().debug(" generateAllDeployeds");
				deployment = deploymentClient.generateAllDeployeds(deployment);
			}
		} else {
			getListener().info("upgrade Deployment");
			String deployedApplicationId = getDeployedApplicationId(source, target);
			previouslyDeployedApplicationId = getPreviousDeployedPackage(deployedApplicationId);
			deployment = deploymentClient.prepareUpgrade(source, deployedApplicationId);
			if (!deploymentOptions.isExplicitMappings() && deploymentOptions.isGenerateDeployedOnUpgrade()) {
				getListener().debug(" generateAllDeployeds");
				deployment = deploymentClient.generateAllDeployeds(deployment);
			}
		}

		if (deploymentOptions.isExplicitMappings()) {
			getListener().debug("use explicits deployeds");
			for (Deployed configuredDeployed : configuredDeployeds) {
				checkNotNull(configuredDeployed.getId(), "id is mandatory in the explicit deployed mode");
				String type = configuredDeployed.getType();
				checkNotNull(type, "type is mandatory in the explicit deployed mode");

				final String deployableId = configuredDeployed.getDeployable(source);
				final String containerId = configuredDeployed.getContainer();
				final boolean exist = any(deployment.getDeployeds(), new Predicate<ConfigurationItem>() {
					@Override
					public boolean apply(ConfigurationItem input) {
						return input.getProperty("deployable").equals(deployableId)
								&& input.getProperty("container").equals(containerId);
					}
				});
				if (exist) {
					getListener().debug(format(" deployed  %s %s %s already exists", deployableId, containerId, type));
				} else {
					getListener().debug(format(" generateSingleDeployed %s %s %s", deployableId, containerId, type));
					deployment = deploymentClient.generateSingleDeployed(deployableId, containerId, type, deployment);
				}
			}
		}

		if (configuredDeployeds != null) {
			getListener().debug("update the generated deployeds with the configured deployeds");
			final List<ConfigurationItem> deployeds = deployment.getDeployeds();
			for (ConfigurationItem configurationItem : deployeds) {
				getListener().debug(" check " + configurationItem);
				final String id = configurationItem.getId();
				final Iterator<? extends Deployed> deployedIterator = filter(configuredDeployeds.iterator(), new Predicate<Deployed>() {
					@Override
					public boolean apply(Deployed input) {
						return input.getId().equals(id);
					}
				});
				if (deployedIterator.hasNext()) {
					updateConfigurationItemValues(configurationItem, deployedIterator.next());
				} else {
					getListener().debug(" no configured deployed found with id " + configurationItem.getId());
				}
			}
		}

		final String orchestrator = deploymentOptions.getOrchestrator();
		if (!Strings.isNullOrEmpty(orchestrator)) {
			getListener().debug("set the orchestrator " + orchestrator);
			deployment.getDeployedApplication().setProperty("orchestrator", orchestrator);
		}

		getListener().debug(" dump Deployeds");
		for (ConfigurationItem itemDto : deployment.getDeployeds()) {
			getListener().debug(" - " + itemDto);
		}

		try {
			getListener().debug("validate");
			deployment = deploymentClient.validate(deployment);
		} catch (RuntimeException e) {
			getListener().error(" RuntimeException: " + e.getMessage());
			if (deploymentOptions.isFailIfNoStepsAreGenerated() || !e.getMessage().contains("The task did not deliver any steps")) {
				throw e;
			}
			return null;
		}

		int validationMessagesFound = 0;
		for (ConfigurationItem configurationItem : deployment.getDeployeds()) {
			if (!(configurationItem instanceof ValidatedConfigurationItem)) {
				continue;
			}
			for (ValidationMessage msg : ((ValidatedConfigurationItem) configurationItem).getValidations()) {
				getListener().error(format("Validation error found on '%s' on field '%s': %s, %s", configurationItem.getId(), msg.getCiId(), msg.getMessage(), configurationItem));
				getListener().error(format(" %s", configurationItem));
				validationMessagesFound++;
			}
		}

		if (validationMessagesFound > 0) {
			throw new IllegalStateException(format("Validation errors (%d) have been found", validationMessagesFound));
		}

		getListener().debug("deploy");

		String taskId = deploymentClient.deploy(deployment);
		try {
			executeTask(taskId, deploymentOptions);
		} catch(RuntimeException e) {
			try {
				if (deploymentOptions.isRollbackOnError()) {
					// perform a rollback
					getListener().error("Deployment failed, performing a rollback");
					executeTask(rollback(taskId, getListener()), deploymentOptions);
				}
			} finally {
				throw e;
			}
		}

		if (shouldDeletePreviousVersion(deploymentOptions, previouslyDeployedApplicationId)) {
			getListener().info("Delete previously deployed dar " + previouslyDeployedApplicationId);
			delete(previouslyDeployedApplicationId);
		}
		return previouslyDeployedApplicationId;
	}

	public String rollback(String taskid, DeploymentListener listener) {
		return deploymentClient.rollback(taskid);
	}
	
	public void undeployAndWait(String source) {
		getListener().info("   undeployAndWait " + source);
		String taskId = this.deploymentClient.undeploy(source);
		executeTask(taskId, new DeploymentOptions());
	}

	private void updateConfigurationItemValues(ConfigurationItem configurationItem, Deployed configuredDeployed) {
		getListener().debug(format(" update values of %s with %s", configuredDeployed.getId(), configuredDeployed.getValues()));

		for (Map.Entry<String, Object> e : configuredDeployed.getValues().entrySet()) {
			configurationItem.setProperty(e.getKey(), e.getValue());
		}

		if (!configuredDeployed.getPlaceholders().isEmpty()) {
			if (!configurationItem.hasProperty("placeholders")) {
				configurationItem.setProperty("placeholders", new HashMap<String, String>());
			}

			Map<String, String> placeholders = (Map<String, String>) configurationItem.getProperty("placeholders");
			placeholders.putAll(configuredDeployed.getPlaceholders());
		}

		getListener().debug(configurationItem.toString());
	}


	private boolean executeTask(String taskId, DeploymentOptions deploymentOptions) {
		if (deploymentOptions.isSkipMode()) {
			getListener().info("skip mode, skip all the steps");
			getDeployitClient().skipSteps(taskId, range(getDeployitClient().retrieveTaskInfo(taskId).getNrSteps() + 1));
		}

		checkTaskState(taskId);
		if (deploymentOptions.isTestMode()) {
			getListener().info("test mode, cancel task " + taskId);
			getDeployitClient().cancelTask(taskId);
			return false;
		}

		try {
			getListener().info("Start deployment task " + taskId);
			getDeployitClient().startTaskAndWait(taskId);
			checkTaskState(taskId);
			communicator.getProxies().getTaskService().archive(taskId);
			return true;
		} catch (RuntimeException e) {
			if (taskId != null) {
				getListener().error(format("Error when executing task %s: %s", taskId, e.getMessage()));
				if (deploymentOptions.isCancelTaskOnError())
					getDeployitClient().cancelTask(taskId);
			}
			throw e;
		}

	}

	private boolean isInitialDeployment(String source, String target) {
		String deployedApplicationId = getDeployedApplicationId(source, target);
		getListener().debug("  deployedApplicationId " + deployedApplicationId);
		try {
			get(deployedApplicationId);
			return false;
		} catch (Exception e) {
			return true;
		}
	}


	private String getDeployedApplicationId(String source, String target) {
		//source = 'Applications/tomcatApps/deployit-petclinic-tomcat/1.0-20120522-173607'
		//target = "Environments/DefaultEnvironment"
		// return "Environments/DefaultEnvironment/deployit-petclinic-tomcat"
		List<String> splitSource =  Lists.newArrayList(Splitter.on("/").split(source));
		final String appName = splitSource.get(splitSource.size() - 2);
		return Joiner.on("/").join(target, appName);
	}

	private String getPreviousDeployedPackage(String target) {
		final ConfigurationItem dp = communicator.getProxies().getRepositoryService().read(target);
		Object source = dp.getProperty("version");
		return (source == null ? null : source.toString());
	}

	private void checkTaskState(String taskId) {
		TaskState taskState = communicator.getProxies().getTaskService().getTask(taskId);
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd hh:mm:ss");
		getListener().info(format("%s Description	%s", taskId, taskState.getDescription()));
		getListener().info(format("%s State      	%s %d/%d", taskId, taskState.getState(), taskState.getCurrentStepNr(), taskState.getNrSteps()));
		if (taskState.getStartDate() != null) {
			final GregorianCalendar startDate = taskState.getStartDate().toGregorianCalendar();
			getListener().info(format("%s Start      %s", taskId, sdf.format(startDate.getTime())));
		}

		if (taskState.getCompletionDate() != null) {
			final GregorianCalendar completionDate = taskState.getCompletionDate().toGregorianCalendar();
			getListener().info(format("%s Completion %s", taskId, sdf.format(completionDate.getTime())));
		}

		StringBuilder sb = new StringBuilder();
		for (int i = 1; i <= taskState.getNrSteps(); i++) {
			final StepState stepInfo = communicator.getProxies().getTaskService().getStep(taskId, i, null);
			final String description = stepInfo.getDescription();
			final String log = stepInfo.getLog();
			String stepInfoMessage;
			if (StringUtils.isEmpty(log) || description.equals(log)) {
				stepInfoMessage = format("%s step #%d %s\t%s", taskId, i, stepInfo.getState(), description);
			} else {
				stepInfoMessage = format("%s step #%d %s\t%s\n%s", taskId, i, stepInfo.getState(), description, log);
			}

			getListener().info(stepInfoMessage);
			if (StepExecutionState.FAILED.equals(stepInfo.getState()))
				sb.append(stepInfoMessage);
		}

		if (TaskExecutionState.STOPPED.equals(taskState.getState()))
			throw new IllegalStateException(format("Errors when executing task %s: %s", taskId, sb));
	}


	private Integer[] range(int end) {
		Integer[] result = new Integer[end - 1];
		for (int i = 1; i < end; i++) {
			result[i - 1] = i;
		}
		return result;
	}

	private boolean shouldDeletePreviousVersion(DeploymentOptions deploymentOptions, String previousPackageId) {
		if (deploymentOptions.isTestMode())
			return false;
		return deploymentOptions.isDeletePreviouslyDeployedArtifact() && Strings.emptyToNull(previousPackageId) != null;
	}


	private DeployitClient getDeployitClient() {
		return deployitClient;
	}


	public DeploymentListener getListener() {
		return listener;
	}
}
