package com.xebialabs.deployit.documentation;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.documentation.IOUtils.copy;
import static com.xebialabs.deployit.documentation.IOUtils.copyDirectory;
import static com.xebialabs.deployit.documentation.IOUtils.createFileNameWithNewExtension;
import static com.xebialabs.deployit.documentation.IOUtils.explodeArchive;
import static com.xebialabs.deployit.documentation.IOUtils.extractFileName;
import static com.xebialabs.deployit.documentation.IOUtils.extractFileNameExtension;
import static com.xebialabs.deployit.documentation.IOUtils.extractFileNameWithoutExtension;
import static com.xebialabs.deployit.documentation.IOUtils.getText;
import static com.xebialabs.deployit.documentation.IOUtils.replacePlaceholders;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;
import org.pegdown.PegDownProcessor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Splitter;
import com.google.common.io.Closeables;
import com.xebialabs.overthere.RuntimeIOException;

public class DocumentGenerator {

    @Option(name = "-pdfCover", usage = "Url to the html page that is to be used as the cover page in the generated pdf.")
    private URL pdfCoverHtml;

    @Option(name = "-pdfFooter", usage = "Url to the html page that is to be used as the footer on each page in the generated pdf.")
    private URL pdfFooterHtml;

    @Option(name = "-pdfHeader", usage = "Url to the html page that is to be used as the footer on each page in the generated pdf.")
    private URL pdfHeaderHtml;

    @Option(name = "-pdfTocXsl", usage = "Url to the xsl that is to be used to generate the html page for the table of content in the generated pdf.")
    private URL pdfTocXsl;

    @Option(name = "-htmlFooter", usage = "Url to the html snippet that is appended to the generated html document.")
    private URL htmlFooterSnippet;

    @Option(name = "-htmlHeader", usage = "Url to the html snippet that is added to the head tag in the generated html document.")
    private URL htmlHeaderSnippet;

    @Option(name = "-resource", usage = "Url to the web resource file(s)/zip(s) or local file(s)/folder(s) containing resources like images, css, etc.")
    private List<URL> resources = newArrayList();

    @Option(name = "-single", usage = "Generate a single document from the multiple sources")
    private boolean generateSingleDocument;

    @Option(name = "-generatePdf", usage = "Indicates if pdf documentation must be generated. Default is false.")
    private boolean generatePdf;

    @Option(name = "-source", usage = "Markdown file(s) used as the source for document generation.", required = true)
    private List<File> markdownSources = newArrayList();

    @Option(name = "-workingDir", usage = "Working directory to which documents are generated to.", required = true)
    private File workingDir;

    @Option(name = "-properties", usage = "Properties used to customize the generation for individual documents.\n" +
            "Property name is a compound key, starting with the document file name without extension. eg. filePluginManual\n" +
            "'appendCiReference' : Specify if the document requires ci reference documentation append. eg. filePluginManual.appendCiReference=true.\n" +
            "'prefixes' : Comma separated string for which plugin namespaces get generated ci documentation.eg. filePluginManual.prefixes=file,udm.\n" +
            "             Omitted prefixes results in ci reference documentation for all types.\n" +
            " 'title' : Tile of the documents as it appears on the pdf cover eg. filePluginManual.title=File Manual\n" +
            " 'version' : Version of the plugin.\n\n" +
            " You can supply any other property you may need to replace during generation. You can use {{prop}} as the placeholder in your markdowon.")
    private File propertiesSource;

    @Option(name = "-wkhtmltopdf", usage = "Command to execute wkhtmltopdf. Default is 'wkhtmltopdf'")
    private String wkHtmlToPdfCommand = "wkhtmltopdf";

    private PegDownProcessor markdownProcessor = new PegDownProcessor();
    private ContextProperties properties;


    public static void main(String[] args) {
        DocumentGenerator documentGenerator = new DocumentGenerator();
        CmdLineParser parser = new CmdLineParser(documentGenerator);
        try {
            parser.parseArgument(args);
            documentGenerator.generate();
        } catch (CmdLineException e) {
            System.err.println(e.getMessage());
            System.err.println("java DocumentGenerator [options...] arguments...");
            parser.printUsage(System.err);
            System.err.println();
            System.exit(1);
        }
    }

    private void generate() {
        downloadResourcesToWorkingDir();
        List<File> htmlFiles = generateHtmlFromMarkdownSources();
        if (generatePdf) {
            generatePdfFromHtml(htmlFiles);
        }
    }

    private void downloadResourcesToWorkingDir() {
        for (URL resource : resources) {
            String filename = extractFileName(resource);
            String ext = extractFileNameExtension(filename);
            logger.info("Processing resource '{}'. Resource has a file name of '{}' with extension '{}'.", new Object[] {resource, filename, ext});
            if ("zip".equals(ext)) {
                copyUncompressed(resource, filename);
            } else if ("".equals(ext)) {
                copyLocalFolder(resource, filename);
            } else {
                copy(resource, new File(workingDir, filename));
            }
        }
    }

    private void copyUncompressed(URL resource, String filename) {
        File archive = new File(workingDir, filename);
        copy(resource, archive);
        explodeArchive(archive, workingDir);
        archive.delete();
    }


    private void copyLocalFolder(URL resource, String folder) {
        File sourceFolder = null;
        try {
            sourceFolder = new File(resource.toURI());
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
        copyDirectory(sourceFolder, new File(workingDir, folder));
    }


    private List<File> generateHtmlFromMarkdownSources() {
        List<File> htmlFiles = newArrayList();
        if (generateSingleDocument) {
            htmlFiles.add(generateSinglePageHtml());
        } else {
            htmlFiles = generateMultiplePageHtml();
        }
        return htmlFiles;
    }

    private List<File> generatePdfFromHtml(List<File> htmlFiles) {
        logger.info("Creating pdf from html sources '{}'.", htmlFiles);
        WkhtmlPdfGenerator generator = new WkhtmlPdfGenerator();
        generator.setCoverHtml(pdfCoverHtml);
        generator.setFooterHtml(pdfFooterHtml);
        generator.setHeaderHtml(pdfHeaderHtml);
        generator.setProperties(getProperties());
        generator.setTocXsl(pdfTocXsl);
        generator.setWkHtmlToPdfCommand(wkHtmlToPdfCommand);
        generator.setWorkingDir(workingDir);

        List<File> pdfFiles = newArrayList();
        for (File htmlFile : htmlFiles) {
            pdfFiles.add(generator.generatePdfFromHtml(htmlFile));
        }

        generator.clearTemporaryFiles();
        return pdfFiles;
    }

    private File generateSinglePageHtml() {
        File singleDocument = new File(workingDir, "singleDocument.html");
        String config = "singleDocument";
        logger.info("Creating single html document '{}' from markdown sources '{}'.", singleDocument.getAbsolutePath(), markdownSources);
        PrintWriter out = null;
        try {
            out = new PrintWriter(new FileWriter(singleDocument));
            openHtmlTagAndInitializeHeadTag(out, config);
            openBodyTag(out);
            for (File markdownSource : markdownSources) {
                generateHtmlBodyFromMarkdownSource(markdownSource, out, config);
            }
            finalizeBodyAndCloseHtmlTag(out, "singleDocument");
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } finally {
            Closeables.closeQuietly(out);
        }
        return singleDocument;
    }

    private List<File> generateMultiplePageHtml() {
        logger.info("Creating multiple html documents from markdown sources '{}'.", markdownSources);
        List<File> generatedFiles = newArrayList();
        for (File markdownSource : markdownSources) {
            String config = extractFileNameWithoutExtension(markdownSource);
            File htmlFile = createFileNameWithNewExtension(markdownSource, workingDir, "html");
            logger.info("Creating html document '{}' from markdown source '{}'.", htmlFile.getAbsolutePath(), markdownSource.getAbsolutePath());
            PrintWriter out = null;
            try {
                out = new PrintWriter(new FileWriter(htmlFile));
                openHtmlTagAndInitializeHeadTag(out, config);
                openBodyTag(out);
                generateHtmlBodyFromMarkdownSource(markdownSource, out, config);
                finalizeBodyAndCloseHtmlTag(out, config);
                generatedFiles.add(htmlFile);
            } catch (IOException e) {
                throw new RuntimeIOException(e);
            } finally {
                Closeables.closeQuietly(out);
            }
        }
        return generatedFiles;
    }

    private PrintWriter openHtmlTagAndInitializeHeadTag(PrintWriter out, String config) {
        ContextProperties ctx = getRootContextMergedWithConfig(config);
        out.print("<html><head>");
        if (htmlHeaderSnippet != null) {
            out.print(replacePlaceholders(getText(htmlHeaderSnippet), ctx));
        }
        initializeHead(out);
        out.print("</head>");
        return out;
    }

    protected PrintWriter initializeHead(PrintWriter out) {
        return out;
    }


    private void openBodyTag(PrintWriter out) {
        out.print("<body>");
    }

    private void finalizeBodyAndCloseHtmlTag(PrintWriter out, String config) {
        finalizeBody(out, config);
        if (htmlFooterSnippet != null) {
            out.print(getText(htmlFooterSnippet));
        }
        out.print("</body></html>");
    }

    protected void finalizeBody(PrintWriter out, String config) {
        ContextProperties ctx = getRootContextMergedWithConfig(config);
        if ("true".equals(ctx.get("appendCiReference"))) {
            List<String> prefixes = Collections.emptyList();
            if (ctx.get("prefixes") != null) {
                Iterable<String> split = Splitter.on(',').trimResults().split(ctx.get("prefixes"));
                prefixes = newArrayList(split);
            }
            CiReferenceHtmlGenerator generator = new CiReferenceHtmlGenerator(prefixes, out);
            generator.generate();
        }
    }

    private void generateHtmlBodyFromMarkdownSource(File markdownSource, PrintWriter out, String config) {
        String markdownText = getText(markdownSource);
        ContextProperties ctx = getRootContextMergedWithConfig(config);
        markdownText = replacePlaceholders(markdownText, ctx);
        String html = markdownProcessor.markdownToHtml(markdownText);
        logger.trace("Generate html from markdown file '{}' :\n {}", markdownSource.getAbsolutePath(), html);
        out.write(html);
    }

    private ContextProperties getRootContextMergedWithConfig(String config) {
        ContextProperties ctx = getProperties().getRootContext();
        ctx.putAll(getProperties().getContext(config));
        return ctx;
    }

    private ContextProperties getProperties() {
        if (properties == null) {
            properties = new ContextProperties();
            if (propertiesSource != null) {
                try {
                    Properties temp = new Properties();
                    temp.load(new FileInputStream(propertiesSource));
                    for (String key : temp.stringPropertyNames()) {
                        properties.put(key, temp.getProperty(key));
                    }
                } catch (IOException e) {
                    throw new RuntimeIOException(e);
                }
            }
        }
        return properties;
    }


    private static final Logger logger = LoggerFactory.getLogger(DocumentGenerator.class);
}
