/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api.execution;

import com.xebialabs.xlplatform.documentation.PublicApiRef;

/**
 *  All possible values for the execution state of a task.
 */
@PublicApiRef
public enum TaskExecutionState {
    UNREGISTERED,
    PENDING,
    SCHEDULED,
    QUEUED,
    EXECUTING,
    FAILING,
    FAILED,
    STOPPING,
    STOPPED,
    ABORTING,
    ABORTED,
    EXECUTED,
    DONE,
    CANCELLED,
    CANCELLING,
    FORCE_CANCELLED;

    /**
     * If the task is in a final state, it will never change again to another state.
     *
     * @return true if task is in a final state.
     */
    public boolean isFinal() {
        return this == DONE || this == CANCELLED || this == FORCE_CANCELLED;
    }

    /**
     * If the task is in an active state, the task can go into a different state by itself.
     *
     * @return true if task is in active state.
     */
    public boolean isActive() {
        return this == UNREGISTERED || this == SCHEDULED || this == QUEUED || this.isExecutingSteps();
    }

    /**
     * If the task is in a passive state, the task will not go into a different state by itself.
     *
     * @return true if task is in active state.
     */
    public boolean isPassive() {
        return this == PENDING || isPassiveAfterExecuting();
    }

    /**
     * Determines if the task is in a state where it is executing steps. This is a subset of {@link #isActive()}
     *
     * @return true if task is executing steps
     */
    public boolean isExecutingSteps() {
        return this == EXECUTING || this == FAILING || this == STOPPING || this == ABORTING || this == CANCELLING;
    }

    /**
     * Determines if the task is passive but it has been executing steps.
     *
     * @return true if task is passive after it has been executing steps
     */
    public boolean isPassiveAfterExecuting() {
        return this == EXECUTED || this.isExecutionHalted() || this.isFinal();
    }

    /**
     * Determines if the task has been executing but is halted and can be continued.
     *
     * @return true if task is halted
     */
    public boolean isExecutionHalted() {
        return this == FAILED || this == STOPPED || this == ABORTED;
    }

    /**
     * Determines if the task is never been executing.
     *
     * @return true if the task is never been executing
     */
    public boolean isNotBeenExecuting() {
        return this == UNREGISTERED || this == PENDING || this == SCHEDULED || this == QUEUED;
    }


}
